package com.ibm.oti.util.semaphore;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2005  All Rights Reserved
 */

import java.util.Hashtable;

/**
 * Base class used to access cross-vm mutex functionality
 */
abstract class Semaphore {

	/**
	 * create and initialize the mutex
	 *
	 * @param		name the name of the mutex
	 * @param 		count the number of threads that can concurrently access the protected region
	 */
	private static native long init(String name,int count);

	/**
	 * close and free the semaphore resources
	 *
	 * @param		handle to the semaphore
	 */
	private static native void closeSemaphore(long handle);

	/**
	 * try to wait on the semaphore
	 *
	 * @param		handle to the semaphore
	 * @return		true if the operation was successful
	 */
	private static native boolean waitOnSemaphore(long handle);

	/**
	 * try to post the semaphore
	 *
	 * @param		handle to the semaphore
	 * @return		true if the operation was successful
	 */
	private static native boolean postOnSemaphore(long handle);

	private static Hashtable existingSemaphores = new Hashtable();
	private String name = null;
	private static long semaphoreHandle = 0;

	/**
	 * Constructor.  This should only be called by subclasses in the same package.  The
	 * name of the semaphore should be unique for its use.  To try and catch the case where
	 * we accidentally use the same semaphore names for different purposes we only allow
	 * a single instance to be created with a given name, this one instance should then
	 * be used whenever that semaphore needs to be accessed.
	 *
	 * @param  name the name of the semaphore
	 *
	 * @throws IllegalArgumentException thrown if a semaphore with this name has allready been created
	 * @throws RuntimeException if the semaphore cannot be created at the native level
	 */
	Semaphore(String name, int count) {
		initializeSemaphore(name,count);
	}

	/**
	 * Default constructor, must be followed by initialize call
	 * for the semaphore to be in a usable state
	 */
	Semaphore(){
	}

	/**
	 * This is called to initialize the semaphore.  We do this in this method
	 * instead of the constructor so that
	 * the required security check can be done in subclasses
	 *
	 * @param  name the name of the semaphore
	 *
	 * @throws IllegalArgumentException thrown if a semaphore with this name has allready been created
	 * @throws RuntimeException if the semaphore cannot be created at the native level
	 */
	protected void initializeSemaphore(String name, int count){
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()){
			// K037b = Must be in bootstrap class path to call this method
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K037b"));
		}

		synchronized(existingSemaphores) {
			if (existingSemaphores.containsKey(name)){
				// K037c = Semaphore allready exists
				throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K037c"));
			} else {
				existingSemaphores.put(name, name);
				semaphoreHandle = init(name,count);
				if (semaphoreHandle == 0){
					// K0353 = Failed to Create Semaphore
					throw new RuntimeException(com.ibm.oti.util.Msg.getString("K0353"));
				}
			}
			this.name = name;
		}
	}

	/**
	 * This is the method called to wait on the semaphore
	 *
	 * @return true if the wait succeeded, false otherwise
	 *
	 * @throws IllegalStateException if the semaphore has allready been closed
	 */
	public boolean waitSemaphore(){
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()){
			// K037b = Must be in bootstrap class path to call this method
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K037b"));
		}

		if (semaphoreHandle != 0){
			return waitOnSemaphore(semaphoreHandle);
		} else {
			// K0352 = Semaphore is not properly initialized or has been closed
			throw new IllegalStateException(com.ibm.oti.util.Msg.getString("K0352"));
		}
	}

	/**
	 * This is the method called to post the semaphore
	 *
	 * @return true if the semaphore was posted, false otherwise
	 *
	 * @throws IllegalStateException if the semaphore has allready been closed
	 */
	public boolean postSemaphore(){
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()){
			// K037b = Must be in bootstrap class path to call this method
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K037b"));
		}

		if (semaphoreHandle != 0){
			return postOnSemaphore(semaphoreHandle);
		} else {
			// K0352 = Semaphore is not properly initialized or has been closed
			throw new IllegalStateException(com.ibm.oti.util.Msg.getString("K0352"));
		}
	}

	/**
	 * This is the method called to close the semaphore and free associated
	 * resources
	 *
	 * @throws IllegalStateException if the semaphore has allready been closed
	 */
	public void close(){
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()){
			// K037b = Must be in bootstrap class path to call this method
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K037b"));
		}

		synchronized(existingSemaphores) {
			if (semaphoreHandle != 0){
				if (existingSemaphores.containsKey(name)){
					existingSemaphores.remove(name);
				}

				closeSemaphore(semaphoreHandle);
				semaphoreHandle=0;
			} else {
				// K0352 = Semaphore is not properly initialized or has been closed
				throw new IllegalStateException(com.ibm.oti.util.Msg.getString("K0352"));
			}
		}
	}

	/**
	 * Finalizer that frees the os resources (semaphore handle) when the object
	 * is colleted.  In most cases once constructed the object will remain until
	 * the vm terminates but this is to handle when this is not the case
	 */
	protected void finalize() throws Throwable {
		close();
	}

}
