
package com.ibm.oti.vm;

import java.util.Vector;
import java.io.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Represents the running virtual machine. All VM specific API
 * are implemented on this class.
 * <p>
 * Note that all methods in VM are static. There is no singleton
 * instance which represents the actively running VM.
 */
public final class VM {

	private static int[] types;
	private static Object[] cache;
	private static boolean[] isBoot;
	private static boolean allowExit = true;
	private static boolean closeSockets = false;
	private static Vector shutdownClasses = new Vector(4);
	private static final Class javaLangClass;

	static {
		Class jlc = null;
		try {
			jlc = Class.forName("java.lang.Class");
		} catch (ClassNotFoundException e) {}
		javaLangClass = jlc;
	}

/**
 * Prevents this class from being instantiated.
 */
private VM() {
}

/**
 * Answer the class at depth.
 *
 * Notes:
 * 	 1) This method operates on the defining classes of methods on stack.
 *		NOT the classes of receivers.
 *
 *	 2) The item at index zero describes the caller of this method.
 *
 */
private static final native Class getStackClass(int depth);

/**
 * Return whether or not the caller of the caller of this method
 * is a bootstrap class.
 */
public static final boolean callerIsBootstrap() {
	Class caller = getStackClass(2);
	if (caller == javaLangClass) {
		// caller is java.lang.Class.newInstanceImpl()
		// get the caller of java.lang.Class.newInstance()
		caller = getStackClass(4);
	}
	return isBootstrapClass(caller);
}

/**
 * Return whether or not the Class is a bootstrap class.
 *
 * @param 		cl ClassLoader
 *					the ClassLoader instance to test
 */
private static final native boolean isBootstrapClass(Class cl);

/**
 * Native used to dump a string to the system console for debugging.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		str String
 *					the String to display
 */
public static native void dumpString(String str);

/*
 * Answers a Class object which represents the class
 * named by the argument. The class is looked up in the
 * Application class path.
 * @param		className	The name of the non-base type class to find
 * @return		the named Class
 * @throws		ClassNotFoundException If the class could not be found
 */
public static Class findAppClass(String className) throws ClassNotFoundException {
 	return Class.forName(className);
}

/**
 * Returns the index of the first user entry
 * on the class path.
 */
private static native int getBootClassPathCount();

/**
 * Does internal initializaion required by VM.
 *
 * @author		OTI
 * @version		initial
 */
public static void initializeVM() {
	int count = getClassPathCount();
	types = new int[count];
	isBoot = new boolean[count];
	int bootCount = getBootClassPathCount();
	for (int i=0; i<bootCount; i++) isBoot[i] = true;
	cache = new Object[count];
}

public static void closeSockets() {
	closeSockets = true;
}

public static void addShutdownClass(Runnable cl) {
	if (!callerIsBootstrap())
		throw new SecurityException();
	shutdownClasses.addElement(cl);
}

/**
 * Internal use only. Shutdown the JCL. Do not create new Threads.
 * If shutting down without calling Runtime.exit(),
 * daemon threads will still be running. If shutting down from a call
 * to System.exit(), JNI DestroyJavaVM(),
 * all threads will still be running.
 *
 * @see #deleteOnExit()
 * @see #closeJars()
 */
private static void shutdown() {
	for (int i=shutdownClasses.size() - 1; i>=0; i--)
		((Runnable)shutdownClasses.elementAt(i)).run();
	if (closeSockets)
		com.ibm.oti.connection.socket.Socket.closeOpenSockets();
}

/**
 * Internal use only.
 * Run the shutdown hooks immediately
 *
 * @author		OTI
 * @version		initial
 */
private static void cleanup() {
}

/**
 * Reads the resource from the zip file.
 *
 * @param		path	the file name of the zip file
 * @param		resName the zip entry
 * @return		a byte array containing the zip entry contents, or
 *				null if the zip file or zip entry do not exist
 */
private static native byte[] readZip(byte[] path, String resName);

/**
 * Reads the entry from the zip file.
 *
 * @param		path	the file name of the zip file
 * @param		entryName the zip entry
 * @return		a byte array containing the zip entry contents, or
 *				null if the zip file or zip entry do not exist
 */
public static byte[] fileFromZip(String path, String entryName) {
	if (!callerIsBootstrap())
		throw new SecurityException();
	return readZip(path.getBytes(), entryName);
}

/**
 * Reads the manifest from the zip file.
 *
 * @param		path	the file name of the zip file
 * @return		a byte array containing the manifest contents, or
 *				null if the zip file or manifest do not exist
 */
static native byte[] readManifest(byte[] path);

/**
 * Reads the manifest from the zip file.
 *
 * @param		path	the file name of the zip file
 * @return		a byte array containing the manifest contents, or
 *				null if the zip file or manifest do not exist
 */
public static byte[] manifestFromZip(String path) {
	if (!callerIsBootstrap())
		throw new SecurityException();
	return readManifest(path.getBytes());
}

static final int CPE_TYPE_UNKNOWN = 0;
static final int CPE_TYPE_DIRECTORY = 1;
static final int CPE_TYPE_JAR = 2;
static final int CPE_TYPE_TCP = 3;
static final int CPE_TYPE_JXE = 4;
static final int CPE_TYPE_UNUSABLE = 5;
static final int CPE_TYPE_PALMDB = 6;

/**
 * Return the type of the specified entry on the class path for a ClassLoader.
 * Valid tyes are:
 * 		CPE_TYPE_UNKNOWN
 *		CPE_TYPE_DIRECTORY
 * 		CPE_TYPE_JAR
 *		CPE_TYPE_TCP - this is obsolete
 *		CPE_TYPE_UNUSABLE
 *
 * @param classLoader the ClassLoader
 * @param cpIndex the index on the class path
 *
 * @return a int which specifies the class path entry type
 */
static final native int getClassPathEntryType(Object classLoader, int cpIndex);

static final native long getJxePointerFromClassPath(Object classLoader, int cpIndex);

/**
 * Reads the resource from the zip file.
 *
 * @param		cpIndex	the classpath index of the zip file
 * @param		resName the zip entry
 * @return		a byte array containing the zip entry contents, or
 *				null if the zip file or zip entry do not exist
 *
 * @author		OTI
 * @version		initial
 */
static native byte[] readZipFromClassPath(int cpIndex, String resName);

/**
 * Reads the manifest from the zip file.
 *
 * @param		cpIndex	the classpath index of the zip file
 * @return		a byte array containing the manifest contents, or
 *				null if the zip file or manifest do not exist
 *
 * @author		OTI
 * @version		initial
 */
private static native byte[] readManifestFromClassPath(int cpIndex);

/**
 * Maps the class path entry to a Jxe.
 */
static Jxe getJxeFromClassPath(Object classLoader, int cpIndex) {
	long pointer = getJxePointerFromClassPath(classLoader, cpIndex);
	if (pointer == 0) return null;
	Jxe jxe = new Jxe(pointer);
	return jxe;
}

private static void fillCache(int i) {
	types[i] = getClassPathEntryType(null, i);
	switch (types[i]) {
		case VM.CPE_TYPE_UNKNOWN:
			types[i] = VM.CPE_TYPE_UNUSABLE;
			break;
		case VM.CPE_TYPE_DIRECTORY:
			byte[] bytes = VM.getPathFromClassPath(i);
			cache[i] = bytes;
			return;
		//case VM.CPE_TYPE_JAR:
		//case VM.CPE_TYPE_TCP:
		case VM.CPE_TYPE_JXE:
			Jxe jxe = VM.getJxeFromClassPath(null, i);
			if (jxe != null) {
				cache[i] = jxe;
				return;
			}
			types[i] = VM.CPE_TYPE_UNUSABLE;
			break;
		//case VM.CPE_TYPE_UNUSABLE:
	}
}

public static java.io.InputStream getResourceAsStream(String resName, boolean restrict) {
	byte[] bytes;
	if (resName.length() >= 6) {
		String sub = resName.substring(resName.length() - 6);
		if (sub.toLowerCase().equals(".class")) return null;
	}

	boolean bootClass = isBootstrapClass(getStackClass(2));
	for (int i=0; i<cache.length; i++) {
		if (isBoot[i] != bootClass && (!bootClass || restrict))
			continue;
		if (types[i] == VM.CPE_TYPE_UNKNOWN) fillCache(i);
		switch (types[i]) {
			case VM.CPE_TYPE_DIRECTORY:
				byte[] path = (byte[])cache[i];
				byte[] resBytes = resName.getBytes();
				byte[] name = new byte[path.length + resBytes.length];
				System.arraycopy(path, 0, name, 0, path.length);
				System.arraycopy(resBytes, 0, name, path.length, resBytes.length);
				try {
					java.io.InputStream in = new com.ibm.oti.connection.file.FileInputStream(name);
					return in;
				} catch (IOException e) {
				}
				break;
			case VM.CPE_TYPE_JAR:
				bytes = readZipFromClassPath(i, resName);
				if (bytes != null) return new java.io.ByteArrayInputStream(bytes);
				break;
			case VM.CPE_TYPE_JXE:
				InputStream result = ((Jxe)cache[i]).getResourceAsStream(resName);
				if (result != null) return result;
				break;
    	}
	}
	return null;
}

public static java.io.InputStream getManifest() {
	if (!callerIsBootstrap())
		throw new SecurityException();

	byte[] bytes;
	String manifestName = "META-INF/MANIFEST.MF";
	for (int i=0; i<cache.length; i++) {
		if (isBoot[i]) continue;
		if (types[i] == VM.CPE_TYPE_UNKNOWN) fillCache(i);
		switch (types[i]) {
			case VM.CPE_TYPE_DIRECTORY:
				try {
					byte[] path = (byte[])cache[i];
					byte[] resBytes = manifestName.getBytes();
					byte[] name = new byte[path.length + resBytes.length];
					System.arraycopy(path, 0, name, 0, path.length);
					System.arraycopy(resBytes, 0, name, path.length, resBytes.length);
					java.io.InputStream in = new com.ibm.oti.connection.file.FileInputStream(name);
					return in;
				} catch (java.io.IOException e) {}
				break;
			case VM.CPE_TYPE_JAR:
				bytes = readManifestFromClassPath(i);
				if (bytes != null) return new java.io.ByteArrayInputStream(bytes);
				break;
			case VM.CPE_TYPE_JXE:
				InputStream result = ((Jxe)cache[i]).getResourceAsStream(manifestName);
				if (result != null) return result;
				break;
    	}
	}
	return null;
}

public static boolean allowExit() {
	return allowExit;
}

public static void stopExit() {
	allowExit = false;
}

public static void allowMidpExit() {
	if (!callerIsBootstrap())
		throw new SecurityException();
	allowExit = true;
}

/**
 * Return the number of entries on the bootclasspath.
 *
 * @return an int which is the number of entries on the bootclasspath
 */
static native int getClassPathCount();

/**
 * Return the specified bootclasspath entry. Directory entries
 * are terminated with a file separator character.
 *
 * @param index the index of the bootclasspath entry
 *
 * @return a byte array containing the bootclasspath entry
 * 			specified in the vm options
 */
static native byte[] getPathFromClassPath(int index);

/**
 * Loads and links the library specified by the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		libName		the name of the library to load
 *
 * @exception	SecurityException
 *							if the library was not allowed to be loaded
 */
public static synchronized void loadLibrary(String libName) throws java.io.IOException {
	if (!callerIsBootstrap())
		throw new SecurityException();
	loadLibrary(
		libName,
		System.getProperty("com.ibm.oti.vm.bootstrap.library.path")
	);
}

/**
 * Loads and links the library specified by the argument.
 * No security check is done.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		libName			the name of the library to load
 * @param		libraryPath		the library path to search, or NULL'
 *
 * @exception	UnsatisfiedLinkError
 *							if the library could not be loaded
 */
private static void loadLibrary(String libName, String libraryPath) throws java.io.IOException {
	byte[] message = loadLibraryWithPath(com.ibm.oti.util.Util.getBytes(libName), libraryPath == null ? null : com.ibm.oti.util.Util.getBytes(libraryPath));
	if (message != null) {
		String error;
		try {
			error = com.ibm.oti.util.Util.convertFromUTF8(message, 0, message.length);
		} catch (java.io.UTFDataFormatException e) {
			error = com.ibm.oti.util.Util.toString(message);
		}
		throw new java.io.IOException(libName + " (" + error + ")");
	}
}

private static native byte[] loadLibraryWithPath(byte[] libName, byte[] libraryPath);

/**
 * This method will cause the GC to do a local collection.
 */
public static native void localGC();

/**
 * This method will cause the GC to do a global collection.
 */
public static native void globalGC();

/**
 * Provides a hint to the virtual machine that it would
 * be useful to attempt to perform any outstanding
 * object finalizations.
 */
public static native void runFinalization();

/**
 * Answer if native implementations should be used.
 */
public final static boolean useNatives() {
	return useNativesImpl();
}

private final static native boolean useNativesImpl();

/**
 * The default list of restricted packages.
 *
 * @see #addRestrictedPackages(String[])
 * @see #checkRestrictedPackage(Class)
 */
private static String[] restrictedPackages = new String[] {
	"java.",
	"javax.microedition.",
	"com.ibm.oti.",
	"com.ibm.j9.",
	"com.ibm.ive.",
	};

/**
 * Called from class verification to check if the specified class, loaded by the application
 * class loader (i.e. on the classpath), is in a restricted package. If the class is in a
 * restricted package a SecurityException is thrown, causing the class verification to fail.
 *
 * @param checkClass the Class to check
 *
 * @throws SecurityException when the class is in a restricted package
 *
 * @see #addRestrictedPackages(String[])
 */
public static void checkRestrictedPackage(Class checkClass) {
	if (!isBootstrapClass(checkClass)) {
		String className = checkClass.getName();
		for (int i=0; i<restrictedPackages.length; i++) {
			if (className.startsWith(restrictedPackages[i])) {
				// K01d2 = {1} - protected system package '{0}'
				throw new SecurityException(com.ibm.oti.util.Msg.getString("K01d2", restrictedPackages[i], className));
			}
		}
	}
}

/**
 * Adds package names to the list of resticted packages. A SecurityException is thrown
 * when one of the restricted package names is a prefix of the name of a class being
 * loaded. Normally the package names should end with '.' to avoid matching word prefixes.
 * For example, using "com.j9" as a restricted package name restricts "com.j9vm.*",
 * while restricting "com.j9." does not.
 *
 * @param packageNames an array of package names to restrict
 *
 * @see #checkRestrictedPackage(Class)
 */
public static synchronized void addRestrictedPackages(String[] packageNames) {
	if (!callerIsBootstrap())
		throw new SecurityException();
	int oldLength = restrictedPackages.length;
	String[] newList = new String[oldLength + packageNames.length];
	System.arraycopy(restrictedPackages, 0, newList, 0, oldLength);
	for (int i=0; i<packageNames.length; i++)
		newList[oldLength + i] = packageNames[i];
	restrictedPackages = newList;
}

/**
 * Enable the JIT and return true only if the vm has been started with
 * the JIT explicitly disabled using -Xjit:noJitUntilEnabled, and enableJIT()
 * has not been called previously.
 */
public static native boolean enableJIT();

/**
 * Sets the value of a particular system property.
 *
 * @param		prop		the system property to change
 * @param		value		the value to associate with prop
 * @return		the old value of the property, or null
 */
public static String setSystemProperty(String prop, String value) {
	if (!callerIsBootstrap())
		throw new SecurityException();
	return setProperty(prop, value);
}

private static native String setProperty(String prop, String value);

/**
 * Sets the Thread's isDaemon flag.
 * This can only be done before the Thread starts running.
 *
 * @param isDaemon true to set the Thread as a Daemon Thread or false otherwise.
 * @throws IllegalThreadStateException if the Thread has already been started
 */
public static void setDaemonThread(Thread thread) {
	if (!callerIsBootstrap())
		throw new SecurityException();
	if (!setDaemonThreadImpl(thread))
		throw new IllegalThreadStateException();
}

private static native boolean setDaemonThreadImpl(Thread thread);

/**
 * This method will enable finalization on aClass.
 * Do not call this on a class that does not implement finalize()V.
 */
public static void enableFinalization(Class aClass) {
	if (!callerIsBootstrap())
		throw new SecurityException();
	enableFinalizationImpl(aClass);
}

private final static native void enableFinalizationImpl(Class aClass);

/*
 * Returns NULL, or an array of 2 String objects:
 *    array[0] = proxy host string
 *    array[1] = proxy port string
 */
public static String[] getHttpProxyParms() {
	return getHttpProxyImpl();
}
private static native String[] getHttpProxyImpl();
}
