package java.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

/**
 * DataInputStream is a filter class which can read typed data
 * from a Stream.  Typically, this stream has been written by
 * a DataOutputStream.  Types that can be read include
 * byte, 16-bit short, 32-bit int, 32-bit float, 64-bit long,
 * 64-bit double, byte strings, and UTF Strings.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			DataOutputStream
 */
public class DataInputStream extends InputStream implements DataInput
{
	/**
	 * The target InputStream which is being filtered.
	 */
	protected InputStream in;
/**
 * Constructs a new DataInputStream on the InputStream <code>in</code>.  All
 * reads can now be filtered through this stream.  Note that data read by this
 * Stream is not in a human readable format and was most likely created by
 * a DataOutputStream.
 *
 * @param		in			the target InputStream to filter reads on.
 *
 * @see		DataOutputStream
 * @see		RandomAccessFile
 */
public DataInputStream(InputStream in) {
	this.in = in;
}

/**
 * Reads bytes from the source stream into the byte array <code>buffer</code>.
 * The number of bytes actually read is returned.
 *
 * @param		buffer		the buffer to read bytes into
 * @return 		the number of bytes actually read or -1 if end of stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#write(byte[])
 * @see 		DataOutput#write(byte[], int, int)
 */
public final int read(byte[] buffer) throws IOException {
	return in.read(buffer, 0, buffer.length);
}

/**
 * Read at most <code>length</code> bytes from this DataInputStream and stores them in byte
 * array <code>buffer</code> starting at <code>offset</code>. Answer the number of bytes
 * actually read or -1 if no bytes were read and end of stream was encountered.
 *
 * @param		buffer		the byte array in which to store the read bytes.
 * @param		offset		the offset in <code>buffer</code> to store the read bytes.
 * @param		length		the maximum number of bytes to store in <code>buffer</code>.
 * @return 		the number of bytes actually read or -1 if end of stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#write(byte[])
 * @see 		DataOutput#write(byte[], int, int)
 */
public final int read(byte[] buffer, int offset, int length) throws IOException {
	return in.read(buffer, offset, length);
}

/**
 * Reads a boolean from this stream.
 *
 * @return		the next boolean value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeBoolean(boolean)
 */
public final boolean readBoolean() throws IOException {
	int temp = in.read();
	if (temp >= 0) return temp != 0;
	throw new EOFException();
}

/**
 * Reads an 8-bit byte value from this stream.
 *
 * @return		the next byte value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeByte(int)
 */
public final byte readByte() throws IOException {
	int temp = in.read();
	if (temp >= 0) return (byte) temp;
	throw new EOFException();
}

/**
 * Reads a 16-bit character value from this stream.
 *
 * @return		the next <code>char</code> value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeChar(int)
 */
public final char readChar() throws IOException {
	int b1 = in.read();
	int b2 = in.read();
	if ((b1 | b2) >= 0)
		return (char) ((b1 << 8) + b2);
	throw new EOFException();
}

/**
 * Reads a 64-bit <code>double</code> value from this stream.
 *
 * @return		the next <code>double</code> value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeDouble(double)
 */
public final double readDouble() throws IOException {
	return Double.longBitsToDouble(readLong());
}

/**
 * Reads a 32-bit <code>float</code> value from this stream.
 *
 * @return		the next <code>float</code> value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeFloat(float)
 */
public final float readFloat() throws IOException {
	return Float.intBitsToFloat(readInt());
}

/**
 * Reads bytes from this stream into the byte array <code>buffer</code>.
 * This method will block until <code>buffer.length</code> number of bytes have
 * been read.
 *
 * @param		buffer to read bytes into
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#write(byte[])
 * @see 		DataOutput#write(byte[], int, int)
 */
public final void readFully(byte[] buffer) throws IOException {
	readFully(buffer, 0, buffer.length);
}

/**
 * Read bytes from this stream and stores them in byte array <code>buffer</code> starting
 * at offset <code>offset</code>.  This method blocks until <code>length</code> number of
 * bytes have been read.
 *
 * @param		buffer	the byte array in which to store the read bytes.
 * @param		offset	the offset in <code>buffer</code> to store the read bytes.
 * @param		length	the maximum number of bytes to store in <code>buffer</code>.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#write(byte[])
 * @see 		DataOutput#write(byte[], int, int)
 */
public final void readFully(byte[] buffer, int offset, int length) throws IOException {
	if (buffer != null) {
		// avoid int overflow
		if (0 <= offset && offset <= buffer.length && 0 <= length && length <= buffer.length - offset) {
			while (length > 0) {
				int result = in.read(buffer, offset, length);
				if (result >= 0) {
					offset += result;
					length -= result;
				} else throw new EOFException();
			}
		} else throw new IndexOutOfBoundsException();
	// K0047 = buffer is null
	} else throw new NullPointerException(com.ibm.oti.util.Msg.getString("K0047"));
}

/**
 * Reads a 32-bit integer value from this stream.
 *
 * @return		the next <code>int</code> value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeInt(int)
 */
public final int readInt() throws IOException {
	int b1 = in.read();
	int b2 = in.read();
	int b3 = in.read();
	int b4 = in.read();
	if ((b1 | b2 | b3 | b4) >= 0)
		return ((b1 << 24) + (b2 << 16) + (b3 << 8) + b4);
	throw new EOFException();
}

/**
 * Reads a 64-bit <code>long</code> value from this stream.
 *
 * @return		the next <code>long</code> value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeLong(long)
 */
public final long readLong() throws IOException {
	int i1 = readInt();
	int b1 = in.read();
	int b2 = in.read();
	int b3 = in.read();
	int b4 = in.read();
	if ((b1 | b2 | b3 | b4) >= 0)
		return (((long) i1) << 32) + ((long)b1 << 24) + (b2 << 16) + (b3 << 8) + b4;
	throw new EOFException();
}

/**
 * Reads a 16-bit <code>short</code> value from this stream.
 *
 * @return		the next <code>short</code> value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeShort(int)
 */
public final short readShort() throws IOException {
	int b1 = in.read();
	int b2 = in.read();
	if ((b1 | b2) >= 0)
		return (short) ((b1 << 8) + b2);
	throw new EOFException();
}

/**
 * Reads an unsigned 8-bit <code>byte</code> value from this stream and returns it
 * as an int.
 *
 * @return		the next unsigned byte value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeByte(int)
 */
public final int readUnsignedByte() throws IOException {
	int temp = in.read();
	if (temp >= 0) return temp;
	throw new EOFException();
}

/**
 * Reads a 16-bit unsigned <code>short</code> value from this stream and returns
 * it as an int.
 *
 * @return		the next unsigned <code>short</code> value from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeShort(int)
 */
public final int readUnsignedShort() throws IOException {
	int b1 = in.read();
	int b2 = in.read();
	if ((b1 | b2) >= 0)
		return ((b1 << 8) + b2);
	throw new EOFException();
}

/**
 * Reads a UTF format String from this Stream.
 *
 * @return		the next UTF String from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeUTF(java.lang.String)
 */
public final String readUTF() throws IOException {
	int utfSize = readUnsignedShort();
	return decodeUTF(utfSize);
}

static final int MAX_BUF_SIZE = 8192;
private static final boolean useNative = com.ibm.oti.vm.VM.useNatives();

String decodeUTF(int utfSize) throws IOException {
	byte[] buf = new byte[utfSize];
	char[] out = null;
	if (!useNative)
		out = new char[utfSize];

	readFully(buf, 0, utfSize);
	String result;
	if (useNative)
		result = com.ibm.oti.util.Util.convertFromUTF8(buf, 0, utfSize);
	else
		result = com.ibm.oti.util.Util.convertUTF8WithBuf(buf, out, 0, utfSize);
	return result;
}

/**
 * Reads a UTF format String from the DataInput Stream <code>in</code>.
 *
 * @param		in			the input stream to read from
 * @return		the next UTF String from the source stream.
 *
 * @throws 		IOException		If a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeUTF(java.lang.String)
 */
public static final String readUTF(DataInput in) throws IOException {
	return in.readUTF();
}

/**
 * Skips <code>count</code> number of bytes in this stream.  Subsequent
 * <code>read()</code>'s will not return these bytes unless <code>reset()</code>
 * is used.
 *
 * @param 		count		the number of bytes to skip.
 * @return		the number of bytes actually skipped.
 *
 * @throws 		IOException 	If the stream is already closed or another IOException occurs.
 */
public final int skipBytes(int count) throws IOException {
	int skipped = 0;
	long skip;
	while (skipped < count && (skip = in.skip(count - skipped)) != 0)
		skipped += skip;
	if (skipped >= 0) return skipped;
	throw new EOFException();
}

/**
 * Answers a int representing the number of bytes that are available
 * before this FilterInputStream will block.  This method returns the
 * number of bytes available in the target stream.
 *
 * @return 		the number of bytes available before blocking.
 *
 * @throws 		IOException		If an error occurs in this stream.
 */
public int available() throws IOException {
	return in.available();
}

/**
 * Close this FilterInputStream. This implementation closes the target stream.
 *
 * @throws 		IOException		If an error occurs attempting to close this stream.
 */
public void close() throws IOException {
	in.close();
}

/**
 * Set a Mark position in this FilterInputStream.  The parameter <code>readLimit</code> indicates how many
 * bytes can be read before a mark is invalidated.  Sending reset() will reposition the
 * Stream back to the marked position provided <code>readLimit</code> has not been surpassed.
 *<p>
 * This implementation sets a mark in the target stream.
 *
 * @param		readlimit	the number of bytes to be able to read before invalidating the mark.
 *
 * @throws 		IOException		If an error occurs attempting mark this stream.
 */
public synchronized void mark(int readlimit) {
	in.mark(readlimit);
}

/**
 * Answers a boolean indicating whether or not this FilterInputStream supports mark() and
 * reset().  This implementation answers whether or not the target stream supports
 * marking.
 *
 * @return 		<code>true</code> if mark() and reset() are supported, <code>false</code> otherwise.
 */
public boolean markSupported() {
	return in.markSupported();
}

/**
 * Reads a single byte from this FilterInputStream and returns the result as
 * an int.  The low-order byte is returned or -1 of the end of stream was
 * encountered.  This implementation returns a byte from the target stream.
 *
 * @return 		the byte read or -1 if end of stream.
 *
 * @throws 		IOException 	If the stream is already closed or another IOException occurs.
 */
public int read() throws IOException {
	return in.read();
}

/**
 * Reset this FilterInputStream to the last marked location.  If the <code>readlimit</code> has
 * been passed or no <code>mark</code> has been set, throw IOException.  This implementation
 * resets the target stream.
 *
 * @throws 		IOException		If the stream is already closed or another IOException occurs.
 */
public synchronized void reset() throws IOException {
	in.reset();
}

/**
 * Skips <code>count</code> number of bytes in this InputStream.  Subsequent
 * <code>read()</code>'s will not return these bytes unless <code>reset()</code>
 * is used. This implementation skips <code>count</code> number of bytes in the
 * target stream.
 *
 * @param 		count		the number of bytes to skip.
 * @return		the number of bytes actually skipped.
 *
 * @throws 		IOException 	If the stream is already closed or another IOException occurs.
 */
public long skip(long count) throws IOException {
	return in.skip(count);
}
}
