package java.io;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2004  All Rights Reserved
 */

/**
 * PrintStream is a class which takes an OutputStream and provides convenience methods
 * for printing common data types in a human readable format on the stream.  This is not
 * to be confused with DataOutputStream which is used for encoding common datatypes so
 * that they can be read back in.  No IOExceptions are thrown by this class.  Instead,
 * callers should call checkError() to see if a problem has been encountered in this Stream.
 *
 * @author		OTI
 * @version		initial
 */
public class PrintStream extends OutputStream {

	/**
	 * The target OutputStream for this filter.
	 */
	OutputStream out;

	/**
	 * protect writes to the underlying stream.
	 */
	private Object lock = new Object();
	/**
	 * indicates whether or not this PrintStream has incurred an error.
	 */
	boolean ioError = false;
	/**
	 * indicates whether or not this PrintStream should flush its contents after
	 * printing a new line.
	 */
	boolean autoflush = false;

	private final String lineSeparator = System.getProperty("line.separator");

/**
 * Constructs a new PrintStream on the OutputStream <code>out</code>.  All writes
 * to the target can now take place through this PrintStream.  By default, the
 * PrintStream is set to not autoflush when a newline is encountered.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		out			the OutputStream to provide convenience methods on.
 */
public PrintStream(OutputStream out) {
	this.out = out;
	if (out == null)
		throw new NullPointerException();
}

/**
 * Answers a boolean indicating whether or not this PrintStream has encountered
 * an error.  If so, the receiver should probably be closed since futher writes
 * will not actually take place.  A side effect of calling checkError is that the
 * target OutputStream is flushed.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		<code>true</code> if an error occurred in this PrintStream, <code>false</code> otherwise.
 */
public boolean checkError() {
	if (out != null)
		flush();
	return ioError;
}

/**
 * Close this PrintStream. This implementation flushes and then closes the
 * target stream. If an error occurs, set an error in this PrintStream
 * to <code>true</code>.
 *
 * @author		OTI
 * @version		initial
 *
 */
public void close() {
	synchronized(lock) {
		flush();
		if (out != null) {
			try {
				out.close();
				out = null;
			} catch (IOException e) {
				setError();
			}
		}
	}
}

/**
 * Flush this PrintStream to ensure all pending data is sent out to the target
 * OutputStream.  This implementation flushes the target OutputStream. If
 * an error occurs, set an error in this PrintStream to <code>true</code>.
 *
 * @author		OTI
 * @version		initial
 */
public void flush() {
	synchronized(lock) {
		if (out != null){
			try {
				out.flush();
				return;
			} catch (IOException e) {}
		}
	}
	setError();
}

private void newline() {
	print(lineSeparator);
}

/**
 * Prints the String representation of the character array parameter <code>charArray</code>
 * to the target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		charArray	the character array to print on this PrintStream.
 */
public void print(char[] charArray) {
	print(new String(charArray, 0, charArray.length));
}

/**
 * Prints the String representation of the character parameter <code>ch</code> to the
 * target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		ch			the character to print on this PrintStream.
 */
public void print(char ch) {
	print(String.valueOf(ch));
}

/**
 * Prints the String representation of the <code>double</code> parameter <code>dnum</code> to the
 * target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		dnum		the <code>double</code> to print on this PrintStream.
 */
public void print(double dnum) {
	print(String.valueOf(dnum));
}

/**
 * Prints the String representation of the <code>float</code> parameter <code>fnum</code> to the
 * target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		fnum		the <code>float</code> to print on this PrintStream.
 */
public void print(float fnum) {
	print(String.valueOf(fnum));
}

/**
 * Prints the String representation of the <code>int</code> parameter <code>inum</code> to the
 * target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		inum		the <code>int</code> to print on this PrintStream.
 */
public void print(int inum) {
	print(String.valueOf(inum));
}

/**
 * Prints the String representation of the <code>long</code> parameter <code>lnum</code> to the
 * target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		lnum		the <code>long</code> to print on this PrintStream.
 */
public void print(long lnum) {
	print(String.valueOf(lnum));
}

/**
 * Prints the String representation of the Object parameter <code>obj</code> to the
 * target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		obj			the Object to print on this PrintStream.
 */
public void print(Object obj) {
	print(String.valueOf(obj));
}

/**
 * Prints the String representation of the <code>String</code> parameter <code>str</code> to the
 * target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		str			the <code>String</code> to print on this PrintStream.
 */
public void print(String str) {
	synchronized(lock) {
		if (out == null) {
			setError();
			return;
		}
		if (str == null) {
			print("null");
			return;
		}

		try {
			write(str.getBytes());
		} catch (IOException e) {
			setError();
		}
	}
}

/**
 * Prints the String representation of the <code>boolean</code> parameter <code>bool</code> to the
 * target OutputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		bool		the <code>boolean</code> to print on this PrintStream.
 */
public void print(boolean bool) {
	print(String.valueOf(bool));
}

/**
 * Prints the String representation of the System property <code>"line.separator"</code> to
 * the target OutputStream.
 *
 * @author		OTI
 * @version		initial
 */
public void println() {
	newline();
}

/**
 * Prints the String representation of the character array parameter <code>charArray</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		charArray	the character array to print on this PrintStream.
 */
public void println(char[] charArray) {
	println(new String(charArray, 0, charArray.length));
}

/**
 * Prints the String representation of the character parameter <code>ch</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		ch			the character to print on this PrintStream.
 */
public void println(char ch) {
	println(String.valueOf(ch));
}

/**
 * Prints the String representation of the <code>double</code> parameter <code>dnum</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		dnum		the double to print on this PrintStream.
 */
public void println(double dnum) {
	println(String.valueOf(dnum));
}

/**
 * Prints the String representation of the <code>float</code> parameter <code>fnum</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		fnum		the float to print on this PrintStream.
 */
public void println(float fnum) {
	println(String.valueOf(fnum));
}

/**
 * Prints the String representation of the <code>int</code> parameter <code>inum</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		inum		the int to print on this PrintStream.
 */
public void println(int inum) {
	println(String.valueOf(inum));
}

/**
 * Prints the String representation of the <code>long</code> parameter <code>lnum</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		lnum		the long to print on this PrintStream.
 */
public void println(long lnum) {
	println(String.valueOf(lnum));
}

/**
 * Prints the String representation of the <code>Object</code> parameter <code>obj</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		obj			the <code>Object</code> to print on this PrintStream.
 */
public void println(Object obj) {
	println(String.valueOf(obj));
}

/**
 * Prints the String representation of the <code>String</code> parameter <code>str</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		str			the <code>String</code> to print on this PrintStream.
 */
public void println(String str) {
	synchronized(lock) {
		print(str);
		newline();
	}
}

/**
 * Prints the String representation of the <code>boolean</code> parameter <code>bool</code> to the
 * target OutputStream followed by the System property <code>"line.separator"</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		bool		the boolean to print on this PrintStream.
 */
public void println(boolean bool) {
	println(String.valueOf(bool));
}

protected void setError() {
	ioError = true;
}

/**
 * Writes <code>count</code> <code>bytes</code> from the byte array
 * <code>buffer</code> starting at <code>offset</code> to this
 * PrintStream.  This implementation writes the <code>buffer</code>
 * to the target OutputStream and if this PrintStream is set to
 * autoflush, flushes it. If an error occurs, set an error in this PrintStream
 * to <code>true</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		buffer		the buffer to be written
 * @param		offset		offset in buffer to get bytes
 * @param		count		number of bytes in buffer to write
 *
 * @throws		IndexOutOfBoundsException 	If offset or count are outside of bounds.
 */
public void write(byte[] buffer, int offset, int count) {
	if (buffer != null) {
		// avoid int overflow
		if (0 <= offset && offset <= buffer.length && 0 <= count && count <= buffer.length - offset) {
			synchronized(lock) {
				if (out == null) {
					setError();
					return;
				}
				try {
					out.write(buffer, offset, count);
					if (autoflush) flush();
				} catch (IOException e) {
					setError();
				}
			}
		// K002f = Arguments out of bounds
		} else throw new ArrayIndexOutOfBoundsException(com.ibm.oti.util.Msg.getString("K002f"));
	} else throw new NullPointerException();
}

/**
 * Writes the specified byte <code>oneByte</code> to this PrintStream.  Only
 * the low order byte of <code>oneByte</code> is written.  This implementation
 * writes <code>oneByte</code> to the target OutputStream.  If <code>oneByte</code> is
 * equal to the character <code>'\n'</code> and this PrintSteam is set to autoflush,
 * the target OutputStream is flushed.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		oneByte		the byte to be written
 */
public void write(int oneByte) {
	synchronized(lock) {
		if (out == null) {
			setError();
			return;
		}
		try {
			out.write(oneByte);
			if (autoflush && (oneByte & 0xFF) == '\n')
				flush();
		} catch (IOException e) {
			setError();
		}
	}
}
}
