
package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Bytes are objects (i.e. non-base types) which represent
 * byte values.
 *
 * @author		OTI
 * @version		initial
 */
public final class Byte {

	/**
	 * The value which the receiver represents.
	 */
	final byte value;

	/**
	 * Most positive and most negative possible byte values.
	 */
	public static final byte MAX_VALUE = (byte) 0x7F;
	public static final byte MIN_VALUE = (byte) 0x80;

/**
 * Constructs a new instance of the receiver which represents
 * the byte valued argument.
 *
 * @param		value		the byte to store in the new instance.
 */
public Byte(byte value) {
	this.value = value;
}

/**
 * Answers the byte value which the receiver represents
 *
 * @return		byte		the value of the receiver.
 */
public byte byteValue() {
	return value;
}

/**
 * Compares the argument to the receiver, and answers true
 * if they represent the <em>same</em> object using a class
 * specific comparison.
 * <p>
 * In this case, the argument must also be a Byte, and
 * the receiver and argument must represent the same byte
 * value.
 *
 * @param		object		the object to compare with this object
 * @return		<code>true</code>
 *					if the object is the same as this object
 *				<code>false</code>
 *					if it is different from this object
 * @see			#hashCode
 */
public boolean equals(Object object) {
	return (object == this) ||
		(object instanceof Byte) && (value == ((Byte) object).value);
}

/**
 * Answers an integer hash code for the receiver. Any two
 * objects which answer <code>true</code> when passed to
 * <code>equals</code> must answer the same value for this
 * method.
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode() {
	return value;
}

/**
 * Parses the string argument as if it was a byte value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent a single byte quantity.
 *
 * @param		string		a string representation of a
 *							single byte quantity.
 * @return		byte		the value represented by the argument
 * @throws	NumberFormatException
 *							if the argument could not be parsed
 *							as a byte quantity.
 */
public static byte parseByte(String string) throws NumberFormatException {
	int intValue = Integer.parseInt(string);
	byte result = (byte) intValue;
	if (result == intValue) return result;
	throw new NumberFormatException();
}

/**
 * Parses the string argument as if it was a byte value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent a single byte quantity.
 * The second argument specifies the radix to use when parsing
 * the value.
 *
 * @param		string		a string representation of a
 *							single byte quantity.
 * @param		radix		the radix to use when parsing.
 * @return		byte		the value represented by the argument
 * @throws	NumberFormatException
 *							if the argument could not be parsed
 *							as a byte quantity.
 */
public static byte parseByte(String string, int radix) throws NumberFormatException {
	int intValue = Integer.parseInt(string, radix);
	byte result = (byte) intValue;
	if (result == intValue) return result;
	throw new NumberFormatException();
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @return		a printable representation for the receiver.
 */
public String toString() {
	return Integer.toString(value);
}

}
