package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Integers are objects (non-base types) which represent
 * int values.
 *
 * @author		OTI
 * @version		initial
 */
public final class Integer {

	/**
	 * The value which the receiver represents.
	 */
	final int value;

	/**
	 * Most positive and most negative possible int values.
	 */
	public static final int MAX_VALUE = 0x7FFFFFFF;
	public static final int MIN_VALUE = 0x80000000;

/**
 * Constructs a new instance of the receiver which represents
 * the int valued argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value		the int to store in the new instance.
 */
public Integer(int value) {
	this.value = value;
}

/**
 * Answers the byte value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		byte		the value of the receiver.
 */
public byte byteValue() {
	return (byte) value;
}

/**
 * Answers the double value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		double		the value of the receiver.
 */
public double doubleValue() {
	return (double) value;
}

/**
 * Compares the argument to the receiver, and answers true
 * if they represent the <em>same</em> object using a class
 * specific comparison.
 * <p>
 * In this case, the argument must also be an Integer, and
 * the receiver and argument must represent the same int
 * value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		o		the object to compare with this object
 * @return		<code>true</code>
 *					if the object is the same as this object
 *				<code>false</code>
 *					if it is different from this object
 * @see			#hashCode
 */
public boolean equals(Object o) {
	return (o == this) ||
		(o instanceof Integer) && (value == ((Integer) o).value);
}

/**
 * Answers the float value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		float		the value of the receiver.
 */
public float floatValue() {
	return (float) value;
}

/**
 * Answers an integer hash code for the receiver. Any two
 * objects which answer <code>true</code> when passed to
 * <code>equals</code> must answer the same value for this
 * method.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode() {
	return value;
}

/**
 * Answers the int value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		int		the value of the receiver.
 */
public int intValue() {
	return value;
}

/**
 * Answers the long value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		long		the value of the receiver.
 */
public long longValue() {
	return (long) value;
}

/**
 * Parses the string argument as if it was an int value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent an int quantity.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		a string representation of an
 *							int quantity.
 * @return		int			the value represented by the argument
 * @exception	NumberFormatException
 *							if the argument could not be parsed
 *							as an int quantity.
 */
public static int parseInt(String string) throws NumberFormatException {
	return parseInt(string, 10);
}

/**
 * Parses the string argument as if it was an int value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent an int quantity. The
 * second argument specifies the radix to use when parsing
 * the value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		a string representation of an
 *							int quantity.
 * @param		radix		the base to use for conversion.
 * @return		int			the value represented by the argument
 * @exception	NumberFormatException
 *							if the argument could not be parsed
 *							as an int quantity.
 */
public static int parseInt(String string, int radix) throws NumberFormatException {
	if (string == null || radix < Character.MIN_RADIX || radix > Character.MAX_RADIX)
		throw new NumberFormatException();
	int length = string.length(), i = 0;
	if (length == 0) throw new NumberFormatException(string);
	boolean negative = string.charAt(i) == '-';
	if (negative && ++i == length) throw new NumberFormatException(string);

	return parse(string, i, radix, negative);
}

private static int parse(String string, int offset, int radix, boolean negative) throws NumberFormatException {
	int max = Integer.MIN_VALUE / radix;
	int result = 0, length = string.length();
	while (offset < length) {
		int digit = Character.digit(string.charAt(offset++), radix);
		if (digit == -1) throw new NumberFormatException(string);
		if (max > result) throw new NumberFormatException(string);
		int next = result * radix - digit;
		if (next > result) throw new NumberFormatException(string);
		result = next;
	}
	if (!negative) {
		result = -result;
		if (result < 0) throw new NumberFormatException(string);
	}
	return result;
}

/**
 * Answers the short value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		short		the value of the receiver.
 */
public short shortValue() {
	return (short) value;
}

/**
 * Answers a string containing '0' and '1' characters which
 * describe the binary representation of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		i			an int to get the binary representation of
 * @return		String		the binary representation of the argument
 */
public static String toBinaryString(int i) {
	int count = 1, j = i;

	if (i < 0) count = 32;
	else while ((j >>>= 1) != 0) count++;

	char[] buffer = new char[count];
	do {
		buffer[--count] = (char) ((i & 1) + (int) '0');
		i >>>= 1;
	} while (count > 0);
	return new String(0, buffer.length, buffer);
}

/**
 * Answers a string containing characters in the range
 * 0..9, a..f which describe the hexadecimal representation
 * of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		i			an int to get the hex representation of
 * @return		String		the hex representation of the argument
 */
public static String toHexString(int i) {
	int count = 1, j = i;

	if (i < 0) count = 8;
	else while ((j >>>= 4) != 0) count++;

	char[] buffer = new char[count];
	do {
		int t = i & 15;
		if (t > 9)
			t = t - 10 + (int) 'a';
		else
			t += (int) '0';
		buffer[--count] = (char) t;
		i >>>= 4;
	} while (count > 0);
	return new String(0, buffer.length, buffer);
}

/**
 * Answers a string containing characters in the range 0..7
 * which describe the octal representation of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		i			an int to get the octal representation of
 * @return		String		the hex representation of the argument
 */
public static String toOctalString(int i) {
	int count = 1, j = i;

	if (i < 0) count = 11;
	else while ((j >>>= 3) != 0) count++;

	char[] buffer = new char[count];
	do {
		buffer[--count] = (char) ((i & 7) + (int) '0');
		i >>>= 3;
	} while (count > 0);
	return new String(0, buffer.length, buffer);
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver.
 */
public String toString() {
	return Integer.toString(value);
}

/**
 * Answers a string containing characters in the range
 * 0..9 which describe the decimal representation
 * of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		i			an int to get the representation of
 * @return		String		the representation of the argument
 */
public static String toString(int i) {
	return toString(i, 10);
}

/**
 * Answers a string containing characters in the range
 * 0..9, a..z (depending on the radix) which describe the
 * representation of the argument in that radix.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		i			an int to get the representation of
 * @param		radix		the base to use for conversion.
 * @return		String		the representation of the argument
 */
public static String toString(int i, int radix) {
	if (radix < Character.MIN_RADIX || radix > Character.MAX_RADIX)
		radix = 10;
	if (i == 0) return "0";

	int count = 2, j = i;
	boolean negative = i < 0;
	if (!negative) {
		count = 1;
		j = -i;
	}
	while ((i /= radix) != 0) count++;

	char[] buffer = new char[count];
	do {
		int ch = 0 - (int) (j % radix);
		if (ch > 9)
			ch = ch - 10 + (int) 'a';
		else
			ch += (int) '0';
		buffer[--count] = (char) ch;
	} while ((j /= radix) != 0);
	if (negative) buffer[0] = '-';
	return new String(0, buffer.length, buffer);
}

/**
 * Parses the string argument as if it was an int value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent an int quantity.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		a string representation of an
 *							int quantity.
 * @return		Integer		the value represented by the argument
 * @exception	NumberFormatException
 *							if the argument could not be parsed
 *							as an int quantity.
 */
public static Integer valueOf(String string) throws NumberFormatException {
	return new Integer(parseInt(string));
}

/**
 * Parses the string argument as if it was an int value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent an int quantity. The
 * second argument specifies the radix to use when parsing
 * the value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		a string representation of an
 *							int quantity.
 * @param		radix		the base to use for conversion.
 * @return		Integer		the value represented by the argument
 * @exception	NumberFormatException
 *							if the argument could not be parsed
 *							as an int quantity.
 */
public static Integer valueOf(String string, int radix) throws NumberFormatException {
	return new Integer(parseInt(string, radix));
}
}
