package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Integers are objects (i.e. non-base types) which represent
 * long values.
 *
 * @author		OTI
 * @version		initial
 */
public final class Long {

	/**
	 * The value which the receiver represents.
	 */
	final long value;

	/**
	 * Most positive and most negative possible long values.
	 */
	public static final long MAX_VALUE = 0x7FFFFFFFFFFFFFFFl;
	public static final long MIN_VALUE = 0x8000000000000000l;

/**
 * Constructs a new instance of the receiver which represents
 * the int valued argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value		the long to store in the new instance.
 */
public Long(long value) {
	this.value = value;
}

/**
 * Answers the double value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		double		the value of the receiver.
 */
public double doubleValue() {
	return (double) value;
}

/**
 * Compares the argument to the receiver, and answers true
 * if they represent the <em>same</em> object using a class
 * specific comparison.
 * <p>
 * In this case, the argument must also be an Long, and
 * the receiver and argument must represent the same long
 * value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		o		the object to compare with this object
 * @return		<code>true</code>
 *					if the object is the same as this object
 *				<code>false</code>
 *					if it is different from this object
 * @see			#hashCode
 */
public boolean equals(Object o) {
	return (o == this) ||
		(o instanceof Long) && (value == ((Long) o).value);
}

/**
 * Answers the float value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		float		the value of the receiver.
 */
public float floatValue() {
	return (float) value;
}

/**
 * Answers an integer hash code for the receiver. Any two
 * objects which answer <code>true</code> when passed to
 * <code>equals</code> must answer the same value for this
 * method.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode() {
	return (int)(value ^ (value>>>32));
}

/**
 * Answers the long value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		long		the value of the receiver.
 */
public long longValue() {
	return value;
}

/**
 * Parses the string argument as if it was a long value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent a long quantity.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		a string representation of a
 *							long quantity.
 * @return		long		the value represented by the argument
 * @exception	NumberFormatException
 *							if the argument could not be parsed
 *							as a long quantity.
 */
public static long parseLong(String string) throws NumberFormatException {
	return parseLong(string, 10);
}

/**
 * Parses the string argument as if it was an long value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent an long quantity. The
 * second argument specifies the radix to use when parsing
 * the value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		a string representation of an
 *							long quantity.
 * @param		radix		the base to use for conversion.
 * @return		long			the value represented by the argument
 * @exception	NumberFormatException
 *							if the argument could not be parsed
 *							as an long quantity.
 */
public static long parseLong(String string, int radix) throws NumberFormatException {
	if (string == null || radix < Character.MIN_RADIX || radix > Character.MAX_RADIX)
		throw new NumberFormatException();
	int length = string.length(), i = 0;
	if (length == 0) throw new NumberFormatException(string);
	boolean negative = string.charAt(i) == '-';
	if (negative && ++i == length) throw new NumberFormatException(string);

	return parse(string, i, radix, negative);
}

private static long parse(String string, int offset, int radix, boolean negative) {
	long max = Long.MIN_VALUE / radix;
	long result = 0, length = string.length();
	while (offset < length) {
		int digit = Character.digit(string.charAt(offset++), radix);
		if (digit == -1) throw new NumberFormatException(string);
		if (max > result) throw new NumberFormatException(string);
		long next = result * radix - digit;
		if (next > result) throw new NumberFormatException(string);
		result = next;
	}
	if (!negative) {
		result = -result;
		if (result < 0) throw new NumberFormatException(string);
	}
	return result;
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver.
 */
public String toString() {
	return Long.toString(value);
}
/**
 * Answers a string containing characters in the range
 * 0..9 which describe the decimal representation
 * of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		l			a long to get the representation of
 * @return		String		the representation of the argument
 */
public static String toString(long l) {
	return toString(l, 10);
}

/**
 * Answers a string containing characters in the range
 * 0..9, a..z (depending on the radix) which describe the
 * representation of the argument in that radix.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		l		a long to get the representation of
 * @param		radix		the base to use for conversion.
 * @return		String		the representation of the argument
 */
public static String toString(long l, int radix) {
	if (radix < Character.MIN_RADIX || radix > Character.MAX_RADIX)
		radix = 10;
	if (l == 0) return "0";

	int count = 2;
	long j = l;
	boolean negative = l < 0;
	if (!negative) {
		count = 1;
		j = -l;
	}
	while ((l /= radix) != 0) count++;

	char[] buffer = new char[count];
	do {
		int ch = 0 - (int) (j % radix);
		if (ch > 9)
			ch = ch - 10 + (int) 'a';
		else
			ch += (int) '0';
		buffer[--count] = (char) ch;
	} while ((j /= radix) != 0);
	if (negative) buffer[0] = '-';
	return new String(0, buffer.length, buffer);
}

}
