package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

/**
 * Class math provides various floating point support
 * routines and some standard constants.
 *
 * @author		OTI
 * @version		initial
 */
public final class Math {

	/**
	 * Standard math constants.
	 */
	public static final double E=2.718281828459045;
	public static final double PI=3.141592653589793;

/**
 * Prevents this class from being instantiated.
 */
private Math() {
}
/**
 * Answers the absolute value of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d		the value to be converted
 * @return		the argument if it is positive, otherwise
 *              the negation of the argument.
 */
public static double abs (double d) {
	long bits = Double.doubleToLongBits(d);
	bits &= 0x7fffffffffffffffL;
	return Double.longBitsToDouble(bits);
}

/**
 * Answers the absolute value of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		f		the value to be converted
 * @return		the argument if it is positive, otherwise
 *              the negation of the argument.
 */
public static float abs (float f) {
	int bits = Float.floatToIntBits(f);
	bits &= 0x7fffffff;
	return Float.intBitsToFloat(bits);
}

/**
 * Answers the absolute value of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		i		the value to be converted
 * @return		the argument if it is positive, otherwise
 *              the negation of the argument.
 */
public static int abs (int i) {
	return i >= (int) 0 ? i : -i;
}
/**
 * Answers the absolute value of the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		l		the value to be converted
 * @return		the argument if it is positive, otherwise
 *              the negation of the argument.
 */
public static long abs (long l) {
	return l >= (long) 0 ? l : -l;
}

/**
 * Answers the double conversion of the most negative (i.e.
 * closest to negative infinity) integer value which is
 * greater than the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d		the value to be converted
 * @return		the ceiling of the argument.
 */
public static native double ceil (double d);

/**
 * Answers the closest double approximation of the
 * cosine of the argument
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d		the value to compute cos of
 * @return		the cosine of the argument.
 */
public static native double cos (double d);

/**
 * Answers the double conversion of the most positive (i.e.
 * closest to positive infinity) integer value which is
 * less than the argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d		the value to be converted
 * @return		the ceiling of the argument.
 */
public static native double floor (double d);

/**
 * Answers the most positive (i.e. closest to positive
 * infinity) of the two arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d1		the first argument to check
 * @param		d2		the second argument
 * @return		the larger of d1 and d2.
 */
public static double max (double d1, double d2) {
	if (d1 > d2) return d1;
	if (d1 < d2) return d2;
	/* if either arg is NaN, return NaN */
	if(d1 != d2) return Double.NaN;
	/* max( +0.0,-0.0) == +0.0 */
	if (d1 == 0.0 &&
		((Double.doubleToLongBits(d1) & Double.doubleToLongBits(d2)) & 0x8000000000000000L) == 0)
			return 0.0;
	return d1;
}

/**
 * Answers the most positive (i.e. closest to positive
 * infinity) of the two arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		f1		the first argument to check
 * @param		f2		the second argument
 * @return		the larger of f1 and f2.
 */
public static float max (float f1, float f2) {
	if (f1 > f2) return f1;
	if(f1 < f2) return f2;
	/* if either arg is NaN, return NaN */
	if (f1 != f2) return Float.NaN;
	/* max( +0.0,-0.0) == +0.0 */
	if (f1 == 0.0f &&
		((Float.floatToIntBits(f1) & Float.floatToIntBits(f2)) & 0x80000000) == 0)
			return 0.0f;
	return f1;
}

/**
 * Answers the most positive (i.e. closest to positive
 * infinity) of the two arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		i1		the first argument to check
 * @param		i2		the second argument
 * @return		the larger of i1 and i2.
 */
public static int max (int i1, int i2) {
	return i1 > i2 ? i1 : i2;
}
/**
 * Answers the most positive (i.e. closest to positive
 * infinity) of the two arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		l1		the first argument to check
 * @param		l2		the second argument
 * @return		the larger of l1 and l2.
 */
public static long max (long l1, long l2) {
	return l1 > l2 ? l1 : l2;
}

/**
 * Answers the most negative (i.e. closest to negative
 * infinity) of the two arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d1		the first argument to check
 * @param		d2		the second argument
 * @return		the smaller of d1 and d2.
 */
public static double min (double d1, double d2) {
	if (d1 > d2) return d2;
	if (d1 < d2) return d1;
	/* if either arg is NaN, return NaN */
	if (d1 != d2) return Double.NaN;
	/* min( +0.0,-0.0) == -0.0 */
	if(d1==0.0 &&
		((Double.doubleToLongBits(d1) | Double.doubleToLongBits(d2)) & 0x8000000000000000l) != 0)
			return 0.0*(-1.0);
	return d1;
}

/**
 * Answers the most negative (i.e. closest to negative
 * infinity) of the two arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		f1		the first argument to check
 * @param		f2		the second argument
 * @return		the smaller of f1 and f2.
 */
public static float min (float f1, float f2) {
	if (f1 > f2) return f2;
	if (f1 < f2) return f1;
	/* if either arg is NaN, return NaN */
	if (f1 != f2) return Float.NaN;
	/* min( +0.0,-0.0) == -0.0 */
	if(f1==0.0f &&
		((Float.floatToIntBits(f1) | Float.floatToIntBits(f2)) & 0x80000000) != 0)
			return 0.0f*(-1.0f);
	return f1;
}

/**
 * Answers the most negative (i.e. closest to negative
 * infinity) of the two arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		i1		the first argument to check
 * @param		i2		the second argument
 * @return		the smaller of i1 and i2.
 */
public static int min (int i1, int i2) {
	return i1 < i2 ? i1 : i2;
}
/**
 * Answers the most negative (i.e. closest to negative
 * infinity) of the two arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		l1		the first argument to check
 * @param		l2		the second argument
 * @return		the smaller of l1 and l2.
 */
public static long min (long l1, long l2) {
	return l1 < l2 ? l1 : l2;
}

/**
 * Answers the closest double approximation of the
 * sine of the argument
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d		the value to compute sin of
 * @return		the sine of the argument.
 */
public static native double sin (double d);

/**
 * Answers the closest double approximation of the
 * square root of the argument
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d		the value to compute sqrt of
 * @return		the square root of the argument.
 */
public static native double sqrt (double d);

/**
 * Answers the closest double approximation of the
 * tangent of the argument
 *
 * @author		OTI
 * @version		initial
 *
 * @param		d		the value to compute tan of
 * @return		the tangent of the argument.
 */
public static native double tan (double d);

/**
 * Returns the measure in radians of the supplied degree angle
 *
 * @author		OTI
 * @version		initial
 *
 * @param		angdeg		an angle in degrees
 * @return		the radian measure of the angle.
 */
public static double toRadians(double angdeg) {
	return angdeg / 180d * PI;
}

/**
 * Returns the measure in degrees of the supplied radian angle
 *
 * @author		OTI
 * @version		initial
 *
 * @param		angrad		an angle in radians
 * @return		the degree measure of the angle.
 */
public static double toDegrees(double angrad) {
	return angrad * 180d / PI;
}

}
