
package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Shorts are objects (non-base types) which represent
 * short values.
 *
 * @author		OTI
 * @version		initial
 */
public final class Short {

	/**
	 * The value which the receiver represents.
	 */
	final short value;

	/**
	 * Most positive and most negative possible short values.
	 */
	public static final short MAX_VALUE = (short) 0x7FFF;
	public static final short MIN_VALUE = (short) 0x8000;

/**
 * Constructs a new instance of the receiver which represents
 * the short valued argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		value		the short to store in the new instance.
 */
public Short(short value) {
	this.value = value;
}

/**
 * Compares the argument to the receiver, and answers true
 * if they represent the <em>same</em> object using a class
 * specific comparison.
 * <p>
 * In this case, the argument must also be a Short, and
 * the receiver and argument must represent the same short
 * value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		object		the object to compare with this object
 * @return		<code>true</code>
 *					if the object is the same as this object
 *				<code>false</code>
 *					if it is different from this object
 * @see			#hashCode
 */
public boolean equals(Object object) {
	return (object == this) ||
		(object instanceof Short) && (value == ((Short) object).value);
}

/**
 * Answers an integer hash code for the receiver. Any two
 * objects which answer <code>true</code> when passed to
 * <code>equals</code> must answer the same value for this
 * method.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the receiver's hash
 *
 * @see			#equals
 */
public int hashCode() {
	return value;
}

/**
 * Parses the string argument as if it was a short value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent an short quantity.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		a string representation of a
 *							short quantity.
 * @return		short		the value represented by the argument
 * @exception	NumberFormatException
 *							if the argument could not be parsed
 *							as a short quantity.
 */
public static short parseShort(String string) throws NumberFormatException {
	return parseShort(string, 10);
}

/**
 * Parses the string argument as if it was a short value
 * and returns the result. Throws NumberFormatException
 * if the string does not represent a single short quantity.
 * The second argument specifies the radix to use when parsing
 * the value.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		string		a string representation of a short quantity.
 * @param		radix		the radix to use when parsing.
 * @return		short		the value represented by the argument
 * @exception	NumberFormatException
 *							if the argument could not be parsed
 *							as a short quantity.
 */
public static short parseShort(String string, int radix) throws NumberFormatException {
	int intValue = Integer.parseInt(string, radix);
	short result = (short) intValue;
	if (result == intValue) return result;
	throw new NumberFormatException();
}

/**
 * Answers the short value which the receiver represents
 *
 * @author		OTI
 * @version		initial
 *
 * @return		short		the value of the receiver.
 */
public short shortValue() {
	return value;
}

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		a printable representation for the receiver.
 */
public String toString() {
	return Integer.toString(value);
}

}
