package java.lang;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 *	A Thread is a unit of concurrent execution in Java. It has its own call stack
 * for methods being called and their parameters. Threads in the same VM interact and
 * synchronize by the use of shared Objects and monitors associated with these objects.
 * Synchronized methods and part of the API in Object also allow Threads to cooperate.
 *
 *	When a Java program starts executing there is an implicit Thread (called "main")
 * which is automatically created by the VM. This Thread belongs to a ThreadGroup
 * (also called "main") which is automatically created by the bootstrap sequence by
 * the VM as well.
 *
 * @version		initial
 *
 * @see			java.lang.Object
 * @see			java.lang.ThreadGroup
 */
public class Thread implements Runnable {

/* Maintain thread shape in all configs */

    static Object activeCountLock = new Object();
	static int activeCount = 0;					// The active threads
	public final static int MAX_PRIORITY = 10;		// Maximum allowed priority for a thread
	public final static int MIN_PRIORITY = 1;			// Minimum allowed priority for a thread
	public final static int NORM_PRIORITY = 5;		// Normal priority for a thread
	private static int createCount = -1;					// Used internally to compute Thread names that comply with the Java specification
	private static final int NANOS_MAX = 999999;		// Max value for nanoseconds parameter to sleep and join
	static final long NO_REF = 0;				// Symbolic constant, no threadRef assigned or already cleaned up

	// Instance variables
	long threadRef;									// Used by the VM
	long stackSize = 0;
	private volatile boolean started;				// If !isAlive(), tells if Thread died already or hasn't even started
	private String name = null;						// The Thread's name
	private int priority = NORM_PRIORITY;			// The Thread's current priority
	private boolean isDaemon = false;				// Tells if the Thread is a daemon thread or not.
	Object threadGroup = null;				// Keep vm happy
	private Runnable runnable = null;				// Target (optional) runnable object
	private Throwable stopThrowable = null;			// Used by the VM
	private Object contextClassLoader;				// Keep vm happy
	private Object localStorage;						// Keep vm happy
	private Object accessControlContext;			// Keep vm happy

	Object lock = new Object();

	private Object slot1;
	private Object slot2;

	private Object slot3;

	private long longSlot1;

	private Object slot4;

	private static boolean booting = true;

/**
 * 	Constructs a new Thread with no runnable object and a newly generated name.
 * The new Thread will belong to the same ThreadGroup as the Thread calling
 * this constructor.
 *
 * @version		initial
 *
 * @see			java.lang.ThreadGroup
 */
public Thread() {
	this(newName());
}

/**
 *
 * Private constructor to be used by the VM for the threads attached through JNI.
 * They already have a running thread with no associated Java Thread, so this is
 * where the binding is done.
 *
 *
 * @version		initial
 *
 * @param		vmName			Name for the Thread being created (or null to auto-generate a name)
 * @param		vmThreadGroup	ThreadGroup for the Thread being created (or null for main threadGroup)
 * @param		vmPriority		Priority for the Thread being created
 * @param		vmIsDaemon		Indicates whether or not the Thread being created is a daemon thread
 *
 * @see			java.lang.ThreadGroup
 */
private Thread(String vmName, Object vmThreadGroup, int vmPriority, boolean vmIsDaemon) {
	super();

	name = (vmName == null) ? newName() : vmName;
	isDaemon = vmIsDaemon;
	priority = vmPriority;	// If we called setPriority(), it would have to be after setting the ThreadGroup (further down),
							// because of the checkAccess() call (which requires the ThreadGroup set). However, for the main
							// Thread or JNI-C attached Threads we just trust the value the VM is passing us, and just assign.

	Object group = null;

	initialize(group, null);	// no parent Thread
	if (booting) {
		booting = false;
		System.completeInitialization();
	}
}

/*
 * Called after everything else is initialized.
 */
void completeInitialization() {
}

/**
 * 	Constructs a new Thread with a runnable object and a newly generated name.
 * The new Thread will belong to the same ThreadGroup as the Thread calling
 * this constructor.
 *
 * @version		initial
 *
 * @param		runnable		a java.lang.Runnable whose method <code>run</code> will be executed by the new Thread
 *
 * @see			java.lang.ThreadGroup
 * @see			java.lang.Runnable
 */
public Thread(Runnable runnable) {
	this.name = newName();		// We avoid the public API 'setName', since it does redundant work (checkAccess)
	this.runnable = runnable;	// No API available here, so just direct access to inst. var.
	Thread currentThread  = currentThread();
	initialize(null, currentThread);
	setPriority(currentThread.getPriority());	// In this case we can call the public API according to the spec - 20.20.10
}

/**
 * Constructs a new Thread with a runnable object and name provided.
 * The new Thread will belong to the same ThreadGroup as the Thread calling
 * this constructor.
 *
 * @version		initial
 *
 *
 * @param		runnable		a java.lang.Runnable whose method <code>run</code> will be executed by the new Thread
 * @param		threadName		Name for the Thread being created
 *
 * @see			java.lang.ThreadGroup
 * @see			java.lang.Runnable
 */
public Thread(Runnable runnable, String threadName) {
	this.name = threadName;		// We avoid the public API 'setName', since it does redundant work (checkAccess)
	this.runnable = runnable;	// No API available here, so just direct access to inst. var.
	Thread currentThread  = currentThread();
	initialize(null, currentThread);
	setPriority(currentThread.getPriority());	// In this case we can call the public API according to the spec - 20.20.10
}

/**
 * 	Constructs a new Thread with no runnable object and the name provided.
 * The new Thread will belong to the same ThreadGroup as the Thread calling
 * this constructor.
 *
 * @version		initial
 *
 * @param		threadName		Name for the Thread being created
 *
 * @see			java.lang.ThreadGroup
 * @see			java.lang.Runnable
 */
public Thread(String threadName) {
	this(null, threadName);
}

/**
 * 	Initialize the thread according to its parent Thread and the ThreadGroup
 * where it should be added.
 *
 * @version		initial
 *
 * @param		parentThread	Thread	The creator Thread from which to inherit some values like local storage, etc.
 *										If null, the receiver is either the main Thread or a JNI-C attached Thread.
 * @param		group	ThreadGroup		The ThreadGroup to which the receiver is being added.
 */
private void initialize(Object group, Thread parentThread) {

	threadGroup = group;

	synchronized( activeCountLock ) {
		activeCount++;
	}
}

/**
 * Returns how many threads are active in the <code>ThreadGroup</code>
 * which the current thread belongs to.
 *
 * @version initial
 *
 *
 * @return Number of Threads
 *
 */
public static int activeCount(){
	return activeCount;
}

/**
 * Answers the instance of Thread that corresponds to the running Thread
 * which calls this method.
 *
 * @version		initial
 *
 * @return		a java.lang.Thread corresponding to the code that called <code>currentThread()</code>
 */
public static native Thread currentThread();

/**
 * Answers the name of the receiver.
 *
 * @version		initial
 *
 * @return		the receiver's name (a java.lang.String)
 */
public final String getName() {
	return String.valueOf(name);
}

/**
 * Answers the priority of the receiver.
 *
 * @version		initial
 *
 * @return		the receiver's priority (an <code>int</code>)
 *
 * @see			Thread#setPriority
 */
public final int getPriority() {
	return priority;
}

/**
 * Posts an interrupt request to the receiver
 *
 * @version		initial
 *
 *
 * @exception	SecurityException
 *					if <code>group.checkAccess()</code> fails with a SecurityException
 *
 * @see			java.lang.SecurityException
 * @see			java.lang.SecurityManager
 * @see			Thread#interrupted
 * @see			Thread#isInterrupted
 */
public void interrupt() {
	synchronized(lock) {
		interruptImpl();
	}
}

/**
 * Posts an interrupt request to the receiver
 *
 * @version		initial
 *
 *
 * @see			Thread#interrupted
 * @see			Thread#isInterrupted
 */
private native void interruptImpl();

/**
 * Answers <code>true</code> if the receiver has
 * already been started and still runs code (hasn't died yet).
 * Answers <code>false</code> either if the receiver hasn't been
 * started yet or if it has already started and run to completion and died.
 *
 * @version		initial
 *
 * @return		a <code>boolean</code>
 *
 * @see			Thread#start
 */
public final boolean isAlive() {
	synchronized(lock) {
		return threadRef != NO_REF;
	}
}

/**
 * Answers <code>true</code> if the receiver has
 * already died and been removed from the ThreadGroup
 * where it belonged.
 *
 * @version		initial
 *
 * @return		a <code>boolean</code>
 *
 * @see			Thread#start
 * @see			Thread#isAlive
 */
private boolean isDead() {
	// Has already started, is not alive anymore, and has been removed from the ThreadGroup
	synchronized(lock) {
		return started && threadRef == NO_REF;
	}
}

/**
 * Blocks the current Thread (<code>Thread.currentThread()</code>) until the
 * receiver finishes its execution and dies.
 *
 * @version		initial
 *
 * @exception	InterruptedException
 *					if <code>interrupt()</code> was called for the receiver while
 *					it was in the <code>join()</code> call
 *
 * @see			Object#notifyAll
 * @see			java.lang.ThreadDeath
 */
public final synchronized void join() throws InterruptedException {
	if (started)
		while (!isDead())
			wait(0);
}

/**
 * Private method that generates Thread names that comply with the Java specification
 *
 * @version		initial
 *
 * @return		a java.lang.String representing a name for the next Thread being generated
 *
 * @see			Thread#createCount
 */
private synchronized static String newName() {
	if (createCount == -1) {
		createCount++;
		return "main";
	} else {
		return "Thread-" + createCount++;
	}
}

/**
 * Calls the <code>run()</code> method of the Runnable object the receiver holds.
 * If no Runnable is set, does nothing.
 *
 * @version		initial
 *
 * @see			Thread#start
 */
public void run() {
	if (runnable != null) {
		runnable.run();
	}
}

/**
 * Sets the priority of the receiver. Note that the final priority set may not be the
 * parameter that was passed - it will depend on the receiver's ThreadGroup. The priority
 * cannot be set to be higher than the receiver's ThreadGroup's maxPriority().
 *
 * @version		initial
 *
 * @param		priority		new priority for the Thread
 *
 * @exception	SecurityException
 *					if <code>checkAccess()</code> fails with a SecurityException
 * @exception	IllegalArgumentException
 *					if the new priority is greater than Thread.MAX_PRIORITY or less than
 *					Thread.MIN_PRIORITY
 *
 * @see			Thread#getPriority
 */
public final void setPriority(int priority){
		if (MIN_PRIORITY <= priority && priority <= MAX_PRIORITY) {
			int finalPriority = priority;
			this.priority = finalPriority;
			synchronized(lock) {
				setPriorityImpl(finalPriority);
			}
		} else throw new IllegalArgumentException();
}

/**
 * Private method to tell the VM that the priority for the receiver is changing.
 *
 * @version		initial
 *
 * @param		priority		new priority for the Thread
 *
 *
 * @see			Thread#setPriority
 */
private native void setPriorityImpl(int priority);

/**
 * Causes the thread which sent this message to sleep an interval
 * of time (given in milliseconds). The precision is not guaranteed -
 * the Thread may sleep more or less than requested.
 *
 * @version		initial
 *
 * @param		time		The time to sleep in milliseconds.
 *
 * @exception	InterruptedException
 *					if <code>interrupt()</code> was called for this Thread while it was sleeping
 *
 * @see			Thread#interrupt()
 */

public static void sleep(long time) throws InterruptedException {
	sleep(time, 0);
}

static native void sleep(long time, int nanos) throws InterruptedException;

/**
 * Starts the new Thread of execution. The <code>run()</code> method of the receiver
 * will be called by the receiver Thread itself (and not the Thread calling <code>start()</code>).
 *
 * @version		initial
 *
 *
 * @exception	IllegalThreadStateException
 *					Unspecified in the Java language specification
 *
 * @see			Thread#run
 */
public synchronized void start() {
 	synchronized(lock) {
 		if (started) {
 			// K0341 = Thread is already started
 			throw new IllegalThreadStateException(com.ibm.oti.util.Msg.getString("K0341"));
 		}
 			startImpl();
	}
 }

private native void startImpl();

/**
 * Answers a string containing a concise, human-readable
 * description of the receiver.
 *
 * @version		initial
 *
 * @return		a printable representation for the receiver.
 */
public String toString() {
	return "Thread[" + name + "," + priority + "]" ;
}

/**
 * Causes the thread which sent this message to yield execution to another Thread
 * that is ready to run. The actual scheduling is implementation-dependent.
 *
 * @version		initial
 */
public static native void yield();

}
