package java.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * This class provides methods that generates pseudorandom number of different types,
 * such as int, long, double and float using either
 *
 * @author		OTI
 * @version		initial
 *
 * @see			Properties
 * @see			PropertyResourceBundle
 */

public class Random {
	static final long multiplier = 0x5deece66dL;

	/**
	 * The boolean value indicating if the second Gaussian number is available.
 	 *
	 * @serial
	 */
	boolean haveNextNextGaussian = false;

	/**
	 *
	 * @serial It is associated with the internal state of this generator.
	 */
	long seed;

	/**
	 * The second Gaussian generated number.
 	 *
	 * @serial
	 */
	double nextNextGaussian=0;
/**
 * Construct a random generator with the curent time of day in milliseconds as the initial state.
 *
 * @see 		#setSeed
 */
public Random() {
	setSeed(System.currentTimeMillis());
}
/**
 * Construct a random generator with the given <code>seed</code> as the initial state.
 *
 * @param seed  the seed that will determine the initial state of this random number generator
 *
 * @see 		#setSeed
 */
public Random(long seed) {
	setSeed(seed);
}
/**
 * Answers a pseudorandom uniformly distributed <code>int</code> value of the number of bits
 * specified by the argument <code>bits</code> as described by Donald E. Knuth in
 * <i>The Art of Computer Programming, Volume 2: Seminumerical Algorithms</i>, section 3.2.1.
 *
 * @return int		a pseudorandom generated int number
 * @param bits 		number of bits of the returned value
 *
 * @see 		#nextBytes
 * @see 		#nextDouble
 * @see 		#nextFloat
 * @see 		#nextInt()
 * @see 		#nextInt(int)
 * @see 		#nextGaussian
 * @see 		#nextLong
 */
protected synchronized int next(int bits) {
	seed = (seed * multiplier + 0xbL) & ((1L << 48) - 1);
	return (int)(seed >>> (48 - bits));
}

/**
 * Generates a normally distributed random double number between 0.0 inclusively and 1.0 exclusively.
 *
 * @return double
 *
 * @see 		#nextFloat
 */
public double nextDouble() {
	return ((((long)next(26) << 27) + next(27)) / (double)(1L << 53));
}

/**
 * Generates a normally distributed random float number between 0.0 inclusively and 1.0 exclusively.
 *
 * @return float	a random float number between 0.0 and 1.0
 *
 * @see 		#nextDouble
 */
public float nextFloat() {
	return (next(24) / 16777216f);
}

/**
 * Generates a uniformly distributed 32-bit <code>int</code> value from the this random number sequence.
 *
 * @return int	uniformly distributed <code>int</code> value
 *
 * @see 		java.lang.Integer#MAX_VALUE
 * @see 		java.lang.Integer#MIN_VALUE
 * @see 		#next
 * @see 		#nextLong
 */
public int nextInt() {
	return next(32);
}

/**
 * Answers a pseudorandom, uniformly distributed <code>int</code> value
 * between 0 (inclusively) and the specified n (exclusively).
 *
 * @return int
 * @param n int
 */
public int nextInt(int n) {
	if (n > 0) {
		if ((n & -n) == n) return (int) ((n * (long) next(31)) >> 31);
		int bits, val;
		do {
			bits = next(31);
			val = bits % n;
		} while (bits - val + (n - 1) < 0);
		return val;
	} else throw new IllegalArgumentException();
}

/**
 * Generates a uniformly distributed 64-bit <code>int</code> value from the this random number sequence.
 *
 * @return 64-bit <code>int</code> random number
 *
 * @see 		java.lang.Integer#MAX_VALUE
 * @see 		java.lang.Integer#MIN_VALUE
 * @see 		#next
 * @see 		#nextInt()
 * @see			#nextInt(int)
 */
public long nextLong() {
	return ((long)next(32) << 32) + next(32);
}
/**
 * Modifies the seed using linear congruential formula presented in
 * <i>The Art of Computer Programming, Volume 2</i>, Section 3.2.1.
 *
 * @param seed 		the seed that alters the state of the random number generator
 *
 * @see 		#next
 * @see 		#Random()
 * @see 		#Random(long)
 */
public synchronized  void setSeed(long seed) {
	this.seed = (seed ^ multiplier) & ((1L << 48) - 1);
	haveNextNextGaussian = false;
}
}
