package java.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * TimeZone is an abstract class which represents a local time zone and its
 * daylight savings time rules. Subclasses support a particular calendar type,
 * such as the gregorian calendar.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			GregorianCalendar
 * @see			SimpleTimeZone
 */

abstract public class TimeZone {
	private static Hashtable AvailableZones;
	private static TimeZone Default;
	static final TimeZone GMT = new SimpleTimeZone(0, "GMT");	// Greenwich Mean Time

private static void initializeAvailable() {
	TimeZone[] zones = TimeZones.getTimeZones();
	AvailableZones = new Hashtable((zones.length + 1) * 4 / 3);
	AvailableZones.put(GMT.getID(), GMT);
	for (int i=0; i<zones.length; i++)
		AvailableZones.put(zones[i].getID(), zones[i]);
}
/**
 * Constructs a new instance of this class.
 *
 * @author		OTI
 * @version		initial
 */
public TimeZone() {
}

/**
 * Gets the available time zone IDs.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		an array of time zone ID strings
 */
public static synchronized String[] getAvailableIDs() {
	if (AvailableZones == null) initializeAvailable();
	int length = AvailableZones.size();
	String[] answer = new String[length];
	Enumeration zones = AvailableZones.elements();
	for (int i = 0; i < length; i++)
		answer[i] = ((TimeZone) zones.nextElement()).getID();
	return answer;
}

/**
 * Gets the default time zone.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the default time zone
 */
public static synchronized TimeZone getDefault() {
	if (Default == null)
		setDefault(null);
	return Default;
}

/**
 * Gets the ID of this TimeZone.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the time zone ID string
 */
public String getID() {
	return null;
}

/**
 * Gets the offset from GMT of this TimeZone for the specified date and time. The
 * offset includes daylight savings time if the specified date and time are within
 * the daylight savings time period.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		era	the GregorianCalendar era, either GregorianCalendar.BC or
 *					GregorianCalendar.AD
 * @param		year	the year
 * @param		month	the Calendar month
 * @param		day	the day of the month
 * @param		dayOfWeek	the Calendar day of the week
 * @param		time	the time of day in milliseconds
 * @return		the offset from GMT in milliseconds
 */
abstract public int getOffset(int era, int year, int month, int day, int dayOfWeek, int time);
/**
 * Gets the offset for standard time from GMT for this TimeZone.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the offset from GMT in milliseconds
 */
abstract public int getRawOffset();

/**
 * Gets the time zone with the specified ID.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name	a time zone string ID
 * @return		the time zone with the specified ID or null if a time zone with the
 *				specified ID does not exist
 */
public static synchronized TimeZone getTimeZone(String name) {
	if (AvailableZones == null) initializeAvailable();
	TimeZone zone = (TimeZone)AvailableZones.get(name);
	if (zone == null) {
		if (name.startsWith("GMT") && name.length() > 3) {
			char sign = name.charAt(3);
			if (sign == '+' || sign == '-') {
				int[] position = new int[1];
				String formattedName = formatTimeZoneName(name, 4);
				if (formattedName == null) {
					return GMT;
				}
				int hour = parseNumber(formattedName, 4, position);
				if (hour < 0 || hour > 23) {
					return GMT;
				}
				int index = position[0];
				if (index != -1) {
					int raw = hour * 3600000;
					if (index < formattedName.length() && formattedName.charAt(index) == ':') {
						int minute = parseNumber(formattedName, index + 1, position);
						if (position[0] == -1 || minute < 0 || minute > 59)
							return GMT;
						raw += minute * 60000;
					} else if (hour >= 30 || index > 6) {
						raw = (hour / 100 * 3600000) + (hour % 100 * 60000);
					}
					if (sign == '-') raw = -raw;
					return new SimpleTimeZone(raw, formattedName);
				}
			}
		}
		zone = GMT;
	}
	return zone;
}

private static String formatTimeZoneName(String name, int offset) {
	StringBuffer buf = new StringBuffer();
	int index = offset, length = name.length();
	buf.append(name.substring(0, offset));
	int colonIndex = -1;

	while (index < length) {
		char c = name.charAt(index);
		// allow only basic latin block digits
		if ('0' <= c && c <= '9') {
			buf.append(c);
			if ((length-(index+1)) == 2) {
				if (colonIndex != -1) return null;
				colonIndex = buf.length();
				buf.append(':');
			}
		} else if (c == ':') {
			if (colonIndex != -1) return null;
			colonIndex = buf.length();
			buf.append(':');
		} else {
			// invalid name
			return null;
		}
		index++;
	}

	if (colonIndex == -1) {
		colonIndex = buf.length();
		buf.append(":00");
	}

	if (colonIndex == 5) {
		buf.insert(4, '0');
	}

	return buf.toString();
}

private static int parseNumber(String string, int offset, int[] position) {
	int index = offset, length = string.length(), digit, result = 0;
	while (index < length && (digit = Character.digit(string.charAt(index), 10)) != -1) {
		index++;
		result = result * 10 + digit;
	}
	position[0] = index == offset ? -1 : index;
	return result;
}

/**
 * Sets the default time zone.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		timezone	a TimeZone object
 */
private static void setDefault(TimeZone timezone) {
	if (timezone != null) {
		Default = timezone;
		return;
	}

	String zone = System.getProperty("user.timezone");

	// if property user.timezone is not set, we call the native method getCustomTimeZone
	if (zone == null)
	{
		int[] tzinfo = new int[10];
		boolean[] isCustomTimeZone = new boolean[1];

		String zoneId = getCustomTimeZone(tzinfo, isCustomTimeZone);

		// if returned TimeZone is a user customized TimeZone
		if (isCustomTimeZone[0])
		{
			// build a new SimpleTimeZone
			switch(tzinfo[1])
			{
				case 0:
					// does not observe DST
					Default = new SimpleTimeZone(tzinfo[0], zoneId);
					break;
				default:
					// observes DST
					Default = new SimpleTimeZone(tzinfo[0], zoneId, tzinfo[5], tzinfo[4], tzinfo[3], tzinfo[2], tzinfo[9], tzinfo[8], tzinfo[7], tzinfo[6], tzinfo[1]);
			}
		}
		else
		{
			// get TimeZone from CLDC list
			Default = getTimeZone(zoneId);
		}
	}
	else
	{
		// if property user.timezone is set in command line (with -D option)
		Default = getTimeZone(zone);
	}
}

/**
 * Answers if this TimeZone has a daylight savings time period.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		true if this time zone has a daylight savings time period, false otherwise
 */
abstract public boolean useDaylightTime();

/**
 * Gets the name and the details of the user-selected TimeZone on the device.
 *
 * @author IBM
 * @param tzinfo int array of 10 elements to be filled with the TimeZone information.
 * Once filled, the contents of the array are formatted as follows:
 * 	tzinfo[0] -> the timezone offset;
 *	tzinfo[1] -> the dst adjustment;
 *	tzinfo[2] -> the dst start hour;
 *	tzinfo[3] -> the dst start day of week;
 *	tzinfo[4] -> the dst start week of month;
 *	tzinfo[5] -> the dst start month;
 *	tzinfo[6] -> the dst end hour;
 *	tzinfo[7] -> the dst end day of week;
 *	tzinfo[8] -> the dst end week of month;
 *	tzinfo[9] -> the dst end month;
 * @param isCustomTimeZone boolean array of size 1 that indicates if a CLDC timezone match is found
 * @return the name of the TimeZone or null if error occurs in native method.
 */
private static native String getCustomTimeZone(int[] tzinfo, boolean[] isCustomTimeZone);
}
