package java.util;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2006  All Rights Reserved
 */

/**
 * Vector is a variable size contiguous indexable array of Objects.
 * The size of the Vector is the number of Objects it contains.
 * The capacity of the Vector is the number of Objects it can hold.
 * <p>
 * Objects may be inserted at any position up to the size of the
 * Vector, increasing the size of the Vector. Objects at any
 * position in the Vector may be removed, shrinking the size of
 * the Vector. Objects at any position in the Vector may be replaced,
 * which does not affect the Vector size.
 * <p>
 * The capacity of a Vector may be specified when the Vector is
 * created. If the capacity of the Vector is exceeded, the capacity
 * is increased, doubling by default.
 *
 * @author		OTI
 * @version		initial
 *
 * @see			java.lang.StringBuffer
 */

public class Vector {
	/**
	 * The number of elements or the size of the vector.
	 */
	protected int elementCount;
	/**
	 * The elements of the vector.
	 */
	protected Object [] elementData;
	/**
	 * The amount by which the capacity of the vector is increased.
	 */
	protected int capacityIncrement;

/**
 * Constructs a new Vector using the default capacity.
 */
public Vector () {
	this (10, 0);
}
/**
 * Constructs a new Vector using the specified capacity.
 *
 * @param		capacity	the initial capacity of the new vector
 */
public Vector (int capacity) {
	this(capacity, 0);
}
/**
 * Constructs a new Vector using the specified capacity and
 * capacity increment.
 *
 * @param		capacity	the initial capacity of the new Vector
 * @param		capacityIncrement	the amount to increase the capacity
 					when this Vector is full
 */
public Vector (int capacity, int capacityIncrement) {
	elementCount = 0;
	try {
		elementData = new Object[capacity];
	} catch (NegativeArraySizeException e) {
		throw new IllegalArgumentException();
	}
	this.capacityIncrement = capacityIncrement;
}

/**
 * Adds the specified object at the end of this Vector.
 *
 * @param		object	the object to add to the Vector
 */
public synchronized void addElement (Object object) {
	if (elementCount == elementData.length) growByOne();
	elementData[elementCount++] = object;
}

/**
 * Answers the number of elements this Vector can hold without
 * growing.
 *
 * @return		the capacity of this Vector
 *
 * @see			#ensureCapacity
 * @see			#size
 */
public synchronized int capacity() {
	return elementData.length;
}

/**
 * Searches this Vector for the specified object.
 *
 * @param		object	the object to look for in this Vector
 * @return		true if object is an element of this Vector, false otherwise
 *
 * @see			#indexOf(Object)
 * @see			#indexOf(Object, int)
 * @see			java.lang.Object#equals
 */
public boolean contains (Object object) {
	return indexOf (object, 0) != -1;
}

/**
 * Copies the elements of this Vector into the specified Object array.
 *
 * @param		elements	the Object array into which the elements
 *		 					of this Vector are copied
 *
 * @see			#clone
 */
public synchronized void copyInto (Object [] elements) {
	System.arraycopy (elementData, 0, elements, 0, elementCount);
}

/**
 * Answers the element at the specified location in this Vector.
 *
 * @param		location	the index of the element to return in this Vector
 * @return		the element at the specified location
 *
 * @exception	ArrayIndexOutOfBoundsException when <code>location < 0 || >= size()</code>
 *
 * @see			#size
 */
public synchronized Object elementAt (int location) {
	if (location < elementCount) {
		return elementData[location];
	}
	throw new ArrayIndexOutOfBoundsException(location);
}

/**
 * Answers an Enumeration on the elements of this Vector. The
 * results of the Enumeration may be affected if the contents
 * of this Vector are modified.
 *
 * @return		an Enumeration of the elements of this Vector
 *
 * @see			#elementAt
 * @see			Enumeration
 */
public Enumeration elements () {
	return new Enumeration() {
		int pos = 0;
		public boolean hasMoreElements() {
				return pos < elementCount;
		}
		public Object nextElement() {
			synchronized(Vector.this) {
				if (pos < elementCount) {
					return elementData[pos++];
				}
			}
			throw new NoSuchElementException();
		}
	};
}

/**
 * Ensures that this Vector can hold the specified number of elements
 * without growing.
 *
 * @param		minimumCapacity	 the minimum number of elements that this
 *					vector will hold before growing
 *
 * @see			#capacity
 */
public synchronized void ensureCapacity(int minimumCapacity) {
	if (elementData.length < minimumCapacity) {
		int next = (capacityIncrement <= 0 ?
			elementData.length : capacityIncrement) + elementData.length;
		grow(minimumCapacity > next ? minimumCapacity : next);
	}
}

/**
 * Answers the first element in this Vector.
 *
 * @return		the element at the first position
 *
 * @exception	NoSuchElementException	when this vector is empty
 *
 * @see			#elementAt
 * @see			#lastElement
 * @see			#size
 */
public synchronized Object firstElement () {
	if (elementCount > 0) {
		return elementData[0];
	}
	throw new NoSuchElementException();
}

private void grow(int newCapacity) {
	Object[] newData = new Object[newCapacity];
	System.arraycopy(elementData, 0, newData, 0, elementCount); // assumes elementCount is <= newCapacity
	elementData = newData;
}

/**
 * jit optimization
 */
private void growByOne() {
	int adding = 0;
	if (capacityIncrement <= 0) {
		if ((adding = elementData.length) == 0) adding = 1;
	} else
		adding = capacityIncrement;

	Object[] newData = new Object[elementData.length + adding];
	System.arraycopy(elementData, 0, newData, 0, elementCount);
	elementData = newData;
}

/**
 * Searches in this Vector for the index of the specified object. The
 * search for the object starts at the beginning and moves towards the
 * end of this Vector.
 *
 * @param		object	the object to find in this Vector
 * @return		the index in this Vector of the specified element, -1 if the
 *				element isn't found
 *
 * @see			#contains
 * @see			#lastIndexOf(Object)
 * @see			#lastIndexOf(Object, int)
 */
public int indexOf (Object object) {
	return indexOf(object, 0);
}

/**
 * Searches in this Vector for the index of the specified object. The
 * search for the object starts at the specified location and moves
 * towards the end of this Vector.
 *
 * @param		object	the object to find in this Vector
 * @param		location	the index at which to start searching
 * @return		the index in this Vector of the specified element, -1 if the
 *				element isn't found
 *
 * @exception	ArrayIndexOutOfBoundsException when <code>location < 0</code>
 *
 * @see			#contains
 * @see			#lastIndexOf(Object)
 * @see			#lastIndexOf(Object, int)
 */
public synchronized int indexOf (Object object, int location) {
	if (object != null) {
		for (int i=location; i<elementCount; i++)
			if (object.equals(elementData[i])) return i;
	} else {
		for (int i=location; i<elementCount; i++)
			if (elementData[i] == null) return i;
	}
	return -1;
}

/**
 * Inserts the specified object into this Vector at the specified
 * location. This object is inserted before any previous element at
 * the specified location. If the location is equal to the size of
 * this Vector, the object is added at the end.
 *
 * @param		object	the object to insert in this Vector
 * @param		location	the index at which to insert the element
 *
 * @exception	ArrayIndexOutOfBoundsException when <code>location < 0 || > size()</code>
 *
 * @see			#addElement
 * @see			#size
 */
public synchronized void insertElementAt (Object object, int location) {
	if (0 <= location && location <= elementCount) {
		if (elementCount == elementData.length) growByOne();
		int count = elementCount - location;
		if (count > 0)
			System.arraycopy (elementData, location, elementData, location + 1, count);
		elementData [location] = object;
		elementCount++;
	} else throw new ArrayIndexOutOfBoundsException(location);
}

/**
 * Answers if this Vector has no elements, a size of zero.
 *
 * @return		true if this Vector has no elements, false otherwise
 *
 * @see			#size
 */
public synchronized boolean isEmpty() {
	return elementCount == 0;
}

/**
 * Answers the last element in this Vector.
 *
 * @return		the element at the last position
 *
 * @exception	NoSuchElementException	when this vector is empty
 *
 * @see			#elementAt
 * @see			#firstElement
 * @see			#size
 */
public synchronized Object lastElement () {
	try {
		return elementData[elementCount - 1];
	} catch (IndexOutOfBoundsException e) {
		throw new NoSuchElementException();
	}
}

/**
 * Searches in this Vector for the index of the specified object. The
 * search for the object starts at the end and moves towards the start
 * of this Vector.
 *
 * @param		object	the object to find in this Vector
 * @return		the index in this Vector of the specified element, -1 if the
 *				element isn't found
 *
 * @see			#contains
 * @see			#indexOf(Object)
 * @see			#indexOf(Object, int)
 */
public synchronized int lastIndexOf (Object object) {
	return lastIndexOf(object, elementCount - 1);
}

/**
 * Searches in this Vector for the index of the specified object. The
 * search for the object starts at the specified location and moves
 * towards the start of this Vector.
 *
 * @param		object	the object to find in this Vector
 * @param		location	the index at which to start searching
 * @return		the index in this Vector of the specified element, -1 if the
 *				element isn't found
 *
 * @exception	ArrayIndexOutOfBoundsException when <code>location >= size()</code>
 *
 * @see			#contains
 * @see			#indexOf(Object)
 * @see			#indexOf(Object, int)
 */
public synchronized int lastIndexOf (Object object, int location) {
	if (location < elementCount) {
		if (object != null) {
			for (int i=location; i >= 0; i--)
				if (object.equals(elementData[i])) return i;
		} else {
			for (int i=location; i >= 0; i--)
				if (elementData[i] == null) return i;
		}
		return -1;
	} else throw new ArrayIndexOutOfBoundsException(location);
}

/**
 * Removes all elements from this Vector, leaving the size zero and
 * the capacity unchanged.
 *
 * @see			#isEmpty
 * @see			#size
 */
public synchronized void removeAllElements() {
	for (int i=0; i<elementCount; i++) elementData[i] = null;
	elementCount = 0;
}
/**
 * Removes the first occurrence, starting at the beginning and
 * moving towards the end, of the specified object from this
 * Vector.
 *
 * @param		object	the object to remove from this Vector
 * @return		true if the specified object was found, false otherwise
 *
 * @see			#removeAllElements
 * @see			#removeElementAt
 * @see			#size
 */
public synchronized boolean removeElement (Object object) {
	int index;
	if ((index = indexOf (object, 0)) == -1) return false;
	removeElementAt (index);
	return true;
}

/**
 * Removes the element at the specified location from this Vector.
 *
 * @param		location	the index of the element to remove
 *
 * @exception	ArrayIndexOutOfBoundsException when <code>location < 0 || >= size()</code>
 *
 * @see			#removeElement
 * @see			#removeAllElements
 * @see			#size
 */
public synchronized void removeElementAt (int location) {
	if (0 <= location && location < elementCount) {
		elementCount--;
		int size = elementCount - location;
		if (size > 0)
			System.arraycopy (elementData, location + 1, elementData, location, size);
		elementData [elementCount] = null;
	} else throw new ArrayIndexOutOfBoundsException(location);
}

/**
 * Replaces the element at the specified location in this Vector with
 * the specified object.
 *
 * @param		object	the object to add to this Vector
 * @param		location	the index at which to put the specified object
 *
 * @exception	ArrayIndexOutOfBoundsException when <code>location < 0 || >= size()</code>
 *
 * @see			#size
 */
public synchronized void setElementAt (Object object, int location) {
	if (location < elementCount) elementData[location] = object;
	else throw new ArrayIndexOutOfBoundsException(location);
}

/**
 * Sets the size of this Vector to the specified size. If there
 * are more than length elements in this Vector, the elements
 * at end are lost. If there are less than length elements in
 * the Vector, the additional elements contain null.
 *
 * @param		length	the new size of this Vector
 *
 * @see			#size
 */
public synchronized void setSize(int length) {
	if (length == elementCount) return;
	ensureCapacity(length);
	if (elementCount > length) {
		for (int i=length; i<elementCount; i++) elementData[i] = null;
	}
	elementCount = length;
}

/**
 * Answers the number of elements in this Vector.
 *
 * @return		the number of elements in this Vector
 *
 * @see			#elementCount
 * @see			#lastElement
 */
public synchronized int size () {
	return elementCount;
}

/**
 * Answers the string representation of this Vector.
 *
 * @return		the string representation of this Vector
 *
 * @see			#elements
 */
public synchronized String toString () {
	if (elementCount == 0) return "[]";
	int length = elementCount - 1;
	StringBuffer buffer = new StringBuffer(size() * 16);
	buffer.append ('[');
	for (int i=0; i<length; i++) {
		if (elementData[i] == this) {
			buffer.append("(this Collection)");
		} else {
			buffer.append (elementData [i]);
		}
		buffer.append (", ");
	}
	if (elementData[length] == this) {
		buffer.append("(this Collection)");
	} else {
		buffer.append (elementData [length]);
	}
	buffer.append (']');
	return buffer.toString ();
}

/**
 * Sets the capacity of this Vector to be the same as the size.
 *
 * @see			#capacity
 * @see			#ensureCapacity
 * @see			#size
 */
public synchronized void trimToSize() {
	if (elementData.length != elementCount)
		grow(elementCount);
}

}
