package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

/*
 * Handles all animation tasks for a given Displayable.
 * These tasks might be:
 *
 * - CONTINUOUS_RUNNING Gauge
 * - Ticker
 *
 * Runnables are executed periodically, so they should
 * not block.  The period is not configurable.
 *
 * There should be only one Animator per MIDlet.
 */
final class Animator implements Runnable {

	private static final int TIMEOUT = 200;

	private Runnable[] fRunnables = new Runnable[10];
	private int fRunnablesLength = 0;

	private boolean fRunning;
	private Thread fThread;
	private Object fSyncObject;

	static Animator[] gAnimators = new Animator[10];
	static Display[] gDisplays = new Display[gAnimators.length];
	static int gDisplaysLength = 0;

	private Animator() {
		fSyncObject = this;
	}

	/*
	 * This method should never be called outside the Thread class.
	 */
	public void run() {
		int size;

		waitForRunnables();

		while (fRunning) {

			/*
			 * Exit if there are no more runnables.  Another Thread
			 * will be created when another Runnable is added.
			 */
			if (fRunnablesLength == 0) {
				fRunning = false;
				break;
			}

			synchronized (fSyncObject) {
				size = fRunnablesLength;
				for (int i = 0; i < size; i++) {
					fRunnables[i].run();
				}
			}

			try {
				Thread.sleep(TIMEOUT);
			} catch (InterruptedException e) {
			}
		}
	}

	private void start() {
		if (!fRunning) {
			fThread = new Thread(this);
			fRunning = true;
			fThread.start();
		}
	}

	private void stop() {
		synchronized (fSyncObject) {
			fRunning = false;
			fThread = null;
			fRunnables = null;
			fRunnablesLength = 0;
			fSyncObject.notifyAll();
		}
	}

	private void waitForRunnables() {
		synchronized (fSyncObject) {
			while (fRunnables == null || fRunnablesLength == 0) {
				try {
					fSyncObject.wait();
				} catch (InterruptedException e) {}
			}
		}
	}

	/*
	 * Add a new Runnable to be executed.  Start the thread if
	 * r is the first Runnable to be added.
	 *
	 * Runnables which are added MUST NOT block in their run()
	 * methods.
	 */
	void add(Runnable r) {
		boolean start = false;

		if (fRunnablesLength == 0) {
			// first Runnable is being added, so start
			start = true;
		}

		synchronized (fSyncObject) {
			if (fRunnablesLength == fRunnables.length) {
				Runnable[] newArray = new Runnable[fRunnables.length * 2];
				System.arraycopy(fRunnables, 0, newArray, 0, fRunnables.length);
				fRunnables = newArray;
			}
			fRunnables[fRunnablesLength] = r;
			fRunnablesLength++;
			fSyncObject.notifyAll();
		}

		// do not start until there is actually a Runnable to execute
		if (start && !fRunning) start();
	}

	void remove(Runnable r) {
		if (fRunnablesLength == 0) return;

		synchronized (fSyncObject) {
			for (int i = 0; i < fRunnablesLength; i++) {
				if (fRunnables[i] == r) {
					fRunnablesLength--;
					for (int j = i; j < fRunnablesLength; j++) {
						fRunnables[j] = fRunnables[j + 1];
					}
					return;
				}
			}
		}
	}

	void shutdown(Display display) {
		stop();

		for (int i = 0; i < gDisplaysLength; i++) {
			if (gDisplays[i] == display) {
				gDisplaysLength--;
				for (int j = i; j < gDisplaysLength; j++) {
					gDisplays[j] = gDisplays[j + 1];
					gAnimators[j] = gAnimators[j + 1];
				}
				return;
			}
		}
	}

	/*
	 * Return the Animator which is associated with
	 * a Display.  If one does not exist, create it.
	 * Mapping Animators to Displays ensures one
	 * Animator per MIDlet.
	 */
	static Animator getAnimator(Display display) {

		for (int i = 0; i < gDisplaysLength; i++) {
			if (gDisplays[i] == display) {
				return gAnimators[i];
			}
		}

		Animator newAnimator = new Animator();

		if (gDisplaysLength == gDisplays.length) {
			Display[] displays = new Display[gDisplays.length * 2];
			System.arraycopy(gDisplays, 0, displays, 0, gDisplays.length);
			gDisplays = displays;

			Animator[] animators = new Animator[gAnimators.length * 2];
			System.arraycopy(gAnimators, 0, animators, 0, gAnimators.length);
			gAnimators = animators;
		}

		gDisplays[gDisplaysLength] = display;
		gAnimators[gDisplaysLength] = newAnimator;
		gDisplaysLength++;
		return newAnimator;
	}
}
