package javax.microedition.lcdui;

import com.ibm.ive.midp.*;
import javax.microedition.lcdui.game.GameCanvas;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

public abstract class Canvas extends Displayable {

	/*
	 * Public constants defined in the MIDP spec.
	 */
	public static final int DOWN      = 6;
	public static final int FIRE      = 8;
	public static final int GAME_A    = 9;
	public static final int GAME_B    = 10;
	public static final int GAME_C    = 11;
	public static final int GAME_D    = 12;
	public static final int LEFT      = 2;
	public static final int RIGHT     = 5;
	public static final int UP        = 1;

	public static final int KEY_NUM0  = 48;
	public static final int KEY_NUM1  = 49;
	public static final int KEY_NUM2  = 50;
	public static final int KEY_NUM3  = 51;
	public static final int KEY_NUM4  = 52;
	public static final int KEY_NUM5  = 53;
	public static final int KEY_NUM6  = 54;
	public static final int KEY_NUM7  = 55;
	public static final int KEY_NUM8  = 56;
	public static final int KEY_NUM9  = 57;
	public static final int KEY_POUND = 35;
	public static final int KEY_STAR  = 42;

	/*
	 * Constants beginning with KEYCODE are the values
	 * which are actually passed to the keyPressed(),
	 * keyRepeated(), and keyReleased() methods when
	 * key events occur.
	 *
	 * The keys which have obvious Unicode meanings
	 * are positive integers which can be cast to
	 * char values.  The keys which do not have
	 * obvious Unicode meanings have negative values.
	 */

	static final int KEYCODE_A     = 'A';
	static final int KEYCODE_B     = 'B';
	static final int KEYCODE_C     = 'C';
	static final int KEYCODE_D     = 'D';
	static final int KEYCODE_UP    = -1;
	static final int KEYCODE_DOWN  = -2;
	static final int KEYCODE_LEFT  = -3;
	static final int KEYCODE_RIGHT = -4;
	static final int KEYCODE_FIRE  = -5;
	static final int KEYCODE_INVALID = 0;

	static final int KEYCODE_TAB = 9;
	static final int KEYCODE_BACKSPACE = 8;
	static final int KEYCODE_DELETE = 127;
	static final int KEYCODE_NULL = 0;
	static final int KEYCODE_TALK = -10;
	static final int KEYCODE_UNKNOWN = -11;

	private boolean fFullScreen = false;

	CanvasMediaAccessor fMediaAccessor;

	protected Canvas() {
		super(null);

		fMediaAccessor = new CanvasMediaAccessor(this);
	}

	public int getGameAction(int keycode) {
		return CanvasPeer.getGameAction(keycode);
	}

	public int getKeyCode(int gameAction) {
		/*
		 * Specific to the game mapping in use
		 * by the peer (either default or keypad).
		 */
		return CanvasPeer.getKeyCode(gameAction);
	}

	public String getKeyName(int keycode) {
		switch (keycode) {

			case 'a':			return "a"; //$NON-NLS-1$
			case KEYCODE_A:		return "A"; //$NON-NLS-1$

			case 'b':			return "b"; //$NON-NLS-1$
			case KEYCODE_B:		return "B"; //$NON-NLS-1$

			case 'c':			return "c"; //$NON-NLS-1$
			case KEYCODE_C:		return "C"; //$NON-NLS-1$

			case 'd':			return "d"; //$NON-NLS-1$
			case KEYCODE_D:		return "D"; //$NON-NLS-1$

			case KEYCODE_LEFT:	return MidpMsg.getString("Canvas.getKeyName.Left"); //$NON-NLS-1$
			case KEYCODE_UP:	return MidpMsg.getString("Canvas.getKeyName.Up"); //$NON-NLS-1$
			case KEYCODE_RIGHT:	return MidpMsg.getString("Canvas.getKeyName.Right"); //$NON-NLS-1$
			case KEYCODE_DOWN:	return MidpMsg.getString("Canvas.getKeyName.Down"); //$NON-NLS-1$

			case KEYCODE_FIRE:	return MidpMsg.getString("Canvas.getKeyName.Fire"); //$NON-NLS-1$

			case KEY_NUM0:		return "0"; //$NON-NLS-1$
			case KEY_NUM1:		return "1"; //$NON-NLS-1$
			case KEY_NUM2:		return "2"; //$NON-NLS-1$
			case KEY_NUM3:		return "3"; //$NON-NLS-1$
			case KEY_NUM4:		return "4"; //$NON-NLS-1$
			case KEY_NUM5:		return "5"; //$NON-NLS-1$
			case KEY_NUM6:		return "6"; //$NON-NLS-1$
			case KEY_NUM7:		return "7"; //$NON-NLS-1$
			case KEY_NUM8:		return "8"; //$NON-NLS-1$
			case KEY_NUM9:		return "9"; //$NON-NLS-1$

			case KEY_STAR:	return "*"; //$NON-NLS-1$

			case KEY_POUND:	return "#"; //$NON-NLS-1$

			default:
				String keyName = Device.getKeyName(keycode);
				if (keyName != null) return keyName;
				throw new IllegalArgumentException();
		}
	}

	public int getHeight() {
		synchronized (Device.gDisplayableLock) {
			/*
			 * 1.0 MIDlets need to have a constant value returned from
			 * getHeight().  This is because sizeChanged() is new in 2.0.
			 */

			/*
			 * There is a small window of time between creation of CanvasPeer
			 * and CanvasContentComponent. Hence need to check for null for
			 * both the cases.
			 */
			if (fPeer != null && fPeer.fContentComponent != null) return fPeer.getHeight();
			return CanvasPeer.getDisplayableHeight(fTitle, fTicker, fCommands, fFullScreen);
		}
	}

	public int getWidth() {
		synchronized (Device.gDisplayableLock) {
			/*
			 * There is a small window of time between creation of CanvasPeer
			 * and CanvasContentComponent. Hence need to check for null for
			 * both the cases.
			 */
			if (fPeer != null && fPeer.fContentComponent != null) return fPeer.getWidth();
			return CanvasPeer.getDisplayableWidth(fTitle, fTicker, fCommands, fFullScreen);
		}
	}

	public boolean hasPointerEvents() {
		return Device.hasPointerEvents();
	}

	public boolean hasPointerMotionEvents() {
		return Device.hasPointerMotionEvents();
	}

	public boolean hasRepeatEvents() {
		return Device.hasKeyRepeatEvents();
	}

	protected void hideNotify() {
	}

	public boolean isDoubleBuffered() {
		return true;
	}

	protected void keyPressed(int key) {
	}

	protected void keyReleased(int key) {
	}

	protected void keyRepeated(int key) {
	}

	protected abstract void paint(Graphics g);

	protected void pointerDragged(int x, int y) {
	}

	protected void pointerPressed(int x, int y) {
	}

	protected void pointerReleased(int x, int y) {
	}

	public final void repaint() {
		synchronized (Device.gDisplayableLock) {
			if (fPeer != null && fPeer.fContentComponent != null) fPeer.fContentComponent.repaint();
		}
	}

	public final void repaint(final int x, final int y, final int width, final int height) {
		synchronized (Device.gDisplayableLock) {
			if (fPeer != null && fPeer.fContentComponent != null) fPeer.fContentComponent.repaint(x, y, width, height);
		}
	}

	public final void serviceRepaints() {
		Runnable repaintRunnable;
		synchronized (Device.gDisplayableLock) {
			CanvasPeer peer = (CanvasPeer)fPeer;
			if (peer == null ||
				peer.fContentComponent == null ||
				peer.fContentComponent.fRepaintRunnable == null ||
				!peer.fContentComponent.fDirty ||
				Device.gCanvasFlushRunnable == null) {
				return;
			}
			repaintRunnable = peer.fContentComponent.fRepaintRunnable;
		}

		try {
			repaintRunnable.run();
		} catch (Exception e) {
			e.printStackTrace();
		}
		Device.flushDisplay(System.currentTimeMillis());

		/*
		 * Some midlets will run continually calling repaint() and serviceRepaints().
		 * In order to allow other threads to run, we need to give them some
		 * time. It may be preferrable to yield only after a number of iterations.
		 * Given the time taken for a paint and a flush though, the yield() hit
		 * may be unnoticeable.
		 */
		Thread.yield();
	}

	public void setFullScreenMode(boolean fullScreen) {
		synchronized (Device.gDisplayableLock) {
			fFullScreen = fullScreen;
			if (fPeer != null) {
				((CanvasPeer)fPeer).setFullScreenMode(fullScreen);
				repaint();
			}
		}
	}

	boolean isFullScreen() {
		return fFullScreen;
	}

	protected void showNotify(){
	}

	protected void sizeChanged(int width, int height) {
	}

	int getDisplayableType() {
		return TYPE_CANVAS;
	}

	/*
	 * Returns true if gameCanvas has game key suppression on.
	 */
	native boolean isKeySuppressed(GameCanvas gameCanvas);

	/*
	 * Notify gameCanvas of a keyPressed event.
	 * Returns true if keycode was a game key, false otherwise.
	 */
	native boolean callHiddenKeyDown(GameCanvas gameCanvas, int keycode);

	/*
	 * Notify gameCanvas that we're about to call showNotify.
	 */
	native void callHiddenShowNotify(GameCanvas gameCanvas);

	/*
	 * Notify gameCanvas that we're about to call hideNotify.
	 */
	native void callHiddenHideNotify(GameCanvas gameCanvas);
}
