package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;

class CanvasComponent extends WindowContent {

	Canvas fCanvas;
	CanvasPeer fCanvasPeer;
	Graphics fGraphics;

	public CanvasComponent(CanvasPeer peer) {
		super(peer);
		fCanvas = (Canvas) peer.fDisplayable;
		fCanvasPeer = peer;
		fGraphics = new Graphics(this);
	}

	boolean getShowsFocus() {
		return false;
	}

	void paint(Graphics g) {
		if (Device.DEBUG_SERIAL_QUEUE) {
//			if (Thread.currentThread() != Device.gSerialThread) System.err.println("Paint outside serial thread");
		}

		g.setFont(Font.STANDARD_FONT);
		g.setColor(0);
		try {
			((Canvas) getDisplayable()).paint(g);
		} catch (Exception e) {
			e.printStackTrace();
		}
		if (g.fData.fStrokeStyle != Graphics.SOLID) g.setStrokeStyle(Graphics.SOLID);
	}

	boolean pointerDragged(int x, int y) {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}

		try {
			fCanvas.pointerDragged(x, y);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean pointerPressed(int x, int y) {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}
		try {
			fCanvas.pointerPressed(x, y);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean pointerReleased(int x, int y) {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}

		try {
			fCanvas.pointerReleased(x, y);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean traverse(int direction, int viewWidth, int viewHeight, int[] visibleRectangle, int x, int y) {
		return true;
	}

	boolean isScrollable() {
		return false;
	}

	void traverseOut() {
		/* No-op. The canvas is the one component that should not repaint on traverseOut(). */
	}

	boolean keyPressed(int keyCode) {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}

		if (fCanvasPeer.fIsGameCanvas && fCanvasPeer.isGameKey(keyCode)) {
			fCanvas.callHiddenKeyDown(fCanvasPeer.fGameCanvas, keyCode);
			if (fCanvasPeer.fSuppressGameKeys) return true;
		}

		try {
			fCanvas.keyPressed(keyCode);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean keyReleased(int keyCode) {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}

		if (fCanvasPeer.fIsGameCanvas && fCanvasPeer.fSuppressGameKeys && fCanvasPeer.isGameKey(keyCode)) {
			return true;
		}

		try {
			fCanvas.keyReleased(keyCode);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	boolean keyRepeated(int keyCode) {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}

		if (fCanvasPeer.fIsGameCanvas && fCanvasPeer.isGameKey(keyCode)) {
			fCanvas.callHiddenKeyDown(fCanvasPeer.fGameCanvas, keyCode);
			if (fCanvasPeer.fSuppressGameKeys) return true;
		}
		try {
			fCanvas.keyRepeated(keyCode);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return true;
	}

	Runnable getRepaintRunnable() {
		return new RepaintRunnable(this);
	}

	class RepaintRunnable implements Runnable {
		Component fComponent;

		RepaintRunnable(Component c) {
			fComponent = c;
		}

		public void run() {
			/*
			 * We can't sync on the gDisplayableLock here (or any lock that could be
			 * accessed as a result of the public API) because we'll be calling
			 * user code. If we call user code while holding the gDisplayableLock,
			 * we can deadlock on the user's monitors.
			 */
			synchronized (Device.gRepaintLock) {
				int x, y, w, h;
				synchronized (fRepaintLock) {
					// make sure that they are in a good state
					x = fDirtyX;
					y = fDirtyY;
					w = fDirtyW;
					h = fDirtyH;
					fDirty = false;
				}

				/*
				 * It's important to do this check after clearing the fDirty
				 * flag. Otherwise, the Canvas will be unable to schedule
				 * another repaint runnable.
				 */
				if (!isShown() || Device.gCanvasFlushRunnable == null) return;

				Graphics g = getGraphics();
				g.activate();
				setOriginAndClip(g, x, y, w, h);
				paint(g);
				Device.gCanvasFlushNeeded = true;
			}
		}
	}

	Graphics getGraphics() {
		return fGraphics;
	}

	void dispose() {
		super.dispose();
		if (fGraphics != null) {
			fGraphics.dispose();
			fGraphics = null;
		}
	}
}
