package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;
import com.ibm.ive.midp.util.*;
import javax.microedition.lcdui.game.GameCanvas;

class CanvasPeer extends DisplayablePeer {

	boolean fFullScreen;
	boolean fSuppressGameKeys;
	GameCanvas fGameCanvas;
	boolean fIsGameCanvas = false;

	public CanvasPeer(Display display, Canvas canvas) {
		super(display, canvas);
		fFullScreen = canvas.isFullScreen();

		/*
		 * If Canvas is really a GameCanvas, we may need to suppress
		 * game key events.
		 */
		fIsGameCanvas = false;
		fSuppressGameKeys = false;

		if (canvas instanceof GameCanvas) {
			/*
			 * GameCanvas has its own double buffer.
			 */
			fIsGameCanvas = true;
			fGameCanvas = (GameCanvas)canvas;
			fSuppressGameKeys = canvas.isKeySuppressed(fGameCanvas);
			/*
			 * Reset clip restriction if fDisplayGraphics exist to account for
			 * screen rotation if it has happened.
			 */
			Graphics g = getDisplayGraphics((GameCanvas)fDisplayable);
			if (g != null) g.resetClipRestriction();
		}
	}

	void createScrollBar() {
		/* No - op for Canvas. */
	}

	boolean isGameKey(int keycode) {
		try {
			if (getGameAction(keycode) != 0) return true;
		} catch (IllegalArgumentException e) {}

		return false;
	}

	void showNotify() {
		((Canvas) fDisplayable).fMediaAccessor.showMedia(
			fContentComponent.fX,
			fContentComponent.fY,
			fContentComponent.fWidth,
			fContentComponent.fHeight);

		if (fIsGameCanvas) ((Canvas) fDisplayable).callHiddenShowNotify((GameCanvas) fDisplayable);

		super.showNotify();
	}

	void hideNotify() {
		((Canvas) fDisplayable).fMediaAccessor.hideMedia();

		if (fIsGameCanvas) ((Canvas) fDisplayable).callHiddenHideNotify((GameCanvas) fDisplayable);

		super.hideNotify();
	}

	static int getKeyCode(int gameAction) {
		switch (gameAction) {
			case Canvas.UP: return Canvas.KEYCODE_UP;
			case Canvas.DOWN: return Canvas.KEYCODE_DOWN;
			case Canvas.LEFT: return Canvas.KEYCODE_LEFT;
			case Canvas.RIGHT: return Canvas.KEYCODE_RIGHT;
			case Canvas.FIRE: return Canvas.KEYCODE_FIRE;
			case Canvas.GAME_A: return Canvas.KEYCODE_A;
			case Canvas.GAME_B: return Canvas.KEYCODE_B;
			case Canvas.GAME_C: return Canvas.KEYCODE_C;
			case Canvas.GAME_D: return Canvas.KEYCODE_D;
			default:
				throw new IllegalArgumentException();
		}
	}

	static int getGameAction(int keyCode) {
		/* Positive keyCodes can be cast to unicode characters.
		 * Negative keyCodes are unprintable characters that don't have a unicode mapping. Return Zero if they are not game actions.
		 * 0 is invalid.
		 */
		switch (keyCode) {
			case Canvas.KEYCODE_UP: return Canvas.UP;
			case Canvas.KEYCODE_DOWN: return Canvas.DOWN;
			case Canvas.KEYCODE_LEFT: return Canvas.LEFT;
			case Canvas.KEYCODE_RIGHT: return Canvas.RIGHT;
			case Canvas.KEYCODE_FIRE:
			case '\n':
			case ' ':
				return Canvas.FIRE;
			case Canvas.KEYCODE_A:
			case 'a':
				return Canvas.GAME_A;
			case Canvas.KEYCODE_B:
			case 'b':
				return Canvas.GAME_B;
			case Canvas.KEYCODE_C:
			case 'c':
				return Canvas.GAME_C;
			case Canvas.KEYCODE_D:
			case 'd':
				return Canvas.GAME_D;
			/* zero */
			case Canvas.KEYCODE_INVALID: throw new IllegalArgumentException();

			default: return 0;
		}
	}

	void setInitialFocus() {
		fFocusComponent = fContentComponent;
	}

	void setFullScreenMode(boolean fullScreen) {
		if (fullScreen == fFullScreen) return;
		fFullScreen = fullScreen;

		fTitleComponent.update();
		fTickerComponent.update();

		layout();
		repaint(); /* DisplayablePeer#repaint() repaints the title, the ticker, and the commands */
	}

	int getHeight() {
		return fContentComponent.fHeight;
	}

	void createDisplayablePeerOnly() {
		super.create();
	}

	void disposeDisplayablePeerOnly() {
		super.dispose();
	}

	void dispose() {
		Device.gCanvasFlushNeeded = false;

		super.dispose();
	}

	void createContents() {
		fContentComponent = new CanvasComponent(this);
		Device.gCanvasFlushRunnable = new Runnable() {
			public void run() {
				synchronized (Device.gDisplayableLock) {
					if (!isShown()) return;
					fGraphics.flush(0, 0, fWidth, fHeight);
				}
			}
		};
	}

	public void resize(int newWidth, int newHeight) {
		if (newWidth <= 0 || newHeight <= 0) return;
		if (newWidth == fWidth && newHeight == fHeight) return;

		synchronized (Device.gRepaintLock) {
			fWidth = newWidth;
			fHeight = newHeight;
			if (!isShown()) return;
			layout();

			if (fGraphics != null) fGraphics.resetClipRestriction();
			if (fIsGameCanvas) {
				Graphics g = getDisplayGraphics((GameCanvas)fDisplayable);
				if (g != null) g.resetClipRestriction();
			}
			repaint(); /* title and commands */
			((Canvas)fDisplayable).repaint(); /* Canvas contents */
		}
	}

	/*
	 * This method returns Graphics object (fDisplayGraphics member) from
	 * GameCanvas object.
	 */
	native Graphics getDisplayGraphics(GameCanvas canvas);

	static int getDisplayableHeight(String title, Ticker ticker, FastVector commands, boolean fullScreen) {
		int totalHeight = DisplayablePeer.getDisplayableHeight();
		if (fullScreen) return totalHeight;

		int titleHeight = title != null ? TitleComponent.TITLE_MIN_HEIGHT : 0;
		int tickerHeight = ticker != null ? TickerPeer.TICKER_MIN_HEIGHT : 0;

		totalHeight -= Math.max(titleHeight, tickerHeight);

		if (MIDletManager.isMidp10()) totalHeight -= CommandManager.getFixedDisplayHeight();
		return totalHeight;
	}

	static int getDisplayableWidth(String title, Ticker ticker, FastVector commands, boolean fullScreen) {
		return Device.getDisplayWidth();
	}

	void dispatchCommand(Command c) {
		super.dispatchCommand(c);
		/*
		 * The command dispatched could dispose this displayable.
		 * Check if we're still shown before handling command traversal.
		 */
		if (isShown()) setFocus(fContentComponent);
	}
}
