package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import javax.microedition.lcdui.game.Sprite;
import com.ibm.ive.midp.*;

class ChoiceElement extends ButtonWidget {

	ChoiceImpl fChoice;
	Image fImage;
	Font fFont;
	int fIndex;
	int fLineHeight;
	TextComponent fTextComponent;

	static final int INDENT = 15;
	static final int SELECTION_INDICATOR_WIDTH = 10;
	static final int SELECTION_INDICATOR_HEIGHT = 10;
	static final int MARGIN_WITH_INDICATOR = ChoiceComponent.TOTAL_H_MARGIN + SELECTION_INDICATOR_WIDTH;

	static final int IMAGE_WIDTH = Device.getIconWidth();
	static final int IMAGE_HEIGHT = Device.getIconHeight();
	static final int IMAGE_HALF_WIDTH = IMAGE_WIDTH / 2;
	static final int IMAGE_HALF_HEIGHT = IMAGE_HEIGHT / 2;

	static final int V_SPACING = 1;
	static final int TOTAL_V_SPACING = 2;

	ChoiceElement(Composite parent, ChoiceImpl choice, int index, int width) {
		super(parent);
		fChoice = choice;
		fIndex = index;
		fButtonListener = (IButtonListener) parent;
		fImage = (Image) fChoice.fImageSnapshots.elementAt(index);
		fFont = fChoice.getFont(index);
		fTextComponent = new TextComponent(Item.TYPE_CHOICEGROUP,fFont);
		fLineHeight = fFont._getHeight();
		fWidth = width;
		setText((String) fChoice.fStrings.elementAt(index), width, choice.fFitPolicy);
	}

	int _getPreferredWidth() {
		int width = ChoiceComponent.TOTAL_H_MARGIN;

		if (fChoice.fType != Choice.IMPLICIT && fChoice.fType != Choice.POPUP) {
			width += SELECTION_INDICATOR_WIDTH + ChoiceComponent.H_MARGIN;
		}
		if (fImage != null) width += ChoiceComponent.IMAGE_MARGIN_WIDTH + IMAGE_WIDTH;
		if (fText != null) width += ChoiceComponent.H_MARGIN + fFont.getStringWidth(fText);

		return width;
	}

	void setText(String text, int width, int fitPolicy) {
		fText = text;

		if (text == null) return;
		if (text.length() == 0) {
			fTextComponent.fFormattedTextLength = 1;
			fTextComponent.fFormattedText[0] = text;
			fHeight = fLineHeight + TOTAL_V_SPACING;
			return;
		}

		if (fImage != null) width -= ChoiceComponent.IMAGE_MARGIN_WIDTH;

		// ChoiceGroups with indicators provide space for the indicator and don't indent subsequent lines
		boolean hasIndicator = fChoice.fType != Choice.IMPLICIT && fChoice.fType != Choice.POPUP;

		if (hasIndicator) width -= MARGIN_WITH_INDICATOR;

		// for MIDP 2.0 check the fitPolicy
		if (fitPolicy == Choice.TEXT_WRAP_OFF) {
			fTextComponent.fFormattedTextLength = 1;
			fTextComponent.fFormattedText[0] = format(stripNewlines(text), width, fFont);
			fHeight = fLineHeight + TOTAL_V_SPACING;
		} else {
			fTextComponent.setText(text, false);
			fTextComponent.parse(fWidth, width, -1, hasIndicator);
			fHeight = (fTextComponent.fFormattedTextLength * fLineHeight) + TOTAL_V_SPACING;
		}

		if (fImage != null) fHeight = Math.max(fHeight, Math.min(IMAGE_HEIGHT, fImage._getHeight()) + TOTAL_V_SPACING);
	}

	int drawSelection(Graphics g, int x, int y, boolean selected, int type) {
		if (type == Choice.IMPLICIT) return x;

		if (type == Choice.POPUP) return x;

		int selectionOffset = y + ((fLineHeight - SELECTION_INDICATOR_HEIGHT) / 2);

		switch (type) {
			case Choice.EXCLUSIVE:
				g.setColor(DisplayPeer.COLOR_BACKGROUND_RGB);
				g._fillRoundRect(x + 1, selectionOffset + 1, SELECTION_INDICATOR_WIDTH - 2, SELECTION_INDICATOR_HEIGHT - 2, SELECTION_INDICATOR_WIDTH, SELECTION_INDICATOR_WIDTH);

				g.setColor(DisplayPeer.COLOR_FOREGROUND_RGB);
				g._drawRoundRect(x, selectionOffset, SELECTION_INDICATOR_WIDTH - 1, SELECTION_INDICATOR_HEIGHT - 1, SELECTION_INDICATOR_WIDTH, SELECTION_INDICATOR_WIDTH);

				if (selected) {
					int arcHeight = SELECTION_INDICATOR_HEIGHT / 2;
					int arcX = x + SELECTION_INDICATOR_WIDTH / 3;
					int arcY = selectionOffset + SELECTION_INDICATOR_HEIGHT / 3;
					g._fillRoundRect(arcX, arcY, arcHeight - 1, arcHeight - 1, arcHeight, arcHeight);
				}

				return x + SELECTION_INDICATOR_WIDTH + ChoiceComponent.H_MARGIN;
			default: /* MULTIPLE */
				g.setColor(DisplayPeer.COLOR_BACKGROUND_RGB);
				g.fillRect(x, selectionOffset, SELECTION_INDICATOR_WIDTH, SELECTION_INDICATOR_HEIGHT);
				g.setColor(DisplayPeer.COLOR_FOREGROUND_RGB);
				g.drawRect(x, selectionOffset, SELECTION_INDICATOR_WIDTH - 1, SELECTION_INDICATOR_HEIGHT - 1);

				if (selected) {
					g.drawLine(x + 2, selectionOffset + 4, x + 4, selectionOffset + 6);
					g.drawLine(x + 2, selectionOffset + 5, x + 4, selectionOffset + 7);
					g.drawLine(x + 2, selectionOffset + 6, x + 4, selectionOffset + 8);

					g.drawLine(x + 4, selectionOffset + 6, x + 8, selectionOffset + 2);
					g.drawLine(x + 4, selectionOffset + 7, x + 8, selectionOffset + 3);
					g.drawLine(x + 4, selectionOffset + 8, x + 8, selectionOffset + 4);
				}
				return x + SELECTION_INDICATOR_WIDTH + ChoiceComponent.H_MARGIN;
		}
	}

	/*
	 * We use centerized cropping to fit the image
	 * that doesn't fit into ICON_WIDTHxICON_HEIGHT
	 * dimension, as suggested by JSR 248(MSA) spec.,
	 * section 6.2.5.5. Though this recommendation is
	 * made for the AMS MIDletList, we make it applicable
	 * for all elements
	 */
	int drawImage(Graphics g, int x, int y) {
		if (fImage == null) return x;

		int clipX = g.getClipX();
		int clipY = g.getClipY();
		int clipWidth = g.getClipWidth();
		int clipHeight = g.getClipHeight();
		int imgWidth = fImage._getWidth();
		int imgHeight = fImage._getHeight();
		int width = Math.min(imgWidth, IMAGE_WIDTH);
		int height = Math.min(imgHeight, IMAGE_HEIGHT);

		y += (fHeight - height) / 2;
		g.setClip(x, y, IMAGE_WIDTH, IMAGE_HEIGHT);

		if (imgWidth <= IMAGE_WIDTH && imgHeight <= IMAGE_HEIGHT) {
			g.drawImage(fImage, x, y, 0);
		} else {
			int xSrc = imgWidth / 2 - IMAGE_HALF_WIDTH;
			if (xSrc < 0) xSrc = 0;

			int ySrc = imgHeight / 2 - IMAGE_HALF_HEIGHT;
			if (ySrc < 0) ySrc = 0;
			g.drawRegion(fImage, xSrc, ySrc, width, height, Sprite.TRANS_NONE, x, y, 0);
		}
		g.setClip(clipX, clipY, clipWidth, clipHeight);
		return x + width;
	}

	void paint(Graphics g) {
		int y = V_SPACING;
		int x = 0;
		int type = fChoice.fType;

		boolean softHighlighted = this == fParent.fFocusComponent && !fParent.hasFocus() && type == Choice.IMPLICIT;
		boolean selected = fChoice.isSelected(fIndex);

		x = drawSelection(g, x, y, selected, type);
		x = drawImage(g, x, y);

		if (softHighlighted) {
			g.setColor(DisplayPeer.COLOR_LIST_SOFT_HIGHLIGHTED_BACKGROUND_RGB);
		} else if (fHighlighted){
			g.setColor(DisplayPeer.COLOR_HIGHLIGHTED_BACKGROUND_RGB);
		} else {
			g.setColor(DisplayPeer.COLOR_BACKGROUND_RGB);
		}
		g.fillRect(x, 0, fWidth - x + ChoiceComponent.TOTAL_H_MARGIN, fHeight); //TODO: width wider than clip area - needs fixing
		if (softHighlighted) {
			g.setColor(DisplayPeer.COLOR_LIST_SOFT_HIGHLIGHTED_FOREGROUND_RGB);
		} else if (fHighlighted) {
			g.setColor(DisplayPeer.COLOR_HIGHLIGHTED_FOREGROUND_RGB);
		} else {
			g.setColor(DisplayPeer.COLOR_FOREGROUND_RGB);
		}

		/*
		 * Leave a margin betweent the beginning of the selection (and/or image) and the first character
		 * of the text.
		 */
		x += ChoiceComponent.H_MARGIN;

		g.setFont(fFont);

		if (type == Choice.IMPLICIT || type == Choice.POPUP) {
			g.drawStringWithTabExpansion(fTextComponent.fFormattedText[0], x, y, 0);
			x += INDENT; // indent subsequent lines
			for (int i = 1; i < fTextComponent.fFormattedTextLength; i++) {
				y += fLineHeight;
				g.drawStringWithTabExpansion(fTextComponent.fFormattedText[i], x, y, 0);
			}
		} else {
			for (int i = 0; i < fTextComponent.fFormattedTextLength; i++) {
				g.drawStringWithTabExpansion(fTextComponent.fFormattedText[i], x, y, 0);
				y += fLineHeight;
			}
		}
	}

	boolean traverse(int direction, int viewWidth, int viewHeight, int[] visibleRectangle, int x, int y) {
		boolean acceptsFocus = super.traverse(direction, viewWidth, viewHeight, visibleRectangle, x, y);
		if (acceptsFocus && fChoice.fType == Choice.IMPLICIT) fChoice.updateSelectionIndex(fIndex);
		return acceptsFocus;
	}
}
