package javax.microedition.lcdui;

import com.ibm.ive.midp.FastVector;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

class ChoiceImpl implements Choice {

	// Holds the labels for each of the elements
	FastVector fStrings;
	// Holds a copy of the image snapshots used for each element
	FastVector fImageSnapshots;
	// Keeps track of mutable images whose snapshots must be disposed
	FastVector fIsSnapshotOfMutableImage;
	// Holds the exact image that was passed into each element
	FastVector fImages;

	IChoicePeer fPeer;

	int fFitPolicy;

	// Tracks the font used for every element. This vector will only be as
	// large as the largest index set by the user.
	FastVector fFonts;

	int fType;

	// This item tracks the currently selected item
	// when the list is a EXCLUSIVE or IMPLICIT list
	int fSelectedIndex = -1;

	// This vector should stay the same length as fStrings and fImages and
	// each item is a Boolean that holds the corresponding selection for
	// that element. This field is only valid for Choice.MULTIPLE
	private FastVector fSelectionList;

	public ChoiceImpl(int type) {
		fType = type;
	}

	public void setContents(String[] strings, Image[] images) {
		// strings must be not null and every item in the array should be
		// not null as well
		if (strings == null) throw new NullPointerException();
		fStrings = new FastVector(strings.length);
		for (int i = 0; i < strings.length; i++) {
			if (strings[i] == null) throw new NullPointerException();
			fStrings.addElement(strings[i]);
		}

		// If the image list if not null then it must be the same size as the
		// string array
		if (images != null) {
			if(images.length != strings.length) throw new IllegalArgumentException();
		}

		// Copy the images locally
		fImages = new FastVector(strings.length);
		fImageSnapshots = new FastVector(strings.length);
		fIsSnapshotOfMutableImage = new FastVector(strings.length);
		if (images == null) {
			// Every image in the vector should be null
			fImages.setSize(strings.length);
			fImageSnapshots.setSize(strings.length);
			fIsSnapshotOfMutableImage.setSize(strings.length);
		} else {
			for (int i = 0; i < images.length; i++) {
				fImages.addElement(images[i]);
				fImageSnapshots.addElement(createSnapshot(images[i]));
				fIsSnapshotOfMutableImage.addElement(new Boolean(images[i] != null && images[i].isMutable()));
			}
		}

		// Keep the selection vector up to date
		if (fType == Choice.MULTIPLE) {
			fSelectionList = new FastVector(strings.length);
			// Default every item to be unselected
			for (int i = 0; i < strings.length; i++) {
				fSelectionList.addElement(new Boolean(false));
			}
		} else {
			// Lists that are not MULTIPLE and contain at least one element
			// must always have a selection
			if (strings.length != 0) updateSelectionIndex(0);
		}
	}

	public int append(String elementName, Image image) {
		if (elementName == null) throw new NullPointerException();

		// Lazy initialize all needed structures
		if (fStrings == null) {
			fStrings = new FastVector();
			fImages = new FastVector();
			fImageSnapshots = new FastVector();
			fIsSnapshotOfMutableImage = new FastVector();
		}
		/*	Lists that are not MULTIPLE and contain at least one element
			must always have a selection */
		if (fType != Choice.MULTIPLE && fStrings.size() == 0) updateSelectionIndex(0);

		fStrings.addElement(elementName);
		fImages.addElement(image);
		Image snapshot = createSnapshot(image);
		fImageSnapshots.addElement(snapshot);
		fIsSnapshotOfMutableImage.addElement(new Boolean(image != null && image.isMutable()));

		// Keep selection vector in sync
		if (fType == Choice.MULTIPLE) {
			if (fSelectionList == null) fSelectionList = new FastVector();
			fSelectionList.addElement(new Boolean(false));
		}

		// Keep font vector in sync
		// Don't need to update the fonts vector until the font is actually
		// set by the user.

		// Keep peer in sync
		int index = fStrings.size() - 1;
		if (fPeer != null) fPeer.insert(index, elementName, snapshot);

		return index;
	}

	protected Image createSnapshot(Image image) {
		if (image == null) return null;
		return Image.createImage(image);
	}

	public void delete(int elementIndex) {
		checkIndex(elementIndex);

		fStrings.removeElementAt(elementIndex);
		fImages.removeElementAt(elementIndex);

		Image snapshot = (Image)fImageSnapshots.elementAt(elementIndex);
		fImageSnapshots.removeElementAt(elementIndex);
		Boolean isSnapshotOfMutableImage = (Boolean)fIsSnapshotOfMutableImage.elementAt(elementIndex);
		fIsSnapshotOfMutableImage.removeElementAt(elementIndex);

		// Free the vectors so they can be GCed
		if (fStrings.size() == 0) {
			fStrings = null;
			fImages = null;
			fImageSnapshots = null;
			fIsSnapshotOfMutableImage = null;
		}

		// Keep the selected index in sync
		if (fType == Choice.MULTIPLE) {
			fSelectionList.removeElementAt(elementIndex);
			if (fSelectionList.size() == 0) fSelectionList = null;
		} else {
			if (elementIndex == fSelectedIndex) {
				/*	Lists that are not MULTIPLE and contain at least one element
					must always have a selection */
				int index = (fStrings == null) ? -1 : Math.min(fSelectedIndex, fStrings.size() - 1);
				updateSelectionIndex(index);
			} else if (elementIndex < fSelectedIndex) {
				updateSelectionIndex(fSelectedIndex-1);
			}
		}

		// Keep fonts in sync
		if (fFonts != null) {
			if (elementIndex < fFonts.size()) {
				fFonts.removeElementAt(elementIndex);
				if (fFonts.size() == 0) fFonts = null;
			}
		}

		// Keep peer in sync
		if (fPeer != null) fPeer.delete(elementIndex);

		// After the peer is done with the native image we can safely free
		// the image snapshot
		if (snapshot != null && isSnapshotOfMutableImage.booleanValue()) snapshot.dispose();
	}

	public void deleteAll() {
		if (fStrings == null) return;

		fStrings = null;
		fImages = null;

		// Keep selection in sync
		if (fType == Choice.MULTIPLE) {
			fSelectionList = null;
		} else {
			updateSelectionIndex(-1);
		}

		// Keep fonts in sync
		if (fFonts != null) fFonts = null;
		// Keep peer in sync
		if (fPeer != null) fPeer.deleteAll();
		// After the peer has been updated we can safely dispose of all
		// images in the snapshot list
		if (fImageSnapshots != null) {
			int snapshotCount = fImageSnapshots.size();
			for (int i=0;i<snapshotCount;i++) {
				Image snapshot = (Image)fImageSnapshots.elementAt(i);
				Boolean isSnapshot = (Boolean)fIsSnapshotOfMutableImage.elementAt(i);
				if (snapshot != null && isSnapshot.booleanValue()) snapshot.dispose();
			}
			fIsSnapshotOfMutableImage = null;
			fImageSnapshots = null;
		}
	}

	public int getFitPolicy() {
		return fFitPolicy;
	}

	public Font getFont(int index) {
		checkIndex(index);

		if (fFonts != null) {
			if (index < fFonts.size()) {
				Font font = (Font)fFonts.elementAt(index);
				if (font != null) return font;
			}
		}

		return Font.getFont(Font.FONT_STATIC_TEXT);
	}

	public Image getImage(int elementIndex) {
		checkIndex(elementIndex);
		return (Image)fImages.elementAt(elementIndex);
	}

	public int getSelectedFlags(boolean[] selectedElementsToQuery) {
		if (selectedElementsToQuery == null) throw new NullPointerException();
		if ((fStrings != null) && (selectedElementsToQuery.length < fStrings.size())) throw new IllegalArgumentException();

		if (fType == Choice.MULTIPLE) {
			return getSelectedFlagsMultiple(selectedElementsToQuery);
		} else {
			return getSelectedFlagsSingle(selectedElementsToQuery);
		}
	}

	private int getSelectedFlagsMultiple(boolean[] selectedElementsToQuery) {
		int count = 0; // The number of elements set in the array so far
		int selectionCount = 0; // The number of selected elements

		// If no items have been added to the list then fSelectionList
		// could be null
		if (fSelectionList != null) {
			for (int i = 0; i < fSelectionList.size(); i++) {
				Boolean bool = (Boolean)fSelectionList.elementAt(i);
				selectedElementsToQuery[i] = bool.booleanValue();
				if (selectedElementsToQuery[i] == true) selectionCount++;
				count++;
			}
		}

		for (int i = count; i < selectedElementsToQuery.length; i++) {
			selectedElementsToQuery[i] = false;
		}

		return selectionCount;
	}

	private int getSelectedFlagsSingle(boolean[] selectedElementsToQuery) {
		for (int i=0;i<selectedElementsToQuery.length;i++) {
			selectedElementsToQuery[i] = (i == fSelectedIndex);
		}
		return fSelectedIndex == -1 ? 0 : 1;
	}

	public int getSelectedIndex() {
		if (fType == Choice.MULTIPLE) {
			// MULTIPLE always returns -1
			return -1;
		} else {
			return fSelectedIndex;
		}
	}

	public String getString(int elementIndex) {
		checkIndex(elementIndex);
		return (String)fStrings.elementAt(elementIndex);
	}

	public void insert(int elementIndex, String elementName, Image image) {
		if (elementName == null) throw new NullPointerException();

		// If no items have ever been created then created then the only valid
		// index is 0
		if ((fStrings == null) && (elementIndex != 0)) throw new IndexOutOfBoundsException();

		// If the element index is equal to the number of elements
		// the it should be treated as an append
		if ((fStrings == null) || (elementIndex == fStrings.size())) {
			append(elementName, image);
			return;
		}

		checkIndex(elementIndex);
		fStrings.insertElementAt(elementName, elementIndex);
		fImages.insertElementAt(image, elementIndex);
		Image snapshot = createSnapshot(image);
		fImageSnapshots.insertElementAt(snapshot, elementIndex);
		fIsSnapshotOfMutableImage.insertElementAt(new Boolean(image != null && image.isMutable()), elementIndex);

		// Keep the selected index in sync with the native widget
		if (fType == Choice.MULTIPLE) {
			fSelectionList.insertElementAt(new Boolean(false), elementIndex);
		} else {
			if (elementIndex <= fSelectedIndex) updateSelectionIndex(fSelectedIndex + 1);
		}

		// Keep fonts in sync
		if ((fFonts != null) && (elementIndex < fFonts.size())) {
			fFonts.insertElementAt(null, elementIndex);
		}

		// Keep peer in sync
		if (fPeer != null) fPeer.insert(elementIndex, elementName, snapshot);
	}

	public boolean isSelected(int elementIndex) {
		checkIndex(elementIndex);
		return isSelectedImpl(elementIndex);
	}

	boolean isSelectedImpl(int elementIndex) {
		if (fType == Choice.MULTIPLE) {
			Boolean bool = (Boolean) fSelectionList.elementAt(elementIndex);
			return bool.booleanValue();
		} else {
			return elementIndex == fSelectedIndex;
		}
	}

	public void set(int elementIndex, String elementName, Image image) {
		if (elementName == null) throw new NullPointerException();
		checkIndex(elementIndex);
		fStrings.setElementAt(elementName, elementIndex);
		fImages.setElementAt(image, elementIndex);

		// Keep the image snapshots up to date
//		Image oldImage = (Image) fImageSnapshots.elementAt(elementIndex);
		Image snapshot = createSnapshot(image);
		fImageSnapshots.setElementAt(snapshot, elementIndex);
		fIsSnapshotOfMutableImage.setElementAt(new Boolean(image != null && image.isMutable()), elementIndex);

		// Keep selected index in sync
		// Setting the label or image should not change the selection

		// Keep fonts in sync
		// Setting the label or image should not change font settings

		// Keep peer in sync
		if (fPeer != null) fPeer.set(elementIndex, elementName, snapshot);
	}

	public void setFitPolicy(int policy) {
		if ((policy != Choice.TEXT_WRAP_DEFAULT) && (policy != Choice.TEXT_WRAP_OFF) && (policy != Choice.TEXT_WRAP_ON)) {
			throw new IllegalArgumentException();
		}

		fFitPolicy = policy;
	}

	public void setFont(int index, Font font) {
		checkIndex(index);

		if (fFonts == null) {
			// Only create the array if a valid font is set
			if (font != null) {
				fFonts = new FastVector(index+1);
				fFonts.setSize(index+1);
				fFonts.setElementAt(font,index);
			}
		} else {
			if (index < fFonts.size()) {
				fFonts.setElementAt(font, index);
			} else {
				// Only grow the font vector if a valid font has been set
				if (font != null) {
					fFonts.setSize(index+1);
					fFonts.setElementAt(font, index);
				}
			}
		}

		// Keep peer in sync
		if (fPeer != null) fPeer.setFont(index, font);
	}

	public void setSelectedFlags(boolean[] selectedArrayToSet) {
		if (selectedArrayToSet == null) throw new NullPointerException();

		// Don't do anything if there is nothing to select.  This check is more
		// than an optimization because without it the code below will throw
		// a IndexOutOfBoundsException when setting the selected flags on an
		// array of length 0
		int size = size();
		if (size == 0) return;

		if (selectedArrayToSet.length < size) throw new IllegalArgumentException();

		if (fType == Choice.MULTIPLE) {
			// Set the proper selection for each element
			for (int i=0;i<size;i++) {
				setSelectedIndex(i, selectedArrayToSet[i]);
			}
		} else {
			int selectedIndex = 0;

			// Find the first true value in the list
			for (int i=0;i<size;i++) {
				if (selectedArrayToSet[i] == true) {
					selectedIndex = i;
					break;
				}
			}

			// Update only the selected index
			setSelectedIndex(selectedIndex, true);
		}
	}

	public void setSelectedIndex(int elementIndex, boolean isSelected) {
		checkIndex(elementIndex);
		switch (fType) {
			case Choice.MULTIPLE:
				fSelectionList.setElementAt(new Boolean(isSelected), elementIndex);
				if (fPeer != null) fPeer.setSelectedIndex(elementIndex, isSelected);
				break;
			case Choice.POPUP:
			case Choice.EXCLUSIVE:
			case Choice.IMPLICIT:
				if (isSelected == false) break;

				updateSelectionIndex(elementIndex);
				if (fPeer != null) fPeer.setSelectedIndex(elementIndex, true);
				break;
		}
	}

	Image getImageSnapshot(int index) {
		return fImageSnapshots == null ? null : (Image)fImageSnapshots.elementAt(index);
	}

	public int size() {
		return fStrings == null ? 0 : fStrings.size();
	}

	private void checkIndex(int index) {
		if (fStrings == null) throw new IndexOutOfBoundsException();
		if ((index < 0) || (index > (size() - 1)))
			throw new IndexOutOfBoundsException();
	}

	void updateSelectionIndex(int newIndex) {
		fSelectedIndex = newIndex;
	}

	void updateSelectionList(int index, boolean selection) {
		fSelectionList.setElementAt(new Boolean(selection), index);
	}
}
