package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;

abstract class Component extends Widget {

	int fX, fY, fWidth, fHeight;
	int fLayoutData = Item.LAYOUT_DEFAULT;

	abstract int getMinimumWidth();
	abstract int getMinimumHeight();

	final static Border gBorder = new PlatformBorder();
	final static Font gFont = Font.getDefaultFont();
	final static int FONT_HEIGHT = gFont._getHeight();

	Runnable fRepaintRunnable = getRepaintRunnable();
	Object fRepaintLock = new Object();

	boolean fDirty;
	int fDirtyX, fDirtyY, fDirtyW, fDirtyH;

	Component(Composite parent) {
		super(parent);
		if (fParent != null) fParent.fComponents.addElement(this);
	}

	Runnable getRepaintRunnable() {
		return new RepaintRunnable(this);
	}

	void dispose() {
		super.dispose();
		if (fParent != null) fParent.fComponents.removeElement(this);
	}

	boolean getShowsFocus() {
		return true;
	}

	int getPreferredWidth() {
		return getMinimumWidth();
	}

	int getPreferredHeight() {
		return getMinimumHeight();
	}

	boolean pointerPressed(int x, int y) {
		return false;
	}

	boolean pointerReleased(int x, int y) {
		return false;
	}

	boolean pointerDragged(int x, int y) {
		return false;
	}

	boolean keyPressed(int keyCode) {
		return false;
	}

	boolean keyReleased(int keyCode) {
		return false;
	}

	boolean keyRepeated(int keyCode) {
		return false;
	}

	boolean traverse(int direction, int viewWidth, int viewHeight, int[] visibleRectangle, int x, int y) {
		return false;
	}

	void traverseOut() {
		repaint();
	}

	int getType() {
		return Item.TYPE_UNUSED;
	}

	void paintBorder(Graphics g, boolean selected, int x, int y, int width, int height) {
		gBorder.paint(this, g, selected, x, y, width, height);
	}

	int getLayoutData() {
		return fLayoutData;
	}

	int getHAlignment() {
		/*
		 * 0x3 == Item.LAYOUT_LEFT | Item.LAYOUT_CENTER | Item.LAYOUT_RIGHT
		 */
		return fLayoutData & 0x3;
	}

	boolean canVShrink() {
		/*
		 * 0x5000 == Item.LAYOUT_2 | Item.LAYOUT_VSHRINK
		 */
		return (fLayoutData & 0x5000) == 0x5000;
	}

	boolean canVExpand() {
		/*
		 * 0x6000 == Item.LAYOUT_2 | Item.LAYOUT_VEXPAND
		 */
		return (fLayoutData & 0x6000) == 0x6000;
	}

	boolean canShrink() {
		/*
		 * 0x4800 == Item.LAYOUT_2 | Item.LAYOUT_SHRINK
		 */
		return (fLayoutData & 0x4400) == 0x4400;
	}

	boolean canExpand() {
		/*
		 * 0x4800 == Item.LAYOUT_2 | Item.LAYOUT_EXPAND
		 */
		return (fLayoutData & 0x4800) == 0x4800;
	}

	public int getVAlignment() {
		/*
		 * 0x30 == Item.LAYOUT_TOP | Item.LAYOUT_BOTTOM | Item.LAYOUT_VCENTER
		 */
		return fLayoutData & 0x30;
	}

	boolean newlineBefore() {
		return (fLayoutData & Item.LAYOUT_NEWLINE_BEFORE) != 0;
	}

	boolean newlineAfter() {
		return (fLayoutData & Item.LAYOUT_NEWLINE_AFTER) != 0;
	}

	void hideNotify() {
		/* No-op */
	}

	void showNotify() {
		/* No-op */
	}

	Rectangle getOriginBounds() {
		return new Rectangle(getOriginX(), getOriginY(), fWidth, fHeight);
	}

	Rectangle getVisibleBounds() {
		if (fParent == null) return new Rectangle(fX, fY, fWidth, fHeight);
		return getOriginBounds().intersection(fParent.getVisibleBounds());
	}

	void invalidate() {
		fParent.layoutChanged(this);
	}

	int getWidth() {
		return fWidth;
	}

	int getHeight() {
		return fHeight;
	}

	int getOriginX() {
		return fParent.getOriginX(this);
	}

	int getOriginY() {
		return fParent.getOriginY(this);
	}

	int translateX(int x) {
		return fParent.translateX(this, x);
	}

	int translateY(int y) {
		return fParent.translateY(this, y);
	}

	void setOriginAndClip(Graphics g) {
		setOriginAndClip(g, 0, 0, fWidth, fHeight);
	}

	void setOriginAndClip(Graphics g, int x, int y, int w, int h) {
		g.fData.fTranslateX = getOriginX();
		g.fData.fTranslateY = getOriginY();

		Rectangle visible = getVisibleBounds();
		g.restrictClip(visible.x, visible.y, visible.width, visible.height);

		g.setClip(x, y, w, h);
	}

	void setBounds(int x, int y, int w, int h) {
		fX = x;
		fY = y;
		fWidth = w;
		fHeight = h;
	}

	abstract void paint(Graphics g);

	void paint() {
		Graphics g = getGraphics();
		setOriginAndClip(g);
		paint(g);
	}

	void repaint() {
		synchronized (fRepaintLock) {
			if (isDisposed()) return;
			fDirtyX = fDirtyY = 0;
			fDirtyW = fWidth;
			fDirtyH = fHeight;
			if (!fDirty) {
				fDirty = true;
				Device.postRunnable(fRepaintRunnable);
			}
		}
	}

	void repaint(int x, int y, int width, int height) {
		try {
			synchronized (fRepaintLock) {
				if (isDisposed()) return;
				if (fDirty) {
					int left = fDirtyX < x ? fDirtyX : x;
					int top = fDirtyY < y ? fDirtyY : y;
					int lhs = fDirtyX + fDirtyW;
					int rhs = x + width;
					int right = lhs > rhs ? lhs : rhs;
					lhs = fDirtyY + fDirtyH;
					rhs = y + height;
					int bottom = lhs > rhs ? lhs : rhs;
					fDirtyX = left; fDirtyY = top; fDirtyW = right - left; fDirtyH = bottom - top;
				} else {
					fDirtyX = x;
					fDirtyY = y;
					fDirtyW = width;
					fDirtyH = height;
					fDirty = true;
					Device.postRunnable(fRepaintRunnable);
				}
			}
		} catch (NullPointerException npe) {
			/* avoid synchronizing:
			 * do nothing in the rare case that fPeer is set to null in another thread
			 */
		}
	}

	boolean contains(int x, int y) {
		return (x >= 0 && x < fWidth) && (y >= 0 && y < fHeight);
	}

	boolean containsAbsolutePoint(int x, int y) {
		int absoluteX = getOriginX();
		int absoluteY = getOriginY();
		return (x >= absoluteX) && (y >= absoluteY) && ((x - absoluteX) < fWidth) && ((y - absoluteY) < fHeight);
	}

	void clearCachedSizes() {
		/* No-op */
	}

	int getBackgroundColor() {
		return DisplayPeer.COLOR_DISPLAYABLE_BACKGROUND_RGB;
	}

	void updatePosition() {
		/* No-op for drawn widgets */
	}

	void updateCommands() {
		/* No-op */
	}

	void sizeChanged(int w, int h) {
		/* No-op */
	}

	Image getOffscreenBuffer() {
		return fParent.getOffscreenBuffer();
	}

	class RepaintRunnable implements Runnable {
		Component fComponent;

		RepaintRunnable(Component c) {
			fComponent = c;
		}

		public void run() {
			synchronized (Device.gRepaintLock) {
				int x, y, w, h;
				synchronized (fRepaintLock) {
					// make sure that they are in a good state
					x = fDirtyX;
					y = fDirtyY;
					w = fDirtyW;
					h = fDirtyH;
					fDirty = false;
				}

				/*
				 * It's important to do this check after clearing the fDirty
				 * flag. Otherwise, the Canvas will be unable to schedule
				 * another repaint runnable.
				 */
				if (!isShown()) return;

				Graphics g = getGraphics();
	  			g.activate();
				setOriginAndClip(g, x, y, w, h);
				g.setColor(getBackgroundColor());
				g.fillRect(x, y, w, h);
				paint(g);
				g.flush(getOriginX() + x, getOriginY() + y, w, h);
			}
		}
	}
}
