package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2005, 2006  All Rights Reserved
 */

import java.util.*;

import com.ibm.ive.midp.*;

abstract class Composite extends Component {

	Layout fLayout;
	Component fFocusComponent;
	FastVector fComponents = new FastVector(10);

	boolean fExpandVertically = false;

	Composite(Composite parent) {
		super(parent);
		fLayout = createLayout();
	}

	Layout createLayout() {
		return new CompositeLayout(this);
	}

	void layout() {
		fLayout.layout();
		for (int i = 0; i < fComponents.size(); i++) {
			Component component = (Component) fComponents.elementAt(i);
			if (component.fShown) component.layout();
		}
	}

	void layoutChanged(Component child) {
		if (fParent == null) return;

		fParent.layoutChanged(child);
	}

	int getOriginX(Component child) {
		if (fParent == null) return fX + child.fX;
		return fParent.getOriginX(this) + child.fX;
	}

	int getOriginY(Component child) {
		if (fParent == null) return fY + child.fY;
		return fParent.getOriginY(this) + child.fY;
	}

	int translateX(Component child, int x) {
		if (fParent == null) return x - child.fX;
		return fParent.translateX(this, x) - child.fX;
	}

	int translateY(Component child, int y) {
		if (fParent == null) return y - child.fY;
		return fParent.translateY(this, y) - child.fY;
	}

	int toChildX(Component child, int x) {
		return x - child.fX;
	}

	int toChildY(Component child, int y) {
		return y - child.fY;
	}

	Component getComponentAt(int x, int y) {
		Enumeration components = fComponents.elements();
		while (components.hasMoreElements()) {
			Component component = (Component) components.nextElement();
			if (component.contains(toChildX(component, x), toChildY(component, y))) return component;
		}
		return null;
	}

	void paint(Graphics g) {
		paintComponents(g);
	}

	void paintComponents(Graphics g) {
		Enumeration e = fComponents.elements();
		while (e.hasMoreElements()) {
			Component component = (Component) e.nextElement();
			if (component.isShown()) component.paint();
		}
	}

	void dispose() {
		hide();
		disposeComponents();
		if (fParent != null) {
			if (hasFocus()) fParent.setFocus(null);
			fParent.fComponents.removeElement(this);
		}
	}

	void disposeComponents() {
		for (int i = fComponents.size()-1; i >= 0; i--) {
			/*
			 * Traverse backwards because the elements remove themselves
			 * from the vector when disposed.
			 */
			((Component) fComponents.elementAt(i)).dispose();
		}
	}

	void showNotify() {
		for (int i = 0; i < fComponents.size(); i++) {
			Component component = (Component) fComponents.elementAt(i);
			component.showNotify();
		}
	}

	void hideNotify() {
		for (int i = 0; i < fComponents.size(); i++) {
			Component component = (Component) fComponents.elementAt(i);
			component.hideNotify();
		}
	}

	boolean pointerDragged(int x, int y) {
		if (fFocusComponent != null) {
			return fFocusComponent.pointerDragged(toChildX(fFocusComponent, x), toChildY(fFocusComponent, y));
		}
		return false;
	}

	boolean pointerPressed(int x, int y) {
		Component component = getComponentAt(x, y);
		if (component != null && (!hasFocus() || fFocusComponent != component)) {
			if (component.traverse(CustomItem.NONE, component.fWidth, component.fHeight, getVisibleRect(component), toChildX(component, x), toChildY(component, y))) {
				setFocus(component);
			}
		}

		if (fFocusComponent != null) {
			if (fFocusComponent == component) {
				return fFocusComponent.pointerPressed(toChildX(fFocusComponent, x), toChildY(fFocusComponent, y));
			} else {
				setFocus(null);
			}
		}
		return false;
	}

	boolean pointerReleased(int x, int y) {
		if (fFocusComponent != null) {
			return fFocusComponent.pointerReleased(toChildX(fFocusComponent, x), toChildY(fFocusComponent, y));
		}
		return false;
	}

	boolean keyPressed(int keyCode) {
		if (fFocusComponent != null) return fFocusComponent.keyPressed(keyCode);
		return false;
	}

	boolean keyReleased(int keyCode) {
		if (fFocusComponent != null) return fFocusComponent.keyReleased(keyCode);
		return false;
	}

	boolean keyRepeated(int keyCode) {
		if (fFocusComponent != null) return fFocusComponent.keyRepeated(keyCode);
		return false;
	}

	int[] getVisibleRect(Component c) {
		Rectangle visibleBounds = c.getVisibleBounds();

		int[] visibleRect = new int[4];
		visibleRect[0] = visibleBounds.x - c.getOriginX();
		visibleRect[1] = visibleBounds.y - c.getOriginY();
		visibleRect[2] = visibleBounds.width;
		visibleRect[3] = visibleBounds.height;
		return visibleRect;
	}

	void ensureVisible(Component c, int[] visRect) {
		/* No-op */
	}

	/*
	 * This method should only be called for a
	 * directional traversal. Do not call it for
	 * CustomItem.NONE.
	 */
	boolean traverse(int direction) {
		int[] visRect = getVisibleRect(this);
		int viewWidth = fWidth;
		int viewHeight = fHeight;
		return traverse(direction, viewWidth, viewHeight, visRect, 0, 0);
	}

	void traverseOut() {
		if (fFocusComponent != null) fFocusComponent.traverseOut();
		super.traverseOut();
	}

	Widget getFocus() {
		return fFocusComponent;
	}

	void setFocus(Component component) {
		if (hasFocus() && component == fFocusComponent) {
			/*
			 * We've walked high enough up the tree that the focus
			 * change is contained below this node.
			 */
			return;
		}

		if (component != fFocusComponent) {
			if (fFocusComponent != null) fFocusComponent.traverseOut();
			fFocusComponent = component;
		}

		if (fParent != null) fParent.setFocus(this);
	}

	int getContentWidth() {
		return fWidth;
	}

	int getMinimumHeight() {
		return fLayout.getMinimumHeight();
	}

	int getMinimumWidth() {
		return fLayout.getMinimumWidth();
	}

	int getPreferredHeight() {
		return fLayout.getPreferredHeight();
	}

	int getPreferredWidth() {
		return fLayout.getPreferredWidth();
	}

	int[] translateComponentRectToChildRect(Component child, int[] visRect) {
		int[] cVisRect = new int[4];
		int visRight = visRect[0] + visRect[2];
		int visBottom = visRect[1] + visRect[3];

		int cRight = child.fX + child.fWidth;
		int cBottom = child.fY + child.fHeight;

		int right = (cRight < visRight) ? cRight : visRight;
		int bottom = (cBottom < visBottom) ? cBottom : visBottom;
		cVisRect[0] = (visRect[0] > child.fX) ? visRect[0] - child.fX : 0;
		cVisRect[1] = (visRect[1] > child.fY) ? visRect[1] - child.fY : 0;

		int width = right - child.fX - cVisRect[0];
		int height = bottom - child.fY - cVisRect[1];
		cVisRect[2] = (width < 0) ? 0 : width;
		cVisRect[3] = (height < 0) ? 0 : height;

		return cVisRect;
	}

	void translateChildRectToComponentRect(Component child, int[] childRect, int[] visRect) {
		visRect[0] = childRect[0] + child.fX;
		visRect[1] = childRect[1] + child.fY;
		visRect[2] = childRect[2];
		visRect[3] = childRect[3];
	}

	boolean traverseComponent(Component child, int direction, int viewWidth, int viewHeight, int[] visRect, int x, int y) {
		int[] childRect = translateComponentRectToChildRect(child, visRect);
		int childViewWidth = (viewWidth > child.fWidth) ? child.fWidth : viewWidth;
		int childViewHeight = (viewHeight > child.fHeight) ? child.fHeight: viewHeight;
		if (child.traverse(direction, childViewWidth, childViewHeight, childRect, toChildX(child, x), toChildY(child, y))) {
			setFocus(child);
			translateChildRectToComponentRect(child, childRect, visRect);
			ensureVisible(this, visRect);
			return true;
		}
		return false;
	}

	boolean traverse(int direction, int viewWidth, int viewHeight, int[] visRect, int x, int y) {
		Component next = null;
		if (direction == CustomItem.NONE) {
			next = getComponentAt(x, y);
		} else if (fFocusComponent == null) {
			next = fLayout.getNextComponent(next, direction);
		} else {
			next = fFocusComponent;
		}

		if (next == null) return false;

		/*
		 * We always want to pass the true direction to the component traverse method.
		 * However, after moving up or down a row, we may want to change the direction
		 * in which we look for the next component.
		 */
		int layoutDirection = direction;

		while (true) {
			if (traverseComponent(next, direction, viewWidth, viewHeight, visRect, x, y)) return true;

			if (direction == CustomItem.NONE) return false;
			next = fLayout.getNextComponent(next, layoutDirection);

			if (next == null) return false;

			/*
			 * When traversing up or down, we should only jump one row and then start
			 * traversing along the row, effectively changing the traversal from up/down
			 * to left/right.
			 */
			if (direction == Canvas.UP) {
				layoutDirection = Canvas.LEFT;
			} else if (direction == Canvas.DOWN) {
				layoutDirection = Canvas.RIGHT;
			}
		}
	}

	boolean expandContentsVertically() {
		return fExpandVertically;
	}

	void clearCachedSizes() {
		for (int i = 0; i < fComponents.size(); i++) {
			Component component = (Component) fComponents.elementAt(i);
			component.clearCachedSizes();
		}
	}

	void update() {
		for (int i = 0; i < fComponents.size(); i++) {
			Component component = (Component) fComponents.elementAt(i);
			component.update();
		}
	}
}
