package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2005  All Rights Reserved
 */

import com.ibm.ive.midp.*;

class CompositeLayout extends Layout {

	Row fRows[] = new Row[20];
	int fRowsLength = 0;
	int fVMargin = 0;
	int fHMarginLeft = 0;
	int fHMarginRight = 0;
	int fVSpacing = 0;
	int fHSpacing = 0;

	static int INITIAL_LAYOUT_DIRECTION = Item.LAYOUT_LEFT;

	CompositeLayout(Composite c) {
		fComposite = c;
	}

	CompositeLayout(Composite c, int hMarginLeft, int hMarginRight, int vMargin, int hSpacing, int vSpacing) {
		fComposite = c;
		fHMarginLeft = hMarginLeft;
		fHMarginRight = hMarginRight;
		fVMargin = vMargin;
		fHSpacing = hSpacing;
		fVSpacing = vSpacing;
	}

	void addRow(Row row) {
		if (fRowsLength == fRows.length) {
			Row[] newArray = new Row[fRows.length * 2];
			System.arraycopy(fRows, 0, newArray, 0, fRows.length);
			fRows = newArray;
		}
		fRows[fRowsLength] = row;
		fRowsLength++;

		row.layoutHorizontally();
		fLayoutHeight += row.fCurrentHeight + fVSpacing;
	}

	Row breakRow(Row currentRow, Component item) {
		if (currentRow.fComponentsLength == 0) {
			currentRow.fCurrentHeight = Component.FONT_HEIGHT;
		}
		addRow(currentRow);
		return currentRow.createNextRow(currentRow.fAlignment);
	}

	Row getRowAt(int x, int y) {
		for (int i = 0; i < fRowsLength; i++) {
			Row row = fRows[i];
			if (y >= row.fY && y <= (row.fY + row.fCurrentHeight)) return row;
		}
		return null;
	}

	void layout() {
		FastVector components = fComposite.fComponents;
		int componentsLength = components.size();

		fRowsLength = 0;
		fLayoutHeight = 0;
		if (componentsLength == 0) return;

		Row currentRow = new Row(this, fHMarginLeft, fVMargin, fComposite.getContentWidth() - (fHMarginLeft + fHMarginRight), INITIAL_LAYOUT_DIRECTION);

		for (int i = 0; i < componentsLength; i++) {
			Component item = (Component) components.elementAt(i);
			if (!item.fShown) continue;

			/* Break the row if necessary before the item. */
			if (item.newlineBefore() && currentRow.fComponentsLength > 0) {
				currentRow = breakRow(currentRow, item);
			}

			/* Add the item to the current row. */
			if (!currentRow.add(item)) {
				addRow(currentRow);
				currentRow = currentRow.createNextRow(item.getHAlignment());
				currentRow.add(item);
			}

			/* Break the row if necessary after the item. */
			if (item.newlineAfter()) {
				currentRow = breakRow(currentRow, item);
			}
		}

		if (currentRow.fComponentsLength > 0) {
			addRow(currentRow);
		}

		layoutVertical();
	}

	void layoutVertical() {
//		System.err.println("Computed layout height: " + fLayoutHeight + " max: " + fComposite.fHeight + " fComposite: " + fComposite);

		if (fComposite.expandContentsVertically() && fComposite.fHeight > fLayoutHeight) {
			int expandableRows = getExpandableRowCount();
			if (expandableRows > 0) {
				int extraPixels = fComposite.fHeight - (2 * fVMargin) - fLayoutHeight;
				int pixels = extraPixels / expandableRows;
				int remaining = extraPixels % expandableRows;
				int expandedCount = 0;

				for (int i = 0; i < fRowsLength; i++) {
					Row row = fRows[i];
					if (row.canVExpand()) {
						if (++expandedCount == expandableRows) {
							row.fCurrentHeight += pixels + remaining;
						} else {
							row.fCurrentHeight += pixels;
						}
					}
				}
			}
		}

		int y = fVMargin;
		for (int i = 0; i < fRowsLength; i++) {
			fRows[i].layoutVerically(y);
			y += fRows[i].fCurrentHeight + fVSpacing;
		}

		fLayoutHeight = y;
	}

	int getExpandableRowCount() {
		int count = 0;
		for (int i = 0; i < fRowsLength; i++) {
			if (fRows[i].canVExpand()) count++;
		}
		return count;
	}

	boolean locateItem(Component peer, int[] rowRet, int[] peerRet) {
		for (int i = 0; i < fRowsLength; i++) {
			Row row = fRows[i];
			if (row.contains(peer)) {
				rowRet[0] = i;
				peerRet[0] = row.getIndex(peer);
				return true;
			}
		}
		return false;
	}

	Component getNextComponent(Component reference, int direction) {
		if (reference == null) {
			switch (direction) {
				case Canvas.LEFT:
				case Canvas.UP:
					if (fRowsLength == 0) return null;
					Row last = fRows[fRowsLength - 1];
					if (last.fComponentsLength == 0) return null;
					return last.fComponents[last.fComponentsLength - 1];
				case Canvas.DOWN:
				case Canvas.RIGHT:
					if (fRowsLength == 0 || fRows[0].fComponentsLength == 0) return null;
					return fRows[0].fComponents[0];
			}
		}

		int[] rowRet = new int[1];
		int[] peerRet = new int[1];
		locateItem(reference, rowRet, peerRet);
		int rowIndex = rowRet[0];
		int peerIndex = peerRet[0];

		switch (direction) {
			case Canvas.LEFT:
				if (peerIndex > 0) {
					Row current = fRows[rowIndex];
					return current.fComponents[--peerIndex];
				} else while (rowIndex > 0) {
					Row previous = fRows[--rowIndex];
					int peerCount = previous.fComponentsLength;
					if (peerCount > 0) {
						return previous.fComponents[--peerCount];
					}
				}
				return null;
			case Canvas.RIGHT:
				Row current = fRows[rowIndex];
				if (current == null) {
					return null;
				} else if (peerIndex + 1 < current.fComponentsLength) {
					return current.fComponents[++peerIndex];
				} else while (rowIndex + 1 < fRowsLength) {
					Row next = fRows[++rowIndex];
					if (next.fComponentsLength > 0) {
						return next.fComponents[0];
					}
				}
				return null;
			case Canvas.UP:
				if (reference == null && fRowsLength > 0) {
					return fRows[rowIndex].fComponentsLength > 0 ? fRows[rowIndex].fComponents[0] : null;
				} else while (rowIndex > 0) {
					Row previous = fRows[--rowIndex];
					Component item = previous.fComponentsLength > 0 ? previous.fComponents[previous.fComponentsLength - 1] : null;
					if (item != null) return item;
				}
				return null;
			case Canvas.DOWN:
				if (reference == null && fRowsLength > 0) {
					return fRows[rowIndex].fComponentsLength > 0 ? fRows[rowIndex].fComponents[0] : null;
				} else while (rowIndex + 1 < fRowsLength) {
					Row next = fRows[++rowIndex];
					Component item = next.fComponents[0];
					if (item != null) return item;
				}
				return null;
		}
		return null;
	}

	int getMinimumWidth() {
		return 0;
	}

	int getMinimumHeight() {
		return 0;
	}

	int getPreferredWidth() {
		return fComposite.fWidth;
	}

	int getPreferredHeight() {
		return fLayoutHeight;
	}

	int getMaxVisibleWidth() {
		return fComposite.getContentWidth() - fHMarginLeft - fHMarginRight;
	}

	int getMaxVisibleHeight() {
		return fComposite.fHeight - fVMargin - fVMargin;
	}

	void hideNotify() {
		int numberOfRows = fRowsLength;
		Row rows [] = fRows;
		for (int i = 0; i < numberOfRows; i++) {
			rows[i].hideNotify();
		}
	}
}
