package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM.
 * (c) Copyright IBM Corp 2000, 2005.
 *
 */

import java.util.*;

import com.ibm.ive.midp.*;

/**
 * A <code>DateField</code> is a UI item allowing the user
 * to select a date, time or both depending on the
 * receiver's type.
 *
 * @see DateField#DATE
 * @see DateField#TIME
 * @see DateField#DATE_TIME
 **/
public class DateField extends Item {

	public static final int DATE = 1;
	public static final int TIME = 2;
	public static final int DATE_TIME = 3;

	Date fDate;
	int fInputMode;
	TimeZone fTimeZone;

	DateFieldPeer fPeer;

	private static final long ONE_DAY_MILLI	= 60*60*24*1000;
	private long zeroEpochInTimeZone;

	public DateField(String label, int inputMode) {
		this(label, inputMode, null);
	}

	public DateField(String label, int inputMode, TimeZone aTimeZone){
		super(label);
		setInputMode(inputMode);
		fTimeZone = aTimeZone;
		init();
	}

	private void init() {
		Calendar zeroEpoch = getCalendar();
		zeroEpoch.set(Calendar.MONTH, Calendar.JANUARY);
		zeroEpoch.set(Calendar.DAY_OF_MONTH, 1);
		zeroEpoch.set(Calendar.YEAR, 1970);
		zeroEpoch.set(Calendar.HOUR_OF_DAY, 0);
		zeroEpoch.set(Calendar.MINUTE, 0);
		zeroEpoch.set(Calendar.SECOND, 0);
		zeroEpoch.set(Calendar.MILLISECOND, 0);
		zeroEpochInTimeZone = zeroEpoch.getTime().getTime();
	}

	public Date getDate() {
		synchronized (Device.gDisplayableLock) {
			Date returnValue = fDate;
			if (fPeer != null) returnValue = fPeer.getDate();
			returnValue = adjustDate(returnValue);
			return returnValue;
		}
	}

	Date adjustDate(Date returnValue) {
		if (returnValue != null) {
			Calendar currentTime = getCalendar();
			currentTime.setTime(returnValue);

			if (fInputMode == DATE) {
				currentTime.set(Calendar.HOUR_OF_DAY, 0);
				currentTime.set(Calendar.MINUTE, 0);
				currentTime.set(Calendar.SECOND, 0);
				currentTime.set(Calendar.MILLISECOND, 0);
				returnValue = currentTime.getTime();
			} else if (fInputMode == TIME) {
				// If date is more than 24 hours away from January 1, 1970
				// in the current timeZone, return null.
				if (Math.abs(zeroEpochInTimeZone-returnValue.getTime()) > ONE_DAY_MILLI) {
					returnValue = null;
				} else {
					currentTime.setTime(returnValue);
					currentTime.set(Calendar.MONTH, Calendar.JANUARY);
					currentTime.set(Calendar.DAY_OF_MONTH, 1);
					currentTime.set(Calendar.YEAR, 1970);
					currentTime.set(Calendar.SECOND, 0);
					currentTime.set(Calendar.MILLISECOND, 0);
					returnValue = currentTime.getTime();
				}
			} else {
				currentTime.set(Calendar.SECOND, 0);
				currentTime.set(Calendar.MILLISECOND, 0);
				returnValue = currentTime.getTime();
			}
		}
		return returnValue;
	}

	void updateDate(Date date) {
		fDate = date;
		try {
			notifyStateChanged();
		} catch (IllegalStateException exp1) {
			// ignore on purpose.  This method should
			// only be called from the UI, which means that the item
			// in a form. ie, this exception should never happen.
		}
	}

	Calendar getCalendar() {
		return fTimeZone == null ? Calendar.getInstance() : Calendar.getInstance(fTimeZone);
	}

	public int getInputMode() {
		synchronized (Device.gDisplayableLock) {
			return fInputMode;
		}
	}

	public void setDate(Date newDate) {
		synchronized (Device.gDisplayableLock) {
			fDate = adjustDate(newDate);
			// Keep peer in sync
			if (fPeer != null) fPeer.setDate(fDate);
		}
	}

	public void setInputMode(int newInputMode) {
		synchronized (Device.gDisplayableLock) {
			if (newInputMode < DATE || newInputMode > DATE_TIME)
				throw new IllegalArgumentException(MidpMsg.getString("DateField.setInputMode.InvalidInputMode")); //$NON-NLS-1$

			// exit if nothing has changed
			if (fInputMode == newInputMode) return;

			fInputMode = newInputMode;

			// Keep peer in sync
			if (fPeer != null) fPeer.updateInputMode();
		}
	}

	int getType() {
		return TYPE_DATEFIELD;
	}

	int getHAlignment() {
		if ((fLayout & Item.LAYOUT_2) != 0) {
			return super.getHAlignment();
		} else {
			return Item.LAYOUT_LEFT;
		}
	}

	ItemComponent getItemComponent() {
		if (fPeer == null) return null;
		return (ItemComponent) fPeer.fParent;
	}
}
