package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import com.ibm.ive.midp.*;
import com.ibm.ive.midp.util.*;
import javax.microedition.midlet.*;
import java.util.*;

/**
 * The DisplayPeer class implements platform-specific Display behavior.
 *
 * <p>It must answer some display properties such as dimensions and system colors.
 *
 * <p>It provides a GraphicsThreadsafe instance that can be used to draw directly to
 * the display at any time.
 *
 * <p>Its primary purpose however is to create DisplayablePeer instances and set them
 * to be the Display's current Displayable.
 *
 * @author IBM
 */
class DisplayPeer implements IEventListener {

	static int COLOR_BACKGROUND_RGB;
	static int COLOR_BORDER_RGB;
	static int COLOR_FOREGROUND_RGB;
	static int COLOR_FOREGROUND_UNEDITABLE_TEXT_RGB;
	static int COLOR_HIGHLIGHTED_BACKGROUND_RGB;
	static int COLOR_HIGHLIGHTED_BORDER_RGB;
	static int COLOR_HIGHLIGHTED_FOREGROUND_RGB;
	static int COLOR_TITLE_BACKGROUND_RGB;
	static int COLOR_TITLE_FOREGROUND_RGB;
	static int COLOR_HYPERLINK_RGB;
	static int COLOR_HIGHLIGHTED_HYPERLINK_RGB;
	static int COLOR_TEXT_HIGHLIGHT_RGB;
	static int COLOR_TEXT_HIGHLIGHT_FOREGROUND_RGB;
	static int COLOR_DISPLAYABLE_BACKGROUND_RGB;
	static int COLOR_DISPLAYABLE_BORDER_RGB;
	static int COLOR_TICKER_BACKGROUND_RGB;
	static int COLOR_TICKER_FOREGROUND_RGB;
	static int COLOR_MENU_BACKGROUND_RGB;
	static int COLOR_MENU_FOREGROUND_RGB;
	static int COLOR_MENU_HIGHLIGHTED_BACKGROUND_RGB;
	static int COLOR_MENU_HIGHLIGHTED_FOREGROUND_RGB;
	static int COLOR_MENU_SHADOW_RGB;
	static int COLOR_MENU_DISABLED_TEXT;
	static int COLOR_MENU_DISABLED_BORDER;
	static int COLOR_LIST_SOFT_HIGHLIGHTED_BACKGROUND_RGB;
	static int COLOR_LIST_SOFT_HIGHLIGHTED_FOREGROUND_RGB;

	Display fDisplay;
	Displayable fDisplayable;
	Window fCurrentWindow;
	FastVector fWindows = new FastVector(4);

	int fHandle;

	Image fOffscreenBuffer;

	boolean fClosing = false;
	boolean fIsOpen = false;
	boolean fMousePressed = false;

	static {
		initImpl();
	}

	DisplayPeer(Display display) {
		fDisplay = display;
		create();
	}

	DisplayablePeer createDisplayablePeer(Displayable displayable) {
		switch (displayable.getDisplayableType()) {
			case Displayable.TYPE_CANVAS:
				return new CanvasPeer(fDisplay, (Canvas) displayable);
			case Displayable.TYPE_FORM:
				return new FormPeer(fDisplay, (Form) displayable);
			case Displayable.TYPE_ALERT:
				return new AlertPeer(fDisplay, (Alert) displayable);
			case Displayable.TYPE_LIST:
			case Displayable.TYPE_TEXTBOX:
				return new DisplayablePeer(fDisplay, displayable);
		}

		return null;
	}

	void setCurrent(final Displayable displayable) {
		if (!fIsOpen) open();
		fDisplayable = displayable;

		if (displayable.fPeer != null) {
			if (displayable.getDisplayableType() == Displayable.TYPE_CANVAS) {
				Device.postRunnable(new Runnable() {
					public void run() {
						displayable.showNotify();
						displayable.fPeer.repaint();
					}
				});
			} else {
				displayable.fPeer.repaint();
			}
			displayable.fPeer.scheduleUpdate(DisplayablePeer.COMMANDS_CHANGED);
			displayable.fPeer.scheduleUpdate(DisplayablePeer.TITLE_UPDATED);
			return;
		}

		Device.postRunnable(new Runnable() {
			public void run() {
				synchronized (fDisplay.fLock) {
					// Check for the cases where setCurrent() was called again with the same or with a
					// different displayable before its runnable got to run to create the first peer
					if (displayable != fDisplay.fCurrent || displayable.fPeer != null) return;

					synchronized (Device.gDisplayableLock) {
						DisplayablePeer peer = createDisplayablePeer(displayable);
						int displayableType = displayable.getDisplayableType();
						if (displayableType != Displayable.TYPE_ALERT) {
							displayable.callShowNotify();
						}
						peer.show();
						/*
						 * Call showNotify for an Alert after show, so that
						 * we start the Alert alarm thread after the user sees
						 * the content of the Alert
						 */
						if (displayableType == Displayable.TYPE_ALERT) {
							displayable.callShowNotify();
						}
					}
				}
			}
		});
	}

	void addWindow(Window window) {
		if (fCurrentWindow != null && window.getWindowHandle() == fHandle) fCurrentWindow.fInvalidated = true;
		fCurrentWindow = window;
		if (!fWindows.contains(window)) fWindows.addElement(window);
	}

	void removeWindow(Window window) {
		fWindows.removeElement(window);
		if (fCurrentWindow == window) {
			fCurrentWindow = (Window) fWindows.lastElement();
		}
		if (fCurrentWindow != null) {
			fDisplayable = fCurrentWindow.getDisplayable();
			fCurrentWindow.fInvalidated = false;
			Device.gDisplayableUpdateRunnable = fCurrentWindow.fUpdateRunnable;
			/*
			 * Don't repaint the Canvas if it is being made the current window
			 * because the popupmenu is shrinking, we draw the cached buffer
			 * in this case.
			 */
//TODO: Need to check if there is any case that would require a repaint() request for sure
			if (window.getType() == Displayable.TYPE_POPUP_MENU && fCurrentWindow.getType() ==  Displayable.TYPE_CANVAS) return;

			paint();
		}
	}

	/**
	 * Answer the display color associated with the given constant. The constant
	 * is one of the COLOR_ constants declared in this class.
	 *
	 * @return the system color associated with the given constant.
	 */
	static int getSpecifiedColor(int specifier) {
		switch (specifier) {
			case Display.COLOR_BACKGROUND: return COLOR_BACKGROUND_RGB;
			case Display.COLOR_BORDER: return COLOR_BORDER_RGB;
			case Display.COLOR_FOREGROUND: return COLOR_FOREGROUND_RGB;
			case Display.COLOR_HIGHLIGHTED_BACKGROUND: return COLOR_HIGHLIGHTED_BACKGROUND_RGB;
			case Display.COLOR_HIGHLIGHTED_BORDER: return COLOR_HIGHLIGHTED_BORDER_RGB;
			case Display.COLOR_HIGHLIGHTED_FOREGROUND: return COLOR_HIGHLIGHTED_FOREGROUND_RGB;
			default: return 0x000000;
		}
	}

	public boolean getAllowMenuEvents() {
		return true;
	}

	int getHeight() {
		final int height[] = new int[1];
		Device.syncExec(new Runnable() {
			public void run() {
				height[0] = getHeightImpl(fHandle);
			}
		});
		return height[0];
	}

	int getWidth() {
		final int width[] = new int[1];
		Device.syncExec(new Runnable() {
			public void run() {
				width[0] = getWidthImpl(fHandle);
			}
		});
		return width[0];
	}

	boolean isObscured() {
		return Device.gObscured;
	}

	void bringToTop() {
		Device.asyncExec(new Runnable() {
			public void run() {
				bringToTopImpl(fHandle);
			}
		});
	}

	void create() {
		final DisplayPeer peer = this;
		Device.syncExec(new Runnable() {
			public void run() {
				fHandle = createImpl(Device.getWindowClassName(), 0, 0, Device.getShellWidth(), Device.getShellHeight());
				Device.addEventListener(peer, fHandle);
			}
		});

		int largestDimension = Math.max(Device.getShellWidth(), Device.getShellHeight());
		fOffscreenBuffer = Image.createImage(largestDimension, largestDimension);
	}

	void open() {
		fIsOpen = true;
		Device.syncExec(new Runnable() {
			public void run() {
				openImpl(fHandle, Device.gMessageWindow);

				/**
				 * This only really needs to be done for the first displayable,
				 * but we'll let the app manager handle the logic.
				 */
				Device.disposeSplashScreen();
			}
		});
	}

	void dispose() {
		fIsOpen = false;

		/*
		 * If there is a popup window, it may be disposed when
		 * it's parent's window is disposed. So - don't assume
		 * we need to loop "fWindows.size()" times.
		 */
		while (fWindows.size() > 0) {
			/*
			 * Accessing the element at 0 is required because the elements
			 * remove themselves from the vector when disposed.
			 */
			((Window) fWindows.elementAt(0)).dispose();
		}

		final DisplayPeer peer = this;
		Device.syncExec(new Runnable() {
			public void run() {
				if (fHandle != 0) {
					Device.removeEventListener(peer, fHandle);
					disposeImpl(fHandle);
					fHandle = 0;
				}
			}
		});
	}

	boolean close() {
		if (fClosing) return false;

		fClosing = true;
		final MIDlet midlet = MIDletManager.getMIDlet(fDisplay);
		if (midlet == null) return false;

		/*
		 * Fix for the case where close() is being called as a result
		 * of the serviceRepaints() method flushing the queue from another
		 * thread. In this case, the user code called in destroyApp()
		 * can hang the app.
		 */
		Device.postRunnable(new Runnable() {
			public void run() {
				AppManager.gAppManager.midletDestroyed(midlet);
			}
		});
		return false;
	}

	boolean handleResize(Event e) {
		int[] sizes = new int[2];
		getWindowDimensionsImpl(fHandle, sizes);
		int newShellWidth = sizes[0];
		int newShellHeight = sizes[1];

		if (Device.gDisplayTotalWidth != newShellWidth || Device.gDisplayTotalHeight != newShellHeight) {

			// The screen orientation has changed, if this is a midp 1.0 midlet
			// then restart the midlet
		  	if (MIDletManager.isMidp10()) {
				  final MIDlet midlet = MIDletManager.getMIDlet(fDisplay);

				  if (fCurrentWindow != null && fCurrentWindow.fShown) {
					  if (Device.requiresMidletRelaunch(fCurrentWindow.getWindowHandle(), MidpMsg.getString("Device.amsMidletClose"), MidpMsg.getString("Device.nonAmsMidletRelaunch"), MIDletManager.gMidletSuiteName + MIDletManager.gVendorName)) {
						  MIDletManager.gReLaunchMidlet = true;
					  } else {
						  MIDletManager.gReLaunchMidlet = false;
					  }

					  Device.postRunnable(new Runnable() {
						  public void run() {
							  AppManager.gAppManager.midletDestroyed(midlet);
						  }
					  });
					  return false;
				  }
		 	 }

			/*
			 * The screen orientation has changed.
			 * Get the accurate Display area dimensions by creating a new Window rather than
			 * believing the existing one which could be wrong if a Scroll bar exists.
			 */
			getDisplayAreaImpl(sizes, newShellWidth, newShellHeight);
			Device.gDisplayTotalWidth = newShellWidth;
			Device.gDisplayTotalHeight = newShellHeight;
			Device.gDisplayClientAreaWidth = sizes[0];
			Device.gDisplayClientAreaHeight = sizes[1];
		}

		if (fCurrentWindow != null) fCurrentWindow.resize(e.fX, e.fY);
		return false;
	}

	void paint() {
		Enumeration e = fWindows.elements();
		while (e.hasMoreElements()) {
			((Window) e.nextElement()).repaint();
		}
	}

	public boolean dispatchEvent(Event e) {
		switch (e.fType) {
			/*
			 * Event.CHARACTER generated for SmartPhone when
			 * we choose a character from symbol window.
			 */
			case Event.CHARACTER:
				if (fCurrentWindow != null && fCurrentWindow.fContentComponent != null) {
					Component focusComponent = fCurrentWindow.fContentComponent.fFocusComponent;
					if (focusComponent != null && focusComponent instanceof ItemComponent) {
						ItemComponent item = (ItemComponent) fCurrentWindow.fContentComponent.fFocusComponent;
						if (item.fContentComponent instanceof TextPeer) {
							((TextPeer) item.fContentComponent).sendWMCharMessage(e);
							return true;
						}
					}
				}
				return false;
			case Event.KEY_PRESSED:
				if (fCurrentWindow != null && fCurrentWindow.isShown()) {
					return fCurrentWindow.keyPressed(e.fKeyCode);
				}
				return false;
			case Event.KEY_RELEASED:
				if (fCurrentWindow != null && fCurrentWindow.isShown()) {
					return fCurrentWindow.keyReleased(e.fKeyCode);
				}
				return false;
			case Event.KEY_REPEATED:
				if (fCurrentWindow != null && fCurrentWindow.isShown()) {
					return fCurrentWindow.keyRepeated(e.fKeyCode);
				}
				return false;
			case Event.WINDOW_ACTIVATE:
				if (Device.gEnablePauseBehavior) AppManager.gAppManager.resumeRequest(MIDletManager.getMIDlet(fDisplay));
				if (fCurrentWindow != null) {
					Device.gDisplayableUpdateRunnable = fCurrentWindow.fUpdateRunnable;

					/*
					 * This will set the focus to whatever component
					 * had the focus when the window was deactivated.
					 */
					if (fCurrentWindow.isShown()) fCurrentWindow.setInitialFocus();
				}
				return false;
			case Event.WINDOW_DEACTIVATE:
				if (fCurrentWindow != null && fCurrentWindow instanceof PopupChoice) {
					((PopupChoice) fCurrentWindow).cancel();
				}
				if (Device.gEnablePauseBehavior) AppManager.gAppManager.pauseApp(MIDletManager.getMIDlet(fDisplay));
				return false;
			case Event.WINDOW_RESIZE:
				return handleResize(e);
			case Event.WINDOW_PAINT:
				if (fCurrentWindow != null) fCurrentWindow.fGraphics.flush();
				return false;
			case Event.WINDOW_CLOSED:
				return close();
			case Event.VERTICAL_SCROLL:
				if (fCurrentWindow != null && fCurrentWindow.isShown()) {
					((DisplayablePeer) fCurrentWindow).fContentComponent.scrollTo(0, e.fScrollOrSelectionIndex);
				}
				return false;
			case Event.EVENT_STOP:
				return close();
			case Event.BACK:
				if (fCurrentWindow != null && fCurrentWindow.fContentComponent != null) {
					Component focusComponent = fCurrentWindow.fContentComponent.fFocusComponent;
					if (focusComponent != null && focusComponent instanceof ItemComponent) {
						ItemComponent item = (ItemComponent) fCurrentWindow.fContentComponent.fFocusComponent;
						if (item.fContentComponent instanceof TextPeer) {
							((TextPeer) item.fContentComponent).backKeyPressed(e);
							return true;
						}
					}
				}
				if (!Device.gDisableBackHotKey) {
					navigateBackImpl();
					/* The user has given focus to another application.
					 * Currently, we're choosing to exit the midlet
					 * in response to this event because we can only
					 * run one midlet at a time. We need to reinvestigate
					 * when multiple midlets can be run simultaneously.
					 */
					return close();
				}
			default:
				if (fCurrentWindow != null && fCurrentWindow.isShown()) {
					fCurrentWindow.dispatchEvent(e);
				}
				return false;
		}
	}

	public boolean dispatchPointerEvent(Event e) {
		switch (e.fType) {
			case Event.POINTER_PRESSED:
				fMousePressed = true;
				if (Device.canGrabPointer()) Device.gCanUpdate = false;
				if (fCurrentWindow != null && fCurrentWindow.isShown()) {
					return fCurrentWindow.pointerPressed(e.fX, e.fY);
				}
				break;
			case Event.POINTER_RELEASED:
				fMousePressed = false;
				if (Device.canGrabPointer()) Device.gCanUpdate = true;
				if (fCurrentWindow != null && fCurrentWindow.isShown()) {
					return fCurrentWindow.pointerReleased(e.fX, e.fY);
				}
				break;
			case Event.POINTER_MOVED:
				if (!fMousePressed) return false;
				if (fCurrentWindow != null && fCurrentWindow.isShown()) {
					return fCurrentWindow.pointerDragged(e.fX, e.fY);
				}
				break;
		}

		return false;
	}

	public int getX() {
		final int ret[] = new int[1];
		Device.syncExec(new Runnable() {
			public void run() {
				ret[0] = getXImpl(fHandle);
			}
		});
		return ret[0];
	}

	public int getY() {
		final int ret[] = new int[1];
		Device.syncExec(new Runnable() {
			public void run() {
				ret[0] = getYImpl(fHandle);
			}
		});
		return ret[0];
	}

	Displayable getDisplayable() {
		return fDisplayable;
	}

	static native void initImpl();
	native int createImpl(String className, int x, int y, int width, int height);
	native void disposeImpl(int handle);
	native void openImpl(int handle, int messageWindowHandle);
	native void bringToTopImpl(int handle);
	native int getXImpl(int handle);
	native int getYImpl(int handle);
	native int getWidthImpl(int handle);
	native int getHeightImpl(int handle);
	native void navigateBackImpl();
	native void getWindowDimensionsImpl(int handle, int[] sizes);
	native void getDisplayAreaImpl(int[] sizes, int shellWidth, int shellHeight);
}
