package javax.microedition.lcdui;

import com.ibm.ive.midp.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

public abstract class Displayable {

	DisplayablePeer fPeer; /* Accessed by INL. Do not move. */

	static final int TYPE_CANVAS = 0;
	static final int TYPE_LIST = 1;
	static final int TYPE_FORM = 2;
	static final int TYPE_ALERT = 3;
	static final int TYPE_TEXTBOX = 4;
	static final int TYPE_DATEEDITOR = 5;
	static final int TYPE_TIMEEDITOR = 6;
	static final int TYPE_MENU = 7;
	static final int TYPE_POPUP_CHOICE = 8;
	static final int TYPE_POPUP_MENU = 9;
	static final int TYPE_POPUP_PHONE_MENU = 10;

	// The midlet-defined commands on this Displayable.
	// Any implementation-specific default commands are not added to this FastVector.
	FastVector fCommands;

	// The command listener
	CommandListener fCommandListener;

	// The title
	String fTitle;
	String fDisplayReadyTitle;

	// The optional ticker
	Ticker fTicker;

	Displayable(String title) {
		fTitle = title;
		fDisplayReadyTitle = null;
	}

	abstract int getDisplayableType();

	public void addCommand(Command command) {
		synchronized (Device.gDisplayableLock) {
			_addCommand(command);
		}
	}

	void _addCommand(Command command) {
		if (command == null) throw new NullPointerException();

		if (fCommands == null) {
			fCommands = new FastVector();
		} else if (fCommands.contains(command)) {
			/* Adding a command to the displayable twice should do nothing */
			return;
		}

		/*
		 * Insert the command into the list of commands based on priority.
		 */
		int insertionIndex = -1;
		for (int i = 0; i < fCommands.size(); i++) {
			if (command.getPriority() < ((Command)fCommands.elementAt(i)).getPriority()) {
				fCommands.insertElementAt(command, i);
				insertionIndex = i;
				break;
			}
		}
		if (insertionIndex == -1){
			insertionIndex = fCommands.size();
			fCommands.addElement(command);
		}

		/* Need to check for a peer and update it if needed */
		if (fPeer != null) fPeer.addCommand(command, insertionIndex);
	}

	public Ticker getTicker() {
		synchronized (Device.gDisplayableLock) {
			return fTicker;
		}
	}

	public String getTitle() {
		synchronized (Device.gDisplayableLock) {
			return fTitle;
		}
	}

	public int getHeight() {
		synchronized (Device.gDisplayableLock) {
			/*
			 * 1.0 MIDlets need to have a constant value returned from
			 * getHeight().  This is because sizeChanged() is new in 2.0.
			 */

			if (fPeer != null) return fPeer.getHeight();
			return DisplayablePeer.getDisplayableHeight();
		}
	}

	public int getWidth() {
		synchronized (Device.gDisplayableLock) {
			if (fPeer != null) return fPeer.getWidth();
			return DisplayablePeer.getDisplayableWidth();
		}
	}

	public boolean isShown() {
		synchronized (Device.gDisplayableLock) {
			return _isShown();
		}
	}

	boolean _isShown() {
		return fPeer != null && fPeer.fShown && !fPeer.fDisplay.fPeer.isObscured();
	}

	public void removeCommand(Command command) {
		synchronized (Device.gDisplayableLock) {
			if (command == null) return;
			if (fCommands == null) return;

			/*
			 * Only remove commands that are actually contained in the
			 * displayable so that the peer will never get told to remove
			 * a command that doesn't exist.
			 */
			int index = fCommands.indexOf(command);
			if (index != -1) {
				fCommands.removeElementAt(index);
				/* Need to check for a peer and remove the command */
				if (fPeer != null) fPeer.removeCommand(command, index);
			}
		}
	}

	public void setCommandListener(CommandListener listener) {
		synchronized (Device.gDisplayableLock) {
			fCommandListener = listener;
		}
	}

	public void setTicker(Ticker ticker) {
		synchronized (Device.gDisplayableLock) {
			fTicker = ticker;
			if (fPeer != null) fPeer.setTicker(ticker);
		}
	}

	public void setTitle(String title) {
		synchronized (Device.gDisplayableLock) {
			if (fTitle == title || (title != null && title.equals(fTitle))) return;

			fTitle = title;
			fDisplayReadyTitle = null; // force it to be rebuilt only when needed
			if (fPeer != null) fPeer.setTitle(title == null ? null : getDisplayReadyTitle());
		}
	}

	/*
	 * The String passed to setTitle() is not necessarily
	 * displayable as is.  If it is null, it should be
	 * displayed as an empty String.  If it contains line
	 * breaks, only the first line should be displayed.
	 */
	String getDisplayReadyTitle() {
		if (fDisplayReadyTitle != null) return fDisplayReadyTitle;

		if (fTitle == null || fTitle.length() == 0) {
			// On the PocketPC if the string is "" then the OS treats it
			// just like NULL and shows the "Start" title instead of a blank
			fDisplayReadyTitle = " "; //$NON-NLS-1$
		} else {
			int length = fTitle.length();
			boolean found = false;
			int index;

			for (index = 0; index < length; index++) {
				char c = fTitle.charAt(index);
				if (c == '\r' || c == '\n') {
					found = true;
					break;
				}
				index++;
			}

			if (found) {
				if (index == 0) {
					// On the PocketPC if the string is "" then the OS treats it
					// just like NULL and shows the "Start" title instead of a blank
					fDisplayReadyTitle = " ";
				} else {
					fDisplayReadyTitle = fTitle.substring(0, index);
				}
			} else {
				fDisplayReadyTitle = fTitle;
			}
		}

		return fDisplayReadyTitle;
	}

	protected void sizeChanged(int width, int height) {
		// Accorrding to the spec, this method MUST be empty in
		// the implementation.  This method only exsists so that
		// it can be overridden in subclasses of Form and Canvas.
	}

	int numApplicationCommands() {
		if (fCommands == null) return 0;
		return fCommands.size();
	}

	FastVector getAllCommands() {
		return fCommands;
	}

	void sendCommand(final Command command) {
		final Displayable d = this;
		Device.postRunnable(new Runnable() {
			public void run() {
				if (fCommandListener != null) {
					try {
						fCommandListener.commandAction(command, d);
					} catch (Throwable e) {
						System.err.println(MidpMsg.getString("Displayable.sendCommand.CommandError", command.fLabel)); //$NON-NLS-1$
						e.printStackTrace();
					}
				}
			}
		});
	}

	boolean hasCommandListener() {
		return fCommandListener != null;
	}

	FastVector getSortedCommands(FastVector allCommands, int[] commandTypes) {
		FastVector sortedCommands = new FastVector();

		for (int i = 0; i < allCommands.size(); i++) {
			// Get the next command
			Command command = (Command) allCommands.elementAt(i);

			// Determine whether this command fits into the
			// list of valid command types
			boolean shouldBeAdded = false;
			if (commandTypes == null) {
				shouldBeAdded = true;
			} else {
				for (int a = 0; a < commandTypes.length; a++) {
					if (command.getCommandType() == commandTypes[a]) {
						shouldBeAdded = true;
						break;
					}
				}
			}

			// Add the command into the proper location in the vector
			if (shouldBeAdded) {
				int pos = 0;
				for (int c = 0; c < sortedCommands.size(); c++, pos++) {
					Command tempCommand = (Command) sortedCommands.elementAt(c);
					if (command.getPriority() < tempCommand.getPriority()) break;
				}

				sortedCommands.insertElementAt(command, pos);
			}

		}
		return sortedCommands;
	}

	Display getDisplay() {
		return fPeer.fDisplay;
	}

	void changePeerSize(int newWidth, int newHeight) {
		if (fPeer == null || fPeer.fDisposed) return;
		fPeer.resize(newWidth, newHeight);
	}

	void showNotify() {
		/* No-op */
	}

	void hideNotify() {
		/* No-op */
	}

	void callShowNotify() {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}
		try {
			showNotify();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	void callHideNotify() {
		if (Device.DEBUG_SERIAL_QUEUE) {
			if (Thread.currentThread() != Device.gSerialThread) throw new IllegalStateException();
		}
		try {
			hideNotify();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
