package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.util.Hashtable;
import com.ibm.ive.midp.*;
import com.ibm.ive.midp.util.JadParser;

public final class Font {

	FontData fData = new FontData();
	Object fLock = new Object();

	/** Constants for FACE_SYSTEM = 0 */
	public static final int FACE_SYSTEM = 0;

	/** Constants for FACE_MONOSPACE = 32 */
	public static final int FACE_MONOSPACE = 32;

	/** Constants for FACE_PROPORTIONAL = 64 */
	public static final int FACE_PROPORTIONAL = 64;

	/** Constants for SIZE_MEDIUM = 0 */
	public static final int SIZE_MEDIUM = 0;

	/** Constants for SIZE_SMALL = 8 */
	public static final int SIZE_SMALL = 8;

	/** Constants for SIZE_LARGE = 16 */
	public static final int SIZE_LARGE = 16;

	/** Constants for STYLE_PLAIN = 0 */
	public static final int STYLE_PLAIN = 0;

	/** Constants for STYLE_BOLD = 1 */
	public static final int STYLE_BOLD = 1;

	/** Constants for STYLE_ITALIC = 2 */
	public static final int STYLE_ITALIC = 2;

	/** Constants for STYLE_UNDERLINED = 4 */
	public static final int STYLE_UNDERLINED = 4;

	private static final int VALID_STYLES = (STYLE_BOLD | STYLE_ITALIC | STYLE_UNDERLINED);

	/**
	 * Font specifier used for drawing user text input.
	 *
	 * @since 2.0
	 */
	public static final int FONT_INPUT_TEXT = 1;

	/**
	 * Font specifier used for drawing Screen and Item text.
	 *
	 * @since 2.0
	 */
	public static final int FONT_STATIC_TEXT = 0;

	static final String LCDUI_FONT_FILE_SPEC = "LcduiFontFileSpec";
	static final String LCDUI_MONOSPACE_FONT = "LcduiMonospaceFont";
	static final String LCDUI_MONOSPACE_BOLD_FONT = "LcduiMonospaceBoldFont";
	static final String LCDUI_PROPORTIONAL_FONT = "LcduiProportionalFont";
	static final String LCDUI_PROPORTIONAL_BOLD_FONT = "LcduiProportionalBoldFont";
	static final String LCDUI_SYSTEM_FONT = "LcduiSystemFont";
	static final String LCDUI_SYSTEM_BOLD_FONT = "LcduiSystemBoldFont";

	static String gResolvedAC3FontPath;
	static String gMonospaceFontPath;
	static String gMonospaceBoldFontPath;
	static String gProportionalFontPath;
	static String gProportionalBoldFontPath;
	static String gSystemFontPath;
	static String gSystemBoldFontPath;

	static native String resolveFontFile(String directory);
	static native void displayFontError(String title, String message);
	static {
		String fontDir = System.getProperty(LCDUI_FONT_FILE_SPEC);
		if (fontDir != null) {
			int index = fontDir.indexOf('*');
			if (index < 0) {
				displayFontError(MidpMsg.getString("Font.ErrorBoxTitle"), MidpMsg.getString("Font.InvalidPath"));
				throw new IllegalArgumentException(MidpMsg.getString("Font.InvalidPath"));
			}
			String fileName = resolveFontFile(fontDir);
			if (fileName != null) {
				String directory = fontDir.substring(0, index);
				gResolvedAC3FontPath = directory + fileName;
			}
		}
		Hashtable font = null;
		try {
			font = JadParser.parse(Device.gFontPropertiesPath);
		} catch (NullPointerException npe) {
			displayFontError(MidpMsg.getString("Font.ErrorBoxTitle"), MidpMsg.getString("Font.NotAvailable"));
		}
		try {
			gMonospaceFontPath = font.get("LcduiMonospaceFont").toString();
			gMonospaceBoldFontPath = font.get("LcduiMonospaceBoldFont").toString();
			gProportionalFontPath = font.get("LcduiProportionalFont").toString();
			gProportionalBoldFontPath = font.get("LcduiProportionalBoldFont").toString();
			gSystemFontPath = font.get("LcduiSystemFont").toString();
			gSystemBoldFontPath = font.get("LcduiSystemBoldFont").toString();
		} catch (Exception npe) {
			displayFontError(MidpMsg.getString("Font.ErrorBoxTitle"), MidpMsg.getString("Font.InvalidPath"));
			throw new RuntimeException();
		}
	}

	private int fFace;
	private int fSize;
	private int fStyle;
	private int fHeight;
	private int fBaselinePosition;

	static Font[] gFonts = new Font[88];

	static final Font STANDARD_FONT = getFont(FACE_SYSTEM, STYLE_PLAIN, SIZE_MEDIUM);
	static final Font TITLE_FONT = getFont(FACE_SYSTEM, STYLE_BOLD, SIZE_MEDIUM);
	static final Font MENU_FONT = STANDARD_FONT;

	static String getFontFileName(int face, int style) {
		if (gResolvedAC3FontPath != null) return gResolvedAC3FontPath;

		if (face == FACE_MONOSPACE) {
			if ((style & STYLE_BOLD) != 0) {
				if (gMonospaceBoldFontPath != null) return gMonospaceBoldFontPath;
			}
			if (gMonospaceFontPath != null) return gMonospaceFontPath;
		} else if (face == FACE_PROPORTIONAL) {
			if ((style & STYLE_BOLD) != 0) {
				if (gProportionalBoldFontPath != null) return gProportionalBoldFontPath;
			}
			if (gProportionalFontPath != null) return gProportionalFontPath;
		}

		if ((style & STYLE_BOLD) != 0) {
			if (gSystemBoldFontPath != null) return gSystemBoldFontPath;
		}
		return gSystemFontPath;
	}

	Font(int face, int style, int size) {
		fData.fHandle = createNewFont(getFontFileName(face, style), face, style, size);
		if (fData.fHandle == Constants.ERROR_OUT_OF_MEMORY) throw new OutOfMemoryError();
		if (fData.fHandle <= 0) {
			displayFontError(MidpMsg.getString("Font.ErrorBoxTitle"), MidpMsg.getString("Font.InvalidPath"));
			throw new RuntimeException();
		}
		fStyle = style;
		fHeight = getHeightImpl();
		fBaselinePosition = getBaselinePositionImpl();
		fFace = face;
		fSize = size;
	}

	/**
	 * Compute the width of chars given the receiver's face, size and style.
	 *
	 * @return int width of characters in <code>chars</code>
	 * between <code>offset</code> and <code>offset+length</code>.
	 *
	 * @throws NullPointerException if <code>charArray</code> is null
	 * @throws ArrayIndexOutOfBoundsException if <code>offset</code>
	 * and <code>length</code> define an invalid range
	 */
	public int charsWidth(char[] chars, int offset, int length) {
		if (chars == null) throw new NullPointerException();

		/*
		 * The TCK test passes offset = Integer.MAX_VALUE and length = 1,
		 * so the sum is actually negative, causing the (offset+length)
		 * check to pass.  Test should be excluded because the spec implies
		 * that we need only check for ((offset+length) > chars.length), not
		 * (offset > chars.length) and (length > chars.length).
		 *
		 * The test really should not pass offsets which bring boundary
		 * conditions of int math into play.
		 */

		if (offset < 0 || length < 0 || offset > chars.length || length > chars.length || (offset+length) > chars.length) throw new ArrayIndexOutOfBoundsException();

		synchronized (fLock) {
			if (isDisposed()) return length * fHeight; // zero causes problems for some MIDlets
			return getStringWidthImpl(new String(chars, offset, length));
		}
	}

	/**
	 * Return the advance width of ch given the receiver's face,
	 * size and style.
	 *
	 * @return int width of character in the receiver.
	 */
	public int charWidth(char ch) {
		synchronized (fLock) {
			if (isDisposed()) return fHeight; // zero might cause problem in some MIDlets

			return getStringWidthImpl(String.valueOf(ch));
		}
	}

	/**
	 * Returns the receiver's baseline position in pixels.
	 *
	 * @return int baseline position (in pixels) of the receiver.
	 */
	public int getBaselinePosition() {
		/*
		 * Some platforms require internal font apis to answer different values than the public
		 * apis. Therefore, we need methods for public and internal font metrics. On this
		 * platform, they return the same values.
		 */
		return _getBaselinePosition();
	}

	int _getBaselinePosition() {
		return fBaselinePosition;
	}

	/**
	 * Return the system default <code>Font</code>.
	 *
	 * @return Font the system's default Font
	 **/
	public static Font getDefaultFont() {
		return STANDARD_FONT;
	}

	/**
	 * Return the receiver's face.
	 *
	 * @return int the receiver's face.
	 * @see Font#FACE_MONOSPACE
	 * @see Font#FACE_PROPORTIONAL
	 * @see Font#FACE_SYSTEM
	 **/
	public int getFace() {
		return fFace;
	}

	/**
	 * Return the Font corresponding to the font specifier.
	 *
	 * @param specifier either FONT_INPUT_TEXT or FONT_STATIC_TEXT
	 *
	 * @throws IllegalArgumentException if specifier is invalid
	 *
	 * @since 2.0
	 */
	public static Font getFont(int specifier) {
		if (!(specifier == FONT_INPUT_TEXT || specifier == FONT_STATIC_TEXT)) throw new IllegalArgumentException();

		return STANDARD_FONT;
	}

	/**
	 * Creates a <code>Font</code> instance using the given
	 * <code>face</code>, <code>style</code> and <code>size</code>.<p>
	 * <code>face</code> must be <code>FACE_MONOSPACE</code>,
	 * <code>FACE_PROPORTIONAL</code> or <code>FACE_SYSTEM</code>.<p>
	 * <code>style</code> must be a valid combination of
	 * <code>STYLE_BOLD</code>, <code>STYLE_ITALIC</code>,
	 * <code>STYLE_UNDERLINED</code>.<p>
	 * <code>size</code> must be one of <code>SIZE_SMALL</code>,
	 * <code>SIZE_MEDIUM</code> or <code>SIZE_LARGE</code>.
	 *
	 * @throws IllegalArgumentException when either <code>face</code>, <code>style</code> or <code>size</code>
	 * is invalid
	 *
	 * @return Font according to parameters specified
	 */
	public static Font getFont(int face, int style, int size) {
		if ((!(face == FACE_MONOSPACE || face == FACE_PROPORTIONAL || face == FACE_SYSTEM)) ||
			(!(style == STYLE_PLAIN || (style > 0 && style <= VALID_STYLES))) ||
			(!(size == SIZE_LARGE || size == SIZE_MEDIUM || size == SIZE_SMALL))) {
			throw new IllegalArgumentException();
		}

		int key = face + style + size;
		if (gFonts[key] != null) return gFonts[key];

		return gFonts[key] = new Font(face, style, size);
	}

	/**
	 * Return the line height in pixels of the receiver.
	 *
	 * @return int height in pixels of the receiver
	 **/
	public int getHeight() {
		/*
		 * Palm requires internal font apis to answer different values than the public
		 * apis. Therefore, we need methods for public and internal font metrics. On
		 * this platform, they return the same values.
		 */
		return _getHeight();
	}

	int _getHeight() {
		return fHeight;
	}

	/**
	 * Return the size of the receiver.
	 *
	 * @return int the size of the receiver.
	 * @see Font#SIZE_SMALL
	 * @see Font#SIZE_MEDIUM
	 * @see Font#SIZE_LARGE
	 **/
	public int getSize() {
		return fSize;
	}

	/**
	 * Returns the receiver's style. Return values are
	 * among <code>STYLE_PLAIN, STYLE_BOLD, STYLE_ITALIC,
	 * STYLE_UNDERLINED</code>.
	 *
	 * @return int the style of the receiver
	 */
	public int getStyle() {
		return fStyle;
	}

	/**
	 * Returns whether the receiver is a bold font.
	 *
	 * @return boolean true if the receiver is in bold style,
	 * false otherwise.
	 **/
	public boolean isBold() {
		return (fStyle & STYLE_BOLD) != 0;
	}

	/**
	 * @return boolean true if the receiver is in italic style,
	 * false otherwise.
	 */
	public boolean isItalic() {
		return (fStyle & STYLE_ITALIC) != 0;
	}

	/**
	 * @return boolean true if the receiver is in plain style,
	 * false otherwise.
	 */
	public boolean isPlain() {
		return fStyle == STYLE_PLAIN;
	}

	/**
	 * @return boolean true if the receiver is in underlined style,
	 * false otherwise.
	 */
	public boolean isUnderlined() {
		return (fStyle & STYLE_UNDERLINED) != 0;
	}

	/**
	 * Returns the width in pixels of <code>string</code> if
	 * rendered with the receiver.
	 *
	 * @return int width (in pixels) of <code>string</code>
	 * in the receiver's face, style and and size.
	 *
	 * @throws NullPointerException if string is null
	 */
	public int stringWidth(String string) {
		if (string == null) throw new NullPointerException();
		synchronized (fLock) {
			if (isDisposed()) return string.length() * fHeight; // zero causes problems for some MIDlets
			return getStringWidthImpl(string);
		}
	}

	int getStringWidth(String string) {
		//TODO - Remove null check - our internal functions should never make this call with null.
		if (string == null) throw new IllegalArgumentException();
		return getStringWidthImpl(string);
	}

	native int getStringWidthImpl(String string);

	/**
	 * Computes the width in pixels of the substring of <code>string</code>.
	 * starting at <code>offset</code> and ending at
	 * <code>offset+length</code>.
	 *
	 * @return int width (in pixels) of <code>string</code>'s substring
	 * starting at <code>offset</code> and <code>length</code> characters long
	 * in the receiver's face, style and and size.
	 * @throws NullPointerException if <code>string</code> is null
	 * @throws StringIndexOutOfBoundsException if <code>offset</code> and <code>length</code>
	 * are out of bounds.
	 */
	public int substringWidth(String string, int offset, int length) {
		if (string == null) throw new NullPointerException();

		if (offset < 0 || length < 0 || (offset+length) > string.length()) throw new StringIndexOutOfBoundsException();

		synchronized (fLock) {
			if (isDisposed()) return length * fHeight; // zero causes problems for some MIDlets
			return getStringWidthImpl(string.substring(offset,offset+length));
		}
	}

	/**
	 * This method gets the number of characters at which the text should be wrapped
	 * @param text The base text
	 * @param startIndex The starting index in the base text
	 * @param lineWidth The width of the line in pixels on which the text will be displayed
	 * @return The number of characters that can be displayed within the width, or -1 if none
	 */
	int getCharsInLine(char[] text, int startIndex, int lineWidth) {
		return getCharsInLineImpl(text, startIndex, lineWidth, fData.fHandle);
	}

	native int getCharsInLineImpl(char[] text, int startIndex, int lineWidth, int fontHandle);

	int getNumberOfLines(String text, int lineWidth) {
		char[] textArray = text.toCharArray();
		int index = 0;
		int count = 0;
		for (int wrapIndex = 0; wrapIndex != -1 && index < textArray.length; count++, index += wrapIndex) {
			wrapIndex = getCharsInLine(textArray, index, lineWidth);
		}
		return count;
	}

	static void disposeAll() {
		if (gFonts != null) {
			for (int i = 0; i < gFonts.length; i++) {
				if (gFonts[i] != null) gFonts[i].dispose();
			}
			gFonts = null;
		}
		closeFontLibrary();
	}

	void dispose() {
		synchronized (fLock) {
			if (fData.fHandle != 0) {
				disposeImpl(fData.fHandle);
				fData.fHandle = 0;
			}
		}
	}

	boolean isDisposed() {
		return fData.fHandle == 0;
	}

	static native int createNewFont(String filePath, int face, int style, int size);
	native void disposeImpl(int handle);
	native int getBaselinePositionImpl();
	native int getHeightImpl();
	static native void closeFontLibrary();
}
