package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import com.ibm.ive.midp.*;

/**
 * Please refer to the MIDP 2.0 specification for documentation about this class.
 *
 * <p>Some additional comments:
 *
 * <p>It is not expected that the implementation will be threadsafe. Anyone wishing to use
 * a Graphics instance must acquire the library lock, call activate(), and perform all
 * drawing operations before releasing the library lock.
 *
 * <p>We add the restrictClip() and resetClipRestriction() calls. These calls allow the
 * LCDUI implementation to limit the values that the user can set for clipping.
 *
 * <p>We add a drawStringWithTabExtension call. This call is expected to expand tabs in the
 * string arg passed when drawing.
 *
 * <p>We add a dispose() call. This call must free any OS resources associated with the
 * Graphics instance.
 *
 * <p>Finally, it is important that new fields are added only after the existing fields
 * as these fields may be accessed directly from INL code.
 *
 * @author IBM
 */

public class Graphics {

	GraphicsData fData;
	Image fImage;
	Component fComponent;

	static final int GRAPHICS_REFERENCE_COUNT_THRESHOLD = 0x20; // MIDlets usually don't use more than 32 instances at once
	static FastVector gReferenceCache = new FastVector();

	public static final int HCENTER  =  1;
	public static final int VCENTER  =  2;
	public static final int LEFT     =  4;
	public static final int RIGHT    =  8;
	public static final int TOP      = 16;
	public static final int BOTTOM   = 32;
	public static final int BASELINE = 64;
	static final int VERTICAL_ANCHOR_FLAGS = 0x72; // Includes the bit sequence 01110010
	static final int HORIZONTAL_ANCHOR_FLAGS = 0x0D; // Includes the bit sequence 00001101
	static final int TWO_TIMES_BASELINE = 2 * BASELINE;

	public static final int DOTTED = 1;
	public static final int SOLID  = 0;

	static final int TOP_LEFT = TOP | LEFT;
	static final int VERTICAL_FLAGS_STRING = BASELINE | TOP | BOTTOM;
	static final int HORIZONTAL_FLAGS = HCENTER | LEFT | RIGHT;
	static final int VERTICAL_FLAGS_IMAGE = VCENTER | TOP | BOTTOM;

	int fRestrictedClipX;
	int fRestrictedClipY;
	int fRestrictedClipWidth;
	int fRestrictedClipHeight;

	Rectangle fClipRect;
	boolean fHasDisplayDestination;

	Graphics() {
		/* All new instances should go through this constructor */
		fData = new GraphicsData();
		fData.fStrokeStyle = Graphics.SOLID;
		fData.fTranslateX = 0;
		fData.fTranslateY = 0;
	}

	/**
	 * Create a graphics instance.
	 *
	 * It is expected that the instance will be initialized with the correct
	 * values as specified by the MIDP 2.0 spec.
	 *
	 * @return a new Graphics instance.
	 */
	Graphics(Image image) {
		this();
		createGraphicsContext(image);
		fImage = image;
		initSetColor(0x000000);
		initSetFont(Font.getDefaultFont());
		fClipRect = new Rectangle(0,0,0,0); // set below in setClip()
		resetClipRestriction();
		setClip(0, 0, image.fData.fWidth, image.fData.fHeight);
	}

	/**
	 * When using GraphicsThreadsafe, overridden by GraphicsThreadSafe#initSetColor()
	 */
	void initSetColor(int color) {
		setColor(color);
	}

	/**
	 * When using GraphicsThreadsafe, overridden by GraphicsThreadSafe#initSetFont()
	 */
	void initSetFont(Font font) {
		setFont(font);
	}

	/**
	 * Create a graphics instance which will draw to the displayable given.
	 *
	 * It is expected that the instance will be initialized with the correct
	 * values as specified by the MIDP 2.0 spec.
	 *
	 * @return a new Graphics instance.
	 */
	Graphics(Component c) {
		this(c.getOffscreenBuffer());
		fData.fDisplayHandle = createDisplayGraphics(c.getWindowHandle());
		fHasDisplayDestination = true;
		fComponent = c;
	}

	void createGraphicsContext(Image image) {
		disposeUnreferencedInstances();
		fData.fHandle = createGC(image.fData.fHandle);
		register();
	}

	/**
	 * This method must prepare the current Graphics instance for drawing.
	 * This may be a no-op on some platforms - others will have to set the
	 * current drawing context in the OS. In either case, the call is expected
	 * to ensure that all drawing properties (color, font, linestyle, etc) are
	 * correctly set.
	 */
	void activate() {
		activateImpl(fData.fHandle, fData.fFont.fData.fHandle);
	}

	/**
	 * Free any associated OS resources.
	 */
	void dispose() {
		synchronized (gReferenceCache) {
			for (int i = 0; i < gReferenceCache.size(); i++) {
				GraphicsReference next = (GraphicsReference) gReferenceCache.elementAt(i);
				if (next.get() == this) {
					gReferenceCache.removeElementAt(i);
					break;
				}
			}
		}

		synchronized (fImage.fLock) {
			disposeImpl(fData.fDisplayHandle, fData.fHandle);
			fData.fHandle = 0;
			fData.fDisplayHandle = 0;
		}
	}

	/**
	 * Makes the clipping region the intersection between the current
	 * clipping region and the passed rectangle.
	 */
	public void clipRect(int x, int y, int w, int h) {
		fClipRect = fClipRect.intersection(new Rectangle(x,y,w,h));
		x = fData.fTranslateX + fClipRect.x;
		y = fData.fTranslateY + fClipRect.y;
		w = fClipRect.width;
		h = fClipRect.height;

		if (x < 0) {
			w += x;
			x = 0;
		}
		if (y < 0) {
			h += y;
			y = 0;
		}
		if (x < fRestrictedClipX) {
			w = w - (fRestrictedClipX - x);
			x = fRestrictedClipX;
		}
		if (y < fRestrictedClipY) {
			h = h - (fRestrictedClipY - y);
			y = fRestrictedClipY;
		}

		w = Math.max(0, Math.min(w, fRestrictedClipX + fRestrictedClipWidth - x));
		h = Math.max(0, Math.min(h, fRestrictedClipY + fRestrictedClipHeight - y));
		setClipRectImpl(x,y,w,h);
	}

	/**
	 * Copy the contents of the area specified by (srcX,srcY,width,height)
	 * to (destX,destY) using anchor as the anchorPoint.
	 *
	 * @throws IllegalStateException if the Graphics object has the display device as its destination
     * @throws IllegalArgumentException if the copied area is larger than the source Image dimensions
	 *
	 * @since 2.0
	 */
	public void copyArea(int srcX, int srcY, int width, int height, int destX, int destY, int anchor) {
		if (hasDisplayDestination()) throw new IllegalStateException();
		if ((fData.fTranslateX + srcX) < 0 || (fData.fTranslateY + srcY) < 0) throw new IllegalArgumentException();
		if ((fData.fTranslateX + srcX + width) > fImage.fData.fWidth || (fData.fTranslateY + srcY + height) > fImage.fData.fHeight) throw new IllegalArgumentException();

		checkAnchorForImage(anchor);

		int posX = fData.fTranslateX + destX;
		int posY = fData.fTranslateY + destY;

		if ((anchor & HCENTER) != 0) {
			posX = posX - width / 2;
		} else if ((anchor & RIGHT) != 0) {
			posX = posX - width;
		}

		if ((anchor & VCENTER) != 0) {
			posY = posY - height / 2;
		} else if ((anchor & BOTTOM) != 0) {
			posY = posY - height;
		}

		copyAreaImpl(srcX, srcY, width, height, posX, posY);
	}

	/**
	 * Answer the height of the destination drawing area. This drawing area
	 * may be the display, a window, or an image.
	 *
	 * @return the height of the Graphics destination drawing area.
	 */
	int getHeight() {
		return fImage.fData.fHeight;
	}

	/**
	 * Answer the width of the destination drawing area. This drawing area
	 * may be the display, a window, or an image.
	 *
	 * @return the width of the Graphics destination drawing area.
	 */
	int getWidth() {
		return fImage.fData.fWidth;
	}

	/**
	 * Draw an arc.
	 *
	 * @param x the left bound of the rectangle containing the arc
	 * @param y the top bound of the rectangle containing the arc
	 * @param width the bounding rectangle's width
	 * @param height the bounding rectangle's height
	 * @param startAngle
	 * @param angle end angle
	 */
	public native void drawArc(int x, int y, int width, int height, int startAngle, int angle);

	/**
	 * Draw a character at location (x,y) using the specified anchor point.
	 * Use the current font and color.
	 *
	 * @param ch the character to draw
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param anchor the anchor point
	 *
	 * @throws IllegalArgumentException if anchor is invalid
	 */
	public void drawChar(char ch, int x, int y, int anchor) {
		drawString(String.valueOf(ch),x,y,anchor);
	}

	/**
	 * Draw some characters at (x,y) using the specified anchor point.
	 * Use the current font and color.
	 *
	 * @param chars the array which contains the characters to draw
	 * @param offset the start index into chars
	 * @param length the number of characters to draw
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param anchor the anchor point
	 *
	 * @throws NullPointerException if chars is null
	 * @throws ArrayIndexOutOfBoundsException if offset or length are invalid.
	 * @throws IllegalArgumentException if anchor is invalid
	 */
	public void drawChars(char[] chars, int offset, int length, int x, int y, int anchor) {
		/* chars.length below will throw the NullPointerException for us if chars is null */
		if (offset < 0 || length < 0 || (offset+length) > chars.length) throw new ArrayIndexOutOfBoundsException();

		drawString(new String(chars,offset,length),x,y,anchor);
	}

	/**
	 * Draw an Image at (x,y) using the specified anchor point.
	 * Any transparency or alpha information in the source Image
	 * will be respected.  If image is the same as the destination
	 * of this Graphics, the results are undefined.  Use copyArea()
	 * instead.
	 *
	 * @param x the x coordinate
     * @param y the y coordinate
	 * @param anchor the anchor point
	 *
	 * @throws NullPointerException if image is null
	 * @throws IllegalArgumentException if anchor is invalid
	 */
	public native void drawImage(Image image, int x, int y, int anchor);

	/**
	 * Draw a line between (x1,y1) and (x2,y2).
	 * Use the current color and stroke style.
	 *
	 * @param x1 the start x coordinate
	 * @param y1 the start y coordinate
	 * @param x2 the end x coordinate
	 * @param y2 the end y coordinate
	 */
	public native void drawLine(int x1, int y1, int x2, int y2);

	/**
	 * Draw a rectangle starting at (x,y) with the specified
	 * width and height.  The resulting rectangle will be
	 * (width + 1) pixels wide by (height + 1) pixels high.
	 * Both x and y must be >= 0.
	 *
	 * Use the current color and stroke style.
	 *
	 * @param x the x coordinate
	 * @param y the y coordinate
	 * @param width the width
	 * @param height the height
	 */
	public native void drawRect(int x, int y, int width, int height);

	/**
	 * Copy an area of image to the destination and transform the image
	 * data as specified.  The source image must not be the same as the
	 * destination Image of this Graphics.  Valid transforms are defined
	 * in the javax.microedition.lcdui.game.Sprite class.
	 *
	 * Any transparency or alpha information in the source Image
	 * will be respected.
	 *
	 * @param image the source Image
	 * @param srcX
	 * @param srcY
	 * @param width
	 * @param height
	 * @param transform
	 * @param destX
	 * @param destY
	 * @param anchor
	 *
	 * @throws NullPointerException if image is null
	 * @throws IllegalArgumentException if image is the same as the destination Image
	 * @throws IllegalArgumentException if transform is invalid
	 * @throws IllegalArgumentException if anchor is invalid
	 * @throws IllegalArgumentException if the copied area is larger than the source Image dimensions
	 *
	 * @since 2.0
	 */
	public void drawRegion(Image image, int srcX, int srcY, int width, int height, int transform, int destX, int destY, int anchor) {
		if (image == null) throw new NullPointerException();
		if (image.hasSameDestination(this)) throw new IllegalArgumentException();

		if (srcX < 0 || srcY < 0) throw new IllegalArgumentException();
		if (srcX+width > image.getWidth() || srcY+height > image.getHeight()) throw new IllegalArgumentException();
		if (transform < 0 || transform > 7) throw new IllegalArgumentException();

		checkAnchorForImage(anchor);

		if (width == 0 || height == 0) return;
		Image transformedImage = new Image(image, srcX, srcY, width, height, transform);
		drawImage(transformedImage, destX, destY, anchor);
		transformedImage.dispose();
	}

	/**
	 * Draw a set of ARGB values in a specified area.
	 *
	 * @param rgb int array of ARGB values
	 * @param offset start index of data to draw in rgb
	 * @param scanLength pixel offset between consecutive lines
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param processAlpha
	 *
	 * @throws ArrayIndexOutOfBoundsException the operation will attempt to access
	 * an invalid index of rgb
	 * @throws NullPointerException if rgb is null
	 *
	 * @since 2.0
	 */
	public void drawRGB(int[] rgb, int offset, int scanLength, int x, int y, int width, int height, boolean processAlpha) {
		if (rgb == null) throw new NullPointerException();

		/*
		 * This strange block is thanks to the TCK.
		 */
		if (width < 0 || height < 0) throw new ArrayIndexOutOfBoundsException();
		if (width == 0 || height == 0) return;
		if (offset < 0 || offset > rgb.length) throw new ArrayIndexOutOfBoundsException();
		int end = offset + height * scanLength;
		if (end < 0 || end > rgb.length) throw new ArrayIndexOutOfBoundsException();

		drawRGBImpl(rgb, offset, scanLength, x, y, width, height, processAlpha);
	}

	native void drawRGBImpl(int[] rgb, int offset, int scanLength, int x, int y, int width, int height, boolean processAlpha);

	/**
	 * Draw a rounded rectangle at (x,y) with the specified
	 * width and height.  Use arcWidth and arcHeight as the
	 * rounded corner dimensions.
	 *
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param arcWidth horizontal diameter of the corner arcs
	 * @param arcHeight vertical diameter of the corner arcs
	 */
	public native void drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight);

	/*
	 * This function is unneeded on this platform. On palm, however, it uses the OS round rect function
	 * rather than our own round rect rendering.
	 */
	void _drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		drawRoundRect(x, y, width, height, arcWidth, arcHeight);
	}

	/**
	 * Draw a String at (x,y) using the specified anchor point.
	 *
	 * @param string
	 * @param x
	 * @param y
	 * @param anchor
	 *
	 * @throws NullPointerException if string is null
	 * @throws IllegalArgumentException if anchor is invalid
	 */
	public void drawString(String string, int x, int y, int anchor) {
		if (string.length() == 0) return; /* throws NullPointerException if string is null */
		checkAnchorForString(anchor);
		drawStringImpl(string, x, y, anchor, fData.fFont.fData.fHandle, fData.fFont.isUnderlined());
	}

	/**
	 * Draw the given string with tab expansion. The drawString() call may
	 * or may not do this already, but this function is provided to ensure
	 * that the drawn widgets can render correctly.
	 *
	 * @param string
	 * @param x
	 * @param y
	 * @param anchor
	 */
	void drawStringWithTabExpansion(String string, int x, int y, int anchor) {
		if (string == null) throw new NullPointerException();
		drawStringWithTabExpansionImpl(string, x, y, anchor, fData.fFont.fData.fHandle, fData.fFont.isUnderlined());
	}

	/**
	 * Draws a substring of string of at (x,y) using the specified
	 * anchor point.  Start with the character at offset and continue
	 * length characters.  Offset is zero based.
	 *
	 * @param string
	 * @param offset
	 * @param length
	 * @param x
	 * @param y
	 * @param anchor
	 *
	 * @throws StringIndexOutOfBoundsException if offset and length specify an invalid range
	 * @throws NullPointerException if string is null
	 * @throws IllegalArgumentException if anchor is invalid
	 */
	public void drawSubstring(String string, int offset, int length, int x, int y, int anchor) {
		if ((offset + length) > string.length() || offset < 0 || length < 0) throw new StringIndexOutOfBoundsException(); // also throws NPE if string is null

		drawString(string.substring(offset,offset+length),x,y,anchor);
	}

	/**
	 * Fill an arc which occupies the rectangle (x,y,width,height),
	 * and begins at startAngle, continuing angle more degrees.
	 *
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param startAngle
	 * @param angle
	 */
	public native void fillArc(int x, int y, int width, int height, int startAngle, int angle);

	/**
	 * Fill a rectangle starting at (x,y) with dimensions width and height.
	 * Use the current color.
	 *
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 */
	public native void fillRect(int x, int y, int width, int height);

	/**
     * Fill a rounded rectangle at (x,y) with the specified
	 * width and height.  Use arcWidth and arcHeight as the
	 * rounded corner dimensions.
	 *
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param arcWidth horizontal diameter of the corner arcs
	 * @param arcHeight vertical diameter of the corner arcs
	 */
	public native void fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight);

	/*
	 * This function is unneeded on this platform. On palm, however, it uses the OS round rect function
	 * rather than our own round rect rendering.
	 */
	void _fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		fillRoundRect(x, y, width, height, arcWidth, arcHeight);
	}

	/**
	 * Fill a triangle which has vertices (x1,y1), (x2,y2), and (x3,y3).
	 * Use the current color.  Include the lines connecting the vertices.
	 *
	 * @param x1
	 * @param y1
	 * @param x2
	 * @param y2
	 * @param x3
	 * @param y3
	 */
	public native void fillTriangle(int x1, int y1, int x2, int y2, int x3, int y3);

	/**
	 * Blit the region specified by the coordinates to the screen.
	 *
	 * This method is only called internally on Graphics instances which render displayables.
	 *
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param arcWidth
	 * @param arcHeight
	 */
	void flush(int x, int y, int w, int h) {
		flushImpl(fData.fDisplayHandle, fData.fHandle, x, y, w, h);
	}

	/**
	 * Blit the entire buffer to the screen.
	 *
	 * This method is only called internally on Graphics instances which render displayables.
	 *
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param arcWidth
	 * @param arcHeight
	 */
	void flush() {
		flush(0, 0, fImage.fData.fWidth, fImage.fData.fHeight);
	}

	/**
	 * Blit the region specified by the rectangle to the screen.
	 *
	 * This method is only called internally on Graphics instances which render displayables.
	 *
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @param arcWidth
	 * @param arcHeight
	 */
	void flush(Rectangle rect) {
		flush(rect.x, rect.y, rect.width, rect.height);
	}

	/**
	 * Returns the receiver's clipping area height.
	 *
	 * @return the height of the clipping area
	 */
	public int getClipHeight() {
		return fClipRect.height;
	}

	/**
	 * Returns the receiver's clipping area width.
	 *
	 * @return the width of the clipping area
	 */
	public int getClipWidth() {
		return fClipRect.width;
	}

	/**
	 * Returns the top-left corner horizontal coordinate
	 * of the receiver's clipping area.
	 *
	 * @return the X coordinate of the clipping area
	 */
	public int getClipX() {
		return fClipRect.x;
	}

	/**
	 * Returns the top-left corner vertical coordinate
	 * of the receiver's clipping area.
	 *
	 * @return the Y coordinate of the clipping area
	 */
	public int getClipY() {
		return fClipRect.y;
	}

	/**
	 * Returns the receiver's drawing color.
	 *
	 * @return current drawing color
	 */
	public int getColor() {
		return fData.fColor;
	}

	/**
	 * Return the RGB value which will be displayed if color
	 * is requested.  Both the argument and the return value
	 * use the format 0x00RRGGBB.
	 *
	 * @param color
	 *
	 * @since 2.0
	 */
	public native int getDisplayColor(int color);

	/**
	 * Returns the receiver's current <code>Font</code>.
	 *
	 * @return current font
	 */
	public Font getFont() {
		return fData.fFont;
	}

	/**
	 * Returns a grayscale integer value
	 * matching the receiver's drawing color.
	 *
	 * @return current drawing grayscale
	 */
	public int getGrayScale() {
		return fData.fColor & 0xFF;
	}

	/**
	 * Returns the red component of the receiver's drawing color.
	 * @return the red component of the drawing color
	 */
	public int getRedComponent() {
		return (fData.fColor & 0x00FF0000) >> 16;
	}

	/**
	 * Returns the greem component of the receiver's drawing color.
	 * @return the green component of the drawing color
	 */
	public int getGreenComponent() {
		return (fData.fColor & 0x0000FF00) >> 8;
	}

	/**
	 * Returns the blue component of the receiver's drawing color.
	 * @return the blue component of the drawing color
	 */
	public int getBlueComponent() {
		return fData.fColor & 0x000000FF;
	}

	/**
	 * Returns the receiver's current stroke style.<p>
	 * The return value can either be <code>Graphics.SOLID</code>
	 * or <code>Graphics.DOTTED</code>.
	 *
	 * @return current stroke style.
	 */
	public int getStrokeStyle() {
		return fData.fStrokeStyle;
	}

	/**
	 * @return current X coordinate for the origin.
	 */
	public int getTranslateX() {
		/*
		 * If the graphics instance is created on a widget, answer the
		 * translation with respect to the widget's origin rather than
		 * the offscreen image origin. This ensures that a graphics
		 * instance drawing to the Canvas answers the translation
		 * according to spec.
		 */
		if (fComponent != null) return fData.fTranslateX - fComponent.getOriginX();
		return fData.fTranslateX;
	}

	/**
	 * @return current Y coordinate for the origin.
	 */
	public int getTranslateY() {
		/*
		 * If the graphics instance is created on a widget, answer the
		 * translation with respect to the widget's origin rather than
		 * the offscreen image origin. This ensures that a graphics
		 * instance drawing to the Canvas answers the translation
		 * according to spec.
		 */
		if (fComponent != null) return fData.fTranslateY - fComponent.getOriginY();
		return fData.fTranslateY;
	}

	/**
	 * Answer true if this Graphics instance draws to a Displayable, false if
	 * it draws on an Image.
	 *
	 * @return whether the Graphics instance draws to the display
	 */
	boolean hasDisplayDestination() {
		return fHasDisplayDestination;
	}

	/**
	 * Set the clipping area to a rectangle with the given coordinates.
	 */
	public void setClip(int x, int y, int w, int h) {
		fClipRect.x = x;
		fClipRect.y = y;
		fClipRect.width = w;
		fClipRect.height = h;

		x += fData.fTranslateX;
		y += fData.fTranslateY;

		if (x < 0) {
			w += x;
			x = 0;
		}
		if (y < 0) {
			h += y;
			y = 0;
		}
		if (x < fRestrictedClipX) {
			w = w - (fRestrictedClipX - x);
			x = fRestrictedClipX;
		}
		if (y < fRestrictedClipY) {
			h = h - (fRestrictedClipY - y);
			y = fRestrictedClipY;
		}

		w = Math.max(0, Math.min(w, fRestrictedClipX + fRestrictedClipWidth - x));
		h = Math.max(0, Math.min(h, fRestrictedClipY + fRestrictedClipHeight - y));

//		System.err.println("Graphics.setClip: " + x + "," + y + "," + w + "," + h);
		setClipRectImpl(x,y,w,h);
	}

	/**
	 * Set the drawing color to color.
	 */
	public native void setColor(int color);

	/**
	 * Set the drawing color to the red, green and blue value provided.
	 *
	 * @throws IllegalArgumentException if any of the argument values less than 0 or greater than 255
	 */
	public native void setColor(int r, int g, int b);

	/**
	 * Set the current font to font.  If font is null,
	 * this call is logically equivalent to setFont(Font.getDefaultFont()).
	 */
	public void setFont(Font font) {
		if (font == null) font = Font.getDefaultFont();
		if (fData.fFont == font) return;
		fData.fFont = font;
		setFontImpl(font.fData.fHandle);
	}

	/**
	 * Set the current drawing color to the equivalent of value.
	 *
	 * @throws IllegalArgumentException if value is less than 0 or greater than 255
	 */
	public native void setGrayScale(int value);

	/**
	 * Set the drawing stroke style.  Style must be either SOLID
	 * or DOTTED.
	 *
	 * @throws IllegalArgumentException if style is invalid
	 */
	public native void setStrokeStyle(int style);

	/**
	 * Translates the origin of this Graphics by (x,y).
	 */
	public void translate(int x, int y) {
		fData.fTranslateX += x;
		fData.fTranslateY += y;
		fClipRect.x -= x;
		fClipRect.y -= y;
	}

	private void checkAnchorForString(int anchor) {
		if (anchor == 0) return;

		// For Text drawing, if anchor is >= 128 or negative, it is invalid
		if (anchor >= TWO_TIMES_BASELINE || anchor < 0) throw new IllegalArgumentException();

		checkAnchorForVerticalFlags(anchor, true);
		checkAnchorForHorizontalFlags(anchor);
	}

	private void checkAnchorForVerticalFlags(int anchor, boolean checkForString) {
		int flagMask = anchor & VERTICAL_ANCHOR_FLAGS;
		if (flagMask == 0 || (flagMask != TOP && flagMask != BOTTOM && (checkForString ? flagMask != BASELINE : flagMask != VCENTER))) throw new IllegalArgumentException();
	}

	private void checkAnchorForHorizontalFlags(int anchor) {
		int flagMask = anchor & HORIZONTAL_ANCHOR_FLAGS;
		if (flagMask == 0 || (flagMask != HCENTER && flagMask != LEFT && flagMask != RIGHT)) throw new IllegalArgumentException();
	}

	private void checkAnchorForImage(int anchor) {
		if (anchor == 0) return;

		// For Images, if anchor is >= 64 or negative, it is invalid
		if (anchor >= BASELINE || anchor < 0) throw new IllegalArgumentException(); //BASELINE is not allowed.

		checkAnchorForVerticalFlags(anchor, false);
		checkAnchorForHorizontalFlags(anchor);
	}

	/**
	 * Restrict the clip region to the area given so that the public
	 * clip APIs cannot set a region larger than that given.
	 *
	 * @param x
	 * @param y
	 * @param w
	 * @param h
	 */
	void restrictClip(int x, int y, int w, int h) {
//		System.err.println("restrictClip: " + x + "," + y + "," + w + "," + h);
		int imageWidth = fImage._getWidth();
		int imageHeight = fImage._getHeight();
		fRestrictedClipX = Math.min(x, imageWidth);
		fRestrictedClipY = Math.min(y, imageHeight);
		fRestrictedClipWidth = Math.min(w, imageWidth - fRestrictedClipX);
		fRestrictedClipHeight = Math.min(h, imageHeight - fRestrictedClipY);
//		System.err.println("restrictClip actually set to: " + fRestrictedClipX + "," + fRestrictedClipY + "," + fRestrictedClipWidth + "," + fRestrictedClipHeight);
	}

	/**
	 * Reset the clip restriction so that the clip APIs can set the
	 * clip to be the entire destination area.
	 *
	 */
	void resetClipRestriction() {
		fRestrictedClipX = 0;
		fRestrictedClipY = 0;
		fRestrictedClipWidth = fImage.fData.fWidth;
		fRestrictedClipHeight = fImage.fData.fHeight;
	}

	void reset() {
		resetImpl(fData.fDisplayHandle);
	}

	static void disposeAll() {
		synchronized (gReferenceCache) {
			for (int size = gReferenceCache.size()-1; size >= 0; size--) {
				GraphicsReference ref = (GraphicsReference) gReferenceCache.elementAt(size);
				Graphics g = (Graphics) ref.get();
				if (g != null) {
					g.dispose();
				} else {
					dispose(ref);
				}
			}
		}
	}

	static void disposeUnreferencedInstances() {
		if (gReferenceCache.size() < GRAPHICS_REFERENCE_COUNT_THRESHOLD) return;
		synchronized (gReferenceCache) {
			for (int size = gReferenceCache.size() - 1; size >= 0; size--) {
				GraphicsReference next = (GraphicsReference) gReferenceCache.elementAt(size);
				if (next.get() == null) {
					gReferenceCache.removeElementAt(size);
					dispose(next);
				}
			}
		}
	}

	static void dispose(GraphicsReference ref) {
		disposeImpl(ref.fDisplayHandle, ref.fHandle);
	}

	void register() {
		synchronized (gReferenceCache) {
			gReferenceCache.addElement(new GraphicsReference(this));
		}
	}

	native void activateImpl(int handle, int fontHandle);
	native void copyAreaImpl(int srcX, int srcY, int w, int h, int destX, int destY);
	static native int createDisplayGraphics(int displayHandle);
	native int createGC(int handle);
	static native void disposeImpl(int displayGcHandle, int imageGcHandle);
	final native void drawStringImpl(String string, int x, int y, int anchor, int fontHandle, boolean underline);
	final native void drawStringWithTabExpansionImpl(String string, int x, int y, int anchor, int fontHandle, boolean underline);
	native void flushImpl(int displayGcHandle, int imageGcHandle, int x, int y, int w, int h);
	native void initGraphics(Font font, int clipX, int clipY, int clipW, int clipH);
	static native void resetImpl(int displayHandle);
	native void setClipRectImpl(int x, int y, int w, int h);
	native void setFontImpl(int fontHandle);
}
