package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

/*
 * Some graphics libraries limit the number of graphics objects available on an image.
 * If we're running on such a system, this class needs to map multiple MIDP graphics
 * instances to a single native graphics instance.
 */

import com.ibm.ive.midp.*;

class GraphicsThreadsafe extends Graphics {

	GraphicsThreadsafe(Image image) {
		super(image);
		disposeUnreferencedInstances();
	}

	void initSetColor(int color) {
		super.setColor(color);
	}

	void initSetFont(Font font) {
		super.setFont(font);
	}

	GraphicsThreadsafe(Window peer) {
		super(peer);
	}

	GraphicsThreadsafe(WindowContent c) {
		super(c);
	}

	/* These fields intentionally have a zero value when first used. */
	int fNativeClipX;
	int fNativeClipY;
	int fNativeClipWidth;
	int fNativeClipHeight;

	boolean isDisposed() {
		return fImage.fData.fHandle <= 0 || fData.fHandle == 0;
	}

	public void setColor(int color) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.setColor(color);
		}
	}

	public void setColor(int r, int g, int b) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.setColor(r,g,b);
		}
	}

	public void setClip(int x, int y, int w, int h) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();

			fClipRect.x = x;
			fClipRect.y = y;
			fClipRect.width = w;
			fClipRect.height = h;

			x += fData.fTranslateX;
			y += fData.fTranslateY;

			if (x < 0) {
				w += x;
				x = 0;
			}
			if (y < 0) {
				h += y;
				y = 0;
			}
			if (x < fRestrictedClipX) {
				w = w - (fRestrictedClipX - x);
				x = fRestrictedClipX;
			}
			if (y < fRestrictedClipY) {
				h = h - (fRestrictedClipY - y);
				y = fRestrictedClipY;
			}

			w = Math.max(0, Math.min(w, fRestrictedClipX + fRestrictedClipWidth - x));
			h = Math.max(0, Math.min(h, fRestrictedClipY + fRestrictedClipHeight - y));
//			System.err.println("setClipRectImpl: " + x + "," + y + "," + w + "," + h);

			fNativeClipX = x;
			fNativeClipY = y;
			fNativeClipWidth = w;
			fNativeClipHeight = h;

			setClipRectImpl(x,y,w,h);
		}
	}

	public void setFont(Font font) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.setFont(font);
		}
	}

	public void copyArea(int srcX, int srcY, int width, int height, int destX, int destY, int anchor) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.copyArea(srcX,srcY,width,height,destX,destY, anchor);
		}
	}

	/*
	 * Restore the GC with our settings if necessary.  This requires
	 * that the Image (referred through fImage) be locked.
	 */
	private void restoreSettings() {
	}

	public void drawArc(int x, int y, int width, int height, int startAngle, int endAngle) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.drawArc(x,y,width,height,startAngle,endAngle);
		}
	}

	public void drawImage(Image srcImage, int x, int y, int anchor) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.drawImage(srcImage,x,y,anchor);
		}
	}

	public void drawLine(int x1, int y1, int x2, int y2) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.drawLine(x1,y1,x2,y2);
		}
	}

	public void drawRect(int x, int y, int width, int height) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.drawRect(x,y,width,height);
		}
	}

	public void drawRegion(Image image, int srcX, int srcY, int width, int height, int transform, int destX, int destY, int anchor) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.drawRegion(image, srcX, srcY, width, height, transform, destX, destY, anchor);
		}
	}

	public void drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.drawRoundRect(x,y,width,height,arcWidth,arcHeight);
		}
	}

	void _drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super._drawRoundRect(x,y,width,height,arcWidth,arcHeight);
		}
	}

	public void drawString(String string, int x, int y, int anchor) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.drawString(string, x, y, anchor);
		}
	}

	public void fillArc(int x, int y, int width, int height, int startAngle, int endAngle) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.fillArc(x,y,width,height,startAngle,endAngle);
		}
	}

	public void fillRect(int x, int y, int width, int height) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.fillRect(x,y,width,height);
		}
	}

	public void fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.fillRoundRect(x,y,width,height,arcWidth,arcHeight);
		}
	}

	void _fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super._fillRoundRect(x,y,width,height,arcWidth,arcHeight);
		}
	}

	public void fillTriangle(int x1, int y1, int x2, int y2, int x3, int y3) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.fillTriangle(x1,y1,x2,y2,x3,y3);
		}
	}

	/**
	 * Makes the clipping region the intersection between the current
	 * clipping region and the passed rectangle.
	 */
	public void clipRect(int x, int y, int w, int h) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			fClipRect = fClipRect.intersection(new Rectangle(x,y,w,h));
			x = fData.fTranslateX + fClipRect.x;
			y = fData.fTranslateY + fClipRect.y;
			w = fClipRect.width;
			h = fClipRect.height;

			if (x < 0) {
				w += x;
				x = 0;
			}
			if (y < 0) {
				h += y;
				y = 0;
			}
			if (x < fRestrictedClipX) {
				w = w - (fRestrictedClipX - x);
				x = fRestrictedClipX;
			}
			if (y < fRestrictedClipY) {
				h = h - (fRestrictedClipY - y);
				y = fRestrictedClipY;
			}

			w = Math.max(0, Math.min(w, fRestrictedClipX + fRestrictedClipWidth - x));
			h = Math.max(0, Math.min(h, fRestrictedClipY + fRestrictedClipHeight - y));

			fNativeClipX = x;
			fNativeClipY = y;
			fNativeClipWidth = w;
			fNativeClipHeight = h;

			setClipRectImpl(x,y,w,h);
		}
	}

	public void setStrokeStyle(int style) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.setStrokeStyle(style);
		}
	}

	public void drawRGB(int[] rgb, int offset, int scanLength, int x, int y, int width, int height, boolean processAlpha) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.drawRGB(rgb, offset, scanLength, x, y, width, height, processAlpha);
		}
	}

	public int getDisplayColor(int color) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return 0;
			restoreSettings();
			return super.getDisplayColor(color);
		}
	}

	public void setGrayScale(int value) {
		synchronized (fImage.fLock) {
			if (isDisposed()) return;
			restoreSettings();
			super.setGrayScale(value);
		}
	}

/*
 * Uncomment the following calls if their implementations become native
 */
//	public void drawChar(char ch, int x, int y, int anchor) {
//		synchronized (fImage.fLock) {
//			if (isDisposed()) return;
//			restoreSettings();
//			super.drawChar(ch, x, y, anchor);
//		}
//	}
//
//	public void drawChars(char[] chars, int offset, int length, int x, int y, int anchor) {
//		synchronized (fImage.fLock) {
//			if (isDisposed()) return;
//			restoreSettings();
//			super.drawChars(chars, offset, length, x, y, anchor);
//		}
//	}
//
//	public void drawSubstring(String string, int offset, int length, int x, int y, int anchor) {
//		synchronized (fImage.fLock) {
//			if (isDisposed()) return;
//			restoreSettings();
//			super.drawSubstring(string, offset, length, x, y, anchor);
//		}
//	}
}
