package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.io.*;
import com.ibm.ive.midp.*;
import com.ibm.ive.midp.util.*;
import javax.microedition.lcdui.game.*;

/**
 * Please refer to the MIDP 2.0 specification for documentation about this class.
 *
 * <p>Some additional comments:
 *
 * <p>We expect that the implementation will provide a mechanism for freeing all
 * allocated Images and that this mechanism will be called by AppManager when it is
 * shutting the system down.
 *
 * <p>We add a dispose() call which must free any OS resources associated with the Image
 * instance.
 *
 * <p>We expect that the implementation will detect when an Image becomes unreferenced
 * and that it will dispose unreferenced Images in a timely manner.
 *
 * <p>Finally, it is important that new fields are added only after the existing fields
 * as these fields may be accessed directly from INL code.
 */

public class Image {

	ImageData fData = new ImageData();
	Object fLock = new Object();

	/*
	 * Arbitrary 1MB value is used for image disposal. When
	 * the memory allocated estimate exceeds this value, we
	 * traverse the reference queue looking for images that
	 * can be freed.
	 */
	static long gMemoryCheckThreshold = 0x100000;
	static int gMemoryEstimate = 0;
	static FastVector gReferenceCache = new FastVector();

	static {
		ImageDecoder.registerDecoders(new String[] {
			"javax.microedition.lcdui.CustomPNGDecoder",
			"javax.microedition.lcdui.JPEGDecoder",
		});

		String property = System.getProperty("LcduiMemoryCheckThreshold");
		if (property != null) {
			try {
				gMemoryCheckThreshold = Long.parseLong(property);
			} catch (NumberFormatException e) {
			}
		}
	}

	Image() {
		/* default constructor cannot be visible */
	}

	Image(Image image, int x, int y, int width, int height, int transform) {
		fData.fHandle = createImageWithTransform(image.fData.fHandle, x, y, width, height, transform);
		if (fData.fHandle == -1) throw new IllegalArgumentException();

		if (transform == Sprite.TRANS_ROT90 ||
			transform == Sprite.TRANS_MIRROR_ROT90 ||
			transform == Sprite.TRANS_MIRROR_ROT270 ||
			transform == Sprite.TRANS_ROT270)
		{
			fData.fWidth = height;
			fData.fHeight = width;
		} else {
			fData.fWidth = width;
			fData.fHeight = height;
		}

		fData.fDepth = image.fData.fDepth;
	}

	Image(byte[] data) {
		this(data, 0, data.length);
	}

	Image(byte[] data, int offset, int length) {
		try {
			fData.fHandle = ImageDecoder.createImageFromBytes(data, offset, length);
		} catch (OutOfMemoryError err) {
			throw err;
		} catch (Throwable t) {
			t.printStackTrace();
			throw new IllegalArgumentException();
		}

		// createImageFromBytes returns 0 or -1 if an error is encountered while decoding the image data
		if (fData.fHandle <= 0) throw new IllegalArgumentException();
		fData.fWidth = getWidth(fData.fHandle);
		fData.fHeight = getHeight(fData.fHandle);
		fData.fDepth = getDepth(fData.fHandle);
	}

	Image(int width, int height) {
		try {
			fData.fHandle = createImageWithSize(width, height);
		} catch (OutOfMemoryError err) {
			throw err;
		} catch (Throwable t) {
			t.printStackTrace();
			throw new IllegalArgumentException();
		}

		if (fData.fHandle <= 0) throw new IllegalArgumentException();
		fData.fWidth = width;
		fData.fHeight = height;
		fData.fDepth = getDepth(fData.fHandle);
		fData.fMutable = true;
	}

	Image(Image image) {
		try {
			fData.fHandle = createImageFromImage(image.fData.fHandle);
		} catch (OutOfMemoryError err) {
			throw err;
		} catch (Throwable t) {
			t.printStackTrace();
			throw new IllegalArgumentException();
		}

		if (fData.fHandle <= 0) throw new IllegalArgumentException();
		fData.fWidth = image.fData.fWidth;
		fData.fHeight = image.fData.fHeight;
		fData.fDepth = image.fData.fDepth;
	}

	Image(int[] rgb, int width, int height, boolean processAlpha) {
		fData.fHandle = createImageFromRGBs(rgb, width, height, processAlpha);
		if (fData.fHandle <= 0) throw new IllegalArgumentException();
		fData.fWidth = width;
		fData.fHeight = height;
		fData.fDepth = getDepth(fData.fHandle);
	}

	/**
	 * Construct an immutable image from a byte array of image data.
	 *
	 * @param data a byte array describing the image
	 * @param offset the starting offset in data
	 * @param length the number of elements to use in data
	 *
	 * @return Image an immutable image
	 */
	public static Image createImage(byte[] data, int offset, int length) {
		synchronized (gReferenceCache) {
			disposeUnreferencedImages();
	        if (offset < 0 || length < 0 || offset + length > data.length) throw new ArrayIndexOutOfBoundsException();
			Image newImage = new Image(data, offset, length);
			register(newImage);
			return newImage;
		}
	}

	/**
	 * Construct a mutable image of the given dimensions.  The
	 * returned Image is for offscreen drawing.  Each pixel in
	 * the new Image willl be white.
	 *
	 * @param width the width of the Image
	 * @param height the height of the Image
	 *
	 * @return Image a mutable image
	 *
	 * @throws IllegalArgumentException if either dimension is <= 0
	 */
	public static Image createImage(int width, int height) {
        if (width <= 0 || height <= 0) throw new IllegalArgumentException();

		synchronized (gReferenceCache) {
			disposeUnreferencedImages();
			Image newImage = new Image(width, height);
			register(newImage);
			return newImage;
		}
	}

	/**
	 * Construct an immutable image from image data read from
	 * an InputStream.
	 *
	 * @param stream the InputStream from which decoded image data is read
	 *
	 * @return Image an immutable image
	 *
	 * @throws NullPointerException if stream is null
	 * @throws IOException if there is a problem with loading or decoding the image data
	 */
	public static Image createImage(InputStream stream) throws IOException {
		if (stream == null) throw new NullPointerException();

		synchronized (gReferenceCache) {
			disposeUnreferencedImages();
			Image newImage = null;
			try {
				newImage = new Image(IOUtil.getByteArray(stream));
			} catch (IllegalArgumentException iae) {
				// spec. asks the implementation to throw IOException() if unable to decode the image data
				throw new IOException();
			}
			register(newImage);
			return newImage;
		}
	}

	/**
	 * Construct an immutable image from image data located
	 * in the named resource.
	 *
	 * @param resource the name of the resource to load
	 *
	 * @throws NullPointerException if name is null
	 * @throws IOException if resource cannot be found, loaded or is not a recognized image format.
	 */
	public static Image createImage(String resource) throws IOException {
		if (resource == null) throw new NullPointerException();

		if (!resource.startsWith("/")) resource = "/" + resource; //$NON-NLS-1$ //$NON-NLS-2$
		InputStream is = resource.getClass().getResourceAsStream(resource);
		if (is == null) throw new IOException("resource not found: " + resource); //$NON-NLS-1$

		synchronized (gReferenceCache) {
			disposeUnreferencedImages();
			Image newImage = null;
			try {
				newImage = new Image(IOUtil.getByteArray(is));
			} catch (Exception e) {
				/* createImage(String) needs to throw an IOException instead of an IllegalArgumentException */
				throw new IOException(e.getMessage());
			} finally {
				is.close();
			}
			register(newImage);
			return newImage;
		}
	}

	/**
	 * Construct an immutable image from an Image source.
	 *
	 * @param image the source Image
	 *
	 * @return Image an immutable copy of image
	 *
	 * @throws NullPointerException if image is null
	 */
	public static Image createImage(Image image) {
		if (image == null) throw new NullPointerException();

		/* if an Image is immutable, returns it directly */
		if (!image.isMutable()) return image;

		synchronized (gReferenceCache) {
			disposeUnreferencedImages();
			Image newImage = new Image(image);
			register(newImage);
			return newImage;
		}
	}

	/**
	 * Construct an immutable image from a specified area of a
	 * source image, and transform the pixel data as requested.
	 *
	 * @param image the source Image
	 * @param x the left bound of the area
	 * @param y the top bound of the area
	 * @param width the width of the area
	 * @param height the height of the area
	 * @param transform the transform to use
	 *
	 * @return Image the immutable Image
	 *
	 * @throws NullPointerException if image is null
	 * @throws IllegalArgumentException the specified area exceeds the dimensions of the source image
	 * @throws IllegalArgumentException if either width or height is <= 0
	 * @throws IllegalArgumentException if transform is invalid
	 *
	 * @since 2.0
	 */
	public static Image createImage(Image image, int x, int y, int width, int height, int transform) {
		if (image == null) throw new NullPointerException();

        if (width <= 0 || height <= 0 ||
        	x < 0 || y < 0 ||
            x + width > image.getWidth() ||
            y + height > image.getHeight())
 		{
			throw new IllegalArgumentException();
 		}

 		if (transform < Sprite.TRANS_NONE || transform > Sprite.TRANS_MIRROR_ROT90) {
			throw new IllegalArgumentException();
 		}

		if (image.fData.fMutable == false && width == image.fData.fWidth
			&& height == image.fData.fHeight && transform == Sprite.TRANS_NONE)
		{
			return image;
		}

		synchronized (gReferenceCache) {
			disposeUnreferencedImages();
			Image newImage = new Image(image, x, y, width, height, transform);
			register(newImage);
			return newImage;
		}
	}

	/**
	 * Construct an immutable image from an array of ARGB values.
	 *
	 * Each element of rgb must be of the form 0xAARRGGBB, where the
	 * AA byte represents opacity, on a scale from 00 (fully transparent)
	 * to FF (fully opaque).  The AA byte is only valid if processAlpha
	 * is true.  If processAlpha is false, then all pixels are fully opaque.
	 *
	 * @param rgb an int array of ARGB values
	 * @param width the width of the Image
	 * @param height the height of the Image
	 * @param processAlpha enable/disable use of alpha information
	 *
	 * @return Image the immutable image
	 *
	 * @since 2.0
	 */
	public static Image createRGBImage(int[] rgb, int width, int height, boolean processAlpha) {
        if (rgb == null) throw new NullPointerException();
        if (width <= 0 || height <= 0) throw new IllegalArgumentException();
        if (rgb.length < (width * height)) throw new ArrayIndexOutOfBoundsException();

		synchronized (gReferenceCache) {
			disposeUnreferencedImages();
			Image newImage = new Image(rgb, width, height, processAlpha);
			register(newImage);
			return newImage;
		}
	}

	/**
	 * Return a newly created graphics object for rendering to
	 * this image.  This Image must be mutable for this call to
	 * succeed.
	 *
	 * @return Graphics for drawing to the receiver
	 *
	 * @throws IllegalStateException if the receiver is immutable.
	 */
	public Graphics getGraphics() {
		if (!isMutable()) throw new IllegalStateException();

		synchronized (Graphics.gReferenceCache) {
			synchronized (fLock) {
				if (isDisposed()) return getFakeGraphics();

				return new GraphicsThreadsafe(this);
			}
		}
	}

	/*
	 * Fake, but TCK compilant Graphics instance. This needs to be created
	 * when a user thread tries to get a Graphics instance for an Image's
	 * during shutdown, where the image is already disposed.
	 */
	Graphics getFakeGraphics() {
		Graphics g = new Graphics();
		g.fImage = this;
		g.fData.fFont = Font.getDefaultFont();
		g.fClipRect = new Rectangle(0,0,0,0); // set below in setClip()
		g.fClipRect.x = 0;
		g.fClipRect.y = 0;
		g.fClipRect.width = fData.fWidth;
		g.fClipRect.height = fData.fHeight;
		return g;
	}

	/**
	 * Returns the height in pixels of the receiver.
	 *
	 * @return int the height of the receiver.
	 */
	public int getHeight() {
		return _getHeight();
	}

	int _getHeight() {
		return fData.fHeight;
	}

	/**
	 * Fill an int array with the ARGB pixel data from a specified
	 * region of the receiver.  Each element of rgb will be of the form
	 * 0xAARRGGBB.
	 *
	 * @param rgb the int array for storing ARGB pixel data
	 * @param offset the index into rgb where the first ARGB value will be stored
	 * @param scanLength the number of pixels per line
	 * @param x the left bound of the area
	 * @param y the top bound of the area
	 * @param width the width of the area
	 * @param height the height of the area
	 *
	 * @throws NullPointerException if rgb is null
	 *
	 * @since 2.0
	 */
	public void getRGB(int[] rgb, int offset, int scanLength, int x, int y, int width, int height) {
		if (rgb == null) throw new NullPointerException();
    	if (x < 0 || y < 0) throw new IllegalArgumentException();

    	if (width <= 0 || height <= 0) return;

    	if (x + width > getWidth() || (y + height > getHeight()))
        	throw new IllegalArgumentException();

        if ((scanLength >= 0 && (scanLength < width)) ||
        	(scanLength < 0 && (-scanLength < width)) )
				throw new IllegalArgumentException();

        if (offset < 0 || offset >= rgb.length ||
			(offset + (scanLength * (height-1)) < 0) ||
			(offset + (scanLength * height) > rgb.length))
				throw new ArrayIndexOutOfBoundsException();

        synchronized (fLock) {
        	if (isDisposed()) return;
    		getRGBImpl(fData.fHandle, rgb, offset, scanLength, x, y, width, height);
		}
	}

	/**
	 * Returns the width in pixels of the receiver.
	 *
	 * @return int the width of the receiver.
	 */
	public int getWidth() {
		return _getWidth();
	}

	int _getWidth() {
		return fData.fWidth;
	}

	/**
	 * Answer whether the Graphics instance passed draws to this Image.
	 *
	 * @return true if the Graphics instance draws on this Image, false otherwise.
	 */
	boolean hasSameDestination(Graphics graphics) {
		return (graphics instanceof GraphicsThreadsafe) && ((GraphicsThreadsafe) graphics).fImage == this;
	}

	/**
	 * Returns whether the receiver is mutable.
	 *
	 * @return true if the receiver is mutable, false otherwise.
	 */
	public boolean isMutable() {
		return fData.fMutable;
	}

	/**
	 * Answer whether the Image has been disposed.
	 *
	 * @return true if the Image has been disposed, false otherwise.
	 */
	boolean isDisposed() {
		return fData.fHandle == 0;
	}

	/**
	 * Dispose any OS resources associated with this Image.
	 */
	void dispose() {
		synchronized (gReferenceCache) {
			for (int i = 0; i < gReferenceCache.size(); i++) {
				ImageReference next = (ImageReference) gReferenceCache.elementAt(i);
				if (next.get() == this) {
					gMemoryEstimate -= next.fMemoryEstimate;
					gReferenceCache.removeElementAt(i);
					break;
				}
			}
		}

		synchronized (fLock) {
			if (fData.fHandle != 0) {
				disposeImage(fData.fHandle);
				fData.fHandle = 0;
			}
		}
	}

	static void dispose(ImageReference ref) {
		if (ref.fHandle != 0) {
			disposeImage(ref.fHandle);
			ref.fHandle = 0;
		}
	}

	static void disposeUnreferencedImages() {
		if (gMemoryEstimate < gMemoryCheckThreshold) return;
		System.gc();
		synchronized (gReferenceCache) {
			for (int size = gReferenceCache.size() - 1; size >= 0; size--) {
				ImageReference next = (ImageReference) gReferenceCache.elementAt(size);
				if (next.get() == null) {
					gMemoryEstimate -= next.fMemoryEstimate;
					gReferenceCache.removeElementAt(size);
					dispose(next);
				}
			}
		}
	}

	static void register(Image image) {
		synchronized (gReferenceCache) {
			ImageReference reference = new ImageReference(image, image.fData.fHandle);
			gReferenceCache.addElement(reference);
			gMemoryEstimate += reference.fMemoryEstimate;
		}
	}

	static void disposeAll() {
		synchronized (gReferenceCache) {
			for (int size = gReferenceCache.size()-1; size >= 0; size--) {
				ImageReference ref = (ImageReference) gReferenceCache.elementAt(size);
				Image image = (Image) ref.get();
				if (image != null) {
					image.dispose();
				} else {
					dispose(ref);
				}
			}
		}
	}

	native int createImageWithSize(int width, int height);
	native int createImageFromImage(int handle);
	native int createImageFromRGBs(int[] rgbs, int width, int height, boolean processAlpha);
	native int createImageWithTransform(int handle, int x, int y, int w, int h, int transform);
	native void getRGBImpl(int handle, int[] rgb, int offset, int scanLength, int x, int y, int width, int height);
	static native void disposeImage(int handle);
	native int getWidth(int handle);
	native int getHeight(int handle);
	native int getDepth(int handle);
}
