package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

class ItemComponent extends Composite implements IChoiceListener {

	Item fItem;
	LabelComponent fLabelComponent;
	Component fContentComponent;

	Object fInternalFocus;

	ItemComponent(Composite parent, Item item) {
		this(parent, item, null);
		fLayoutData = item.fLayout;
	}

	Layout createLayout() {
		return new ItemComponentLayout(this);
	}

	ItemComponent(Composite parent, Item item, Component peer) {
		super(parent);
		fItem = item;

		if (fItem.fLabel != null && fItem.fLabel.length() > 0) {
			fLabelComponent = createLabelComponent(item.fLabel);
		}

		fContentComponent = peer == null ? createContents(item) : peer;
	}

	LabelComponent createLabelComponent(String label) {
		return new LabelComponent(this, label);
	}

	Component createContents(Item item) {
		switch (item.getType()) {
			case Item.TYPE_STRING :
				return new StringItemPeer(this, (StringItem) item);
			case Item.TYPE_IMAGE :
				return new ImageItemPeer(this, (ImageItem) item);
			case Item.TYPE_GAUGE :
				return new GaugePeer(this, (Gauge) item);
			case Item.TYPE_CUSTOM :
				return new CustomItemPeer(this, (CustomItem) item);
			case Item.TYPE_TEXTFIELD :
				if (item.fScreen.getDisplayableType() == Displayable.TYPE_TEXTBOX) {
					return new TextPeer(this, (TextField) item, ((TextField)item).fText, TextPeer.TEXTBOX_TYPE);
				} else {
					return new TextPeer(this, (TextField) item, ((TextField)item).fText, TextPeer.TEXTFIELD_TYPE);
				}
			case Item.TYPE_DATEFIELD :
				return new DateFieldPeer(this, (DateField) item);
			case Item.TYPE_CHOICEGROUP :
				{
					ChoiceGroup group = (ChoiceGroup) item;
					boolean isPopup = group.fChoice.fType == Choice.POPUP;

					if (isPopup && getWindow().getType() == Displayable.TYPE_FORM) {
						return new PopupChoiceButton(this, group);
					} else {
						return new ChoiceComponent(this, group.fChoice, this);
					}
				}
			case Item.TYPE_SPACER :
				return new SpacerPeer(this, (Spacer) item);
		}

		return null;
	}

	int getMaxContentWidth() {
		CompositeLayout layout = (CompositeLayout) getWindow().fContentComponent.fLayout;
		return layout.getMaxVisibleWidth();
	}

	int getMaxContentHeight() {
		CompositeLayout layout = (CompositeLayout) getWindow().fContentComponent.fLayout;
		return layout.getMaxVisibleHeight();
	}

	boolean newlineBefore() {
		if (super.newlineBefore()) return true;

		/* Break rows for items with labels. */
		if (fLabelComponent != null) return true;

		if ((fItem.fLayout & Item.LAYOUT_2) == 0) {
			/* MIDP 1.0 layout requires newlines for certain items. */
			switch (fItem.getType()) {
				case Item.TYPE_CHOICEGROUP:
				case Item.TYPE_DATEFIELD:
				case Item.TYPE_GAUGE:
				case Item.TYPE_TEXTFIELD:
					return true;
			}
		}

		if (fItem.getType() == Item.TYPE_STRING) {
			StringItem item = (StringItem) fItem;
			String text = item.fText;
			if (text == null) return false;
			int length = text.length();
			for (int i = 0; i < length; i++) {
				char c = text.charAt(i);
				if (c == '\n') return true;
				if (c == ' ') continue;
				return false;
			}
		}

		return false;
	}

	boolean newlineAfter() {
		if (super.newlineAfter()) return true;

		if ((fItem.fLayout & Item.LAYOUT_2) == 0) {
			/* MIDP 1.0 layout requires newlines for certain items. */
			switch (fItem.getType()) {
				case Item.TYPE_CHOICEGROUP:
				case Item.TYPE_DATEFIELD:
				case Item.TYPE_GAUGE:
				case Item.TYPE_TEXTFIELD:
					return true;
			}
		}

		if (fItem.getType() == Item.TYPE_STRING) {
			StringItem item = (StringItem) fItem;
			String text = item.fText;

			if (text == null) return false;
			for (int i = text.length() - 1; i >= 0; i--) {
				char c = text.charAt(i);
				if (c == '\n') return true;
				if (c == ' ') continue;
				return false;
			}
		}

		return false;
	}

	void clearCachedSizes() {
		if (fLayout != null) ((ItemComponentLayout) fLayout).clearCachedSizes();
		if (fContentComponent != null) fContentComponent.clearCachedSizes();
	}

	void invalidate() {
		clearCachedSizes();
		super.invalidate();
	}

	void hideNotify() {
		fContentComponent.hideNotify();
	}

	void showNotify() {
		fContentComponent.showNotify();
	}

	void setLabel(String label) {
		if (label == null) {
			if (fLabelComponent != null) {
				fLabelComponent.dispose();
				fLabelComponent = null;
			}
		} else {
			if (fLabelComponent == null) {
				fLabelComponent = createLabelComponent(label);
			} else {
				fLabelComponent.setText(label);
			}
		}
		invalidate();
	}

	void updateCommands() {
		fContentComponent.updateCommands();
		getDisplayablePeer().fCommandManager.update();
	}

	void layoutChanged(Component child) {
		clearCachedSizes();
		super.layoutChanged(child);
	}

	public void itemSelected(int index) {
		fItem.notifyStateChanged();
	}
}
