package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

class Row {

	CompositeLayout fLayout;

	// The width of all the peers in this row
	int fCurrentWidth;

	// The width of the row if all items were expanded all the way to their
	// preferred size
	int fPreferredWidth;

	// The max height of all the peers in this row
	int fCurrentHeight;

	// The starting position of this row
	int fX;
	int fY;

	// The maximum width that this row is allowed to assume
	int fMaxWidth;

	// The peers in this row
	Component[] fComponents = new Component[5];
	int fComponentsLength = 0;

	// This is the sum of the differences between preferred size and minimum
	// size for every item on this row.  If this value is 0 then no items can be
	// shrunk.
	int fPixelsLostByShrunkenItems;

	// The number of peers in this row which have the LAYOUT_EXPAND flag set
	int fExpandableItemsCount;

	// The alignment for this row
	int fAlignment;

	/**
	 * Constructs a new Row starting at the given position.  The alignment
	 * value defines how the items in this row should be layed out horizontally.
	 *
	 * @param x the x position of this row
	 * @param y the y position of this row
	 * @param maxWidth the width that the sum of all items in this row cannot
	 *                  exceed
	 * @param alignment the horizontal alignment of the items in this row
	 */
	public Row(CompositeLayout layout, int x, int y, int maxWidth, int alignment) {
		fLayout = layout;
		fX = x;
		fY = y;
		fMaxWidth = maxWidth;
		fCurrentWidth = 0;
		fCurrentHeight = 0;
		fPixelsLostByShrunkenItems = 0;
		fAlignment = alignment;
	}

	void addComponent(Component c) {
		if (fComponentsLength == fComponents.length) {
			Component[] newArray = new Component[fComponents.length * 2];
			System.arraycopy(fComponents, 0, newArray, 0, fComponents.length);
			fComponents = newArray;
		}
		fComponents[fComponentsLength] = c;
		fComponentsLength++;
	}

	boolean matchesAlignment(Component item) {
		int itemAlignment = item.getHAlignment();
		if (itemAlignment == Item.LAYOUT_DEFAULT) return true;
		return itemAlignment == fAlignment;
	}

	/**
	 * Creates a new row directly below this row.
	 *
	 * @param alignment the alignment that should be used for the next row
	 * @return the new row.
	 */
	public Row createNextRow(int alignment) {
		// If the next item has an alignment set to the default then
		// the alignment from the row above should be used
		if (alignment == Item.LAYOUT_DEFAULT) alignment = fAlignment;
		return new Row(fLayout, fX, fY + fCurrentHeight, fMaxWidth, alignment);
	}

	/**
	 * Attempts to add a peer to this row.  This method will return false
	 * if there is not enough room on this row to hold the peer
	 *
	 * @param peer the peer to add
	 * @return true if the peer was added to the row, or false if the peer
	 *          doesn't fit on the row
	 */
	public boolean add(Component peer) {
		int width;
		int shrinkDelta = 0;

		if (!matchesAlignment(peer)) return false;

		int preferredWidth = peer.getPreferredWidth();

		// If this item can be shrunk, then the minimum size
		// should be used to see if the item can fit onto this row
		if (peer.canShrink()) {
			width = peer.getMinimumWidth();
			shrinkDelta = preferredWidth - width;
		} else {
			width = preferredWidth;
		}
		width = Math.min(width, fMaxWidth);

		// If the peer won't fit onto this row then we should return the failure
		int newWidth = width + fCurrentWidth;
		if (newWidth > fMaxWidth) return false;
		fCurrentWidth = newWidth + fLayout.fHSpacing;
		fPreferredWidth += preferredWidth;
		peer.fWidth = width;

		// Track the number of expanding items on this row so that
		// the extra space can be used proportionally
		if (peer.canExpand()) fExpandableItemsCount++;

		// Track the number of pixels given up by shrink items.
		fPixelsLostByShrunkenItems += shrinkDelta;

		// If the peer has the LAYOUT_VSHRINK flag set then the minimum height
		// will be used, otherwise the preferred height will be used.
		int peerHeight = peer.canVShrink() ? peer.getMinimumHeight() : peer.getPreferredHeight();

		// The height of the row should only grow if the peer will not fit in
		// the row as it is
		if (peerHeight > fCurrentHeight) fCurrentHeight = peerHeight;

		addComponent(peer);
		return true;
	}

	/**
	 * Positions and sizes the peers in this row to reflect layout directives
	 */
	public void layoutHorizontally() {
		/* At this point, we have one extra fHSpacing value added (at the end of the row) */
		int availableSpace = fMaxWidth - fCurrentWidth + fLayout.fHSpacing;

		/* Grow shrunken items if necessary. */
		if (fPixelsLostByShrunkenItems > 0) {
			int pixelsForShrinkItems = Math.min(availableSpace, fPixelsLostByShrunkenItems);
			availableSpace -= pixelsForShrinkItems;
		    for (int i = 0; i < fComponentsLength; i++) {
		        Component peer = fComponents[i];
				/* Setting width. */
		        if (peer.canShrink()) {
		            int delta = peer.getPreferredWidth() - peer.fWidth;
					int pixels = delta * pixelsForShrinkItems / fPixelsLostByShrunkenItems;
					peer.fWidth += pixels;
		        }
		    }
		}

		/* Grow expand items if necessary. */
		if (fExpandableItemsCount > 0) {
			int pixelsForExpandItems = availableSpace;
			int expandPixels = pixelsForExpandItems / fExpandableItemsCount;
		    for (int i = 0; i < fComponentsLength; i++) {
		        Component peer = fComponents[i];
	            if (peer.canExpand()) peer.fWidth += expandPixels;
		    }
		}

		/* Determine startX. */
		int x = 0;
		if (fAlignment == Item.LAYOUT_LEFT) {
		    x = fX;
		} else if (fAlignment == Item.LAYOUT_RIGHT) {
		    x = fX + fMaxWidth;
		} else {
			/*
			 * If there are expandable items, then Layout_Center
			 * is the same thing as Layout_Left. Otherwise, start
			 * laying out items at availableSpace / 2;
			 */
			if (fExpandableItemsCount == 0) {
				x = availableSpace >> 1;
			} else {
				x = fX;
			}
		}

		/*
		 * Walk through the row's components setting each components
		 * x, y, and h.
		 */
	    for (int i = 0; i < fComponentsLength; i++) {
	        Component peer = fComponents[i];

			/* Setting x. */
		    if (fAlignment == Item.LAYOUT_RIGHT) {
		        x -= peer.fWidth + fLayout.fHSpacing;
				peer.fX = x;
		    } else {
				peer.fX = x;
		        x += peer.fWidth + fLayout.fHSpacing;
		    }
			/* Setting height. */
		    if (peer.canVShrink()) {
		        peer.fHeight = Math.min(fCurrentHeight, peer.getPreferredHeight());
		    } else if (peer.canVExpand()) {
		        peer.fHeight = fCurrentHeight;
		    } else {
		        peer.fHeight = peer.getPreferredHeight();
		    }

			int peerVAlignment = peer.getVAlignment();
			if (peerVAlignment == Item.LAYOUT_TOP) {
		        peer.fY = fY;
			} else if (peerVAlignment == Item.LAYOUT_VCENTER) {
		        peer.fY = fY + ((fCurrentHeight - peer.fHeight) / 2);
			} else {
		        peer.fY = fY + fCurrentHeight - peer.fHeight;
		    }
		}
	}

	public void layoutVerically(int y) {
		fY = y;

		/*
		 * Walk through the row's components setting each components y and h.
		 */
		for (int i = 0; i < fComponentsLength; i++) {
			Component peer = fComponents[i];

			/* Setting height. */
			if (peer.canVShrink()) {
				peer.fHeight = Math.min(fCurrentHeight, peer.getPreferredHeight());
			} else if (peer.canVExpand()) {
				peer.fHeight = fCurrentHeight;
			} else {
				peer.fHeight = peer.getPreferredHeight();
			}

			/* Setting y. */
			int peerVAlignment = peer.getVAlignment();
			if (peerVAlignment == Item.LAYOUT_TOP) {
				peer.fY = fY;
			} else if (peerVAlignment == Item.LAYOUT_VCENTER) {
				peer.fY = fY + ((fCurrentHeight - peer.fHeight) / 2);
			} else {
				peer.fY = fY + fCurrentHeight - peer.fHeight;
			}
		}
	}

	boolean isVisible(int scrollX, int scrollY, int width, int height) {
		if (fY + fCurrentHeight - scrollY <= 0) return false;
		if (fY - scrollY >= height) return false;

		if (fX + fCurrentWidth - scrollX <= 0) return false;
		if (fX - scrollX >= width) return false;

		return true;
	}

	boolean contains(Component peer) {
		for (int i = 0; i < fComponentsLength; i++) {
			if (fComponents[i] == peer) return true;
		}
		return false;
	}

	Component getNearestPeer(int x) {
		if (fComponentsLength == 0) return null;

		if (fAlignment == Item.LAYOUT_RIGHT) {
			for (int i = 0; i < fComponentsLength; i++) {
				Component next = fComponents[i];
				if (x >= next.fX) return next;
			}
			return fComponents[0];
		} else {
			for (int i = 0; i < fComponentsLength; i++) {
				Component next = fComponents[i];
				if (x <= (next.fX + next.fWidth)) return next;
			}
			return fComponents[fComponentsLength - 1];
		}
	}

	int getIndex(Component peer) {
		for (int i = 0; i < fComponentsLength; i++) {
			if (fComponents[i] == peer) return i;
		}
		return -1;
	}

	void hideNotify() {
		for (int i = 0; i < fComponentsLength; i++) {
			fComponents[i].hideNotify();
		}
	}

	void showNotify() {
		for (int i = 0; i < fComponentsLength; i++) {
			fComponents[i].showNotify();
		}
	}

	boolean canVExpand() {
		for (int i = 0; i < fComponentsLength; i++) {
			if (!fComponents[i].canVExpand()) return false;
		}
		return true;
	}
}
