package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005 All Rights Reserved
 */
class StringItemPeer extends Component implements IButtonListener {

	static final int BORDER_WIDTH = gBorder.getWidth(Item.TYPE_STRING);
	static final int TOTAL_BORDER_WIDTH = 2 * BORDER_WIDTH;
	static final int BORDER_HEIGHT = gBorder.getHeight(Item.TYPE_STRING);
	static final int TOTAL_BORDER_HEIGHT = 2 * BORDER_HEIGHT;
	static final int BUTTON_MARGIN_WIDTH = 8;
	static final int HYPERLINK_MARGIN_WIDTH = 4;
	static final int MARGIN_HEIGHT = 2;

	boolean fPressed = false;
	boolean fInvalidParsedStrings = false;
	private boolean fInteractive;
	boolean fHighlighted;

	StringItem fStringItem;

	// Cache variables used for Item dimension calls
	// These are calculated values stored in the cache
	int fTotalMarginHeight;

	int fPrefContentWidth;
	int fButtonWidth;
	int fHyperlinkWidth;

	int fCachedWidth;
	int fCachedHeight;
	int fTextHeight;
	int fLineHeight;

	Font fFont = gFont;
	TextComponent fTextComponent;
	PushButton fStringButton;

	StringItemPeer(Composite parent, StringItem item) {
		super(parent);
		fStringItem = item;
		item.fPeer = this;
		fFont = item._getFont();
		fTextComponent = new TextComponent(Item.TYPE_STRING, fFont);
		fLineHeight =  fFont._getHeight();
		setText(item.fText);

		fInteractive = item.fCommandListener != null && item.fDefaultCommand != null;

		if (usesButton()) {
			fStringButton = new PushButton(parent, item.fText);
			fStringButton.setListener(this);
			fStringButton.setEnabled(fInteractive);
		}
	}

	boolean usesButton() {
		return fStringItem.fAppearanceMode == Item.BUTTON
			|| fStringItem.fAppearanceMode == Item.PLAIN && fStringItem.fDefaultCommand != null;
	}

	void dispose() {
		super.dispose();
		fStringItem.fPeer = null;
	}

	/**
	 * Return the preferred height for the displayed String.
	 * @return int The line height in pixels
	 */
	int getPreferredHeight() {
		if (usesButton()) return fStringButton.fHeight;

		int lineWidth = getPreferredWidth() - ((fStringItem.fAppearanceMode == Item.HYPERLINK) ? HYPERLINK_MARGIN_WIDTH : TOTAL_BORDER_WIDTH);
		return fTextComponent.getNumberOfLines(lineWidth, fStringItem.fAppearanceMode, fInteractive) * getDrawnRowHeight() + TOTAL_BORDER_HEIGHT;
	}

	/**
	 * Return the preferred width for the displayed String.
	 * @return int The line width in pixels
	 */
	int getPreferredWidth() {
		if (fPrefContentWidth == -1) {
			int maxDisplayableWidth = ((ItemComponent) fParent).getMaxContentWidth();
			if (fTextComponent.fStrippedText == null) {
				fPrefContentWidth = 0;
			} else if (fTextComponent.fStrippedText.indexOf("\n") != -1) { //$NON-NLS-1$
				fPrefContentWidth = maxDisplayableWidth;
			} else {
				fPrefContentWidth = fFont.getStringWidth(fTextComponent.fStrippedText) + TOTAL_BORDER_WIDTH;
				if (fStringItem.fAppearanceMode == Item.HYPERLINK) {
					fPrefContentWidth += HYPERLINK_MARGIN_WIDTH;
				} else if (fInteractive || fStringItem.fAppearanceMode == Item.BUTTON) {
					fPrefContentWidth += BUTTON_MARGIN_WIDTH;
				}

				if (fPrefContentWidth > maxDisplayableWidth) fPrefContentWidth = maxDisplayableWidth;
			}
		}

		return fPrefContentWidth;
	}

	/**
	 * Return the minimum width for the displayed String.  This width
	 * is the smaller of the length of the String, or half the display width
	 * @return int The line width in pixels
	 */
	public int getMinimumWidth(){
		int halfContentWidth = ((ItemComponent) fParent).getMaxContentWidth() / 2;

		if (fTextComponent.fStrippedText == null) {
			return halfContentWidth;
		} else {
			int calculatedValue = fFont.getStringWidth(fTextComponent.fStrippedText) + TOTAL_BORDER_WIDTH;
			if (fStringItem.fAppearanceMode == Item.HYPERLINK) {
				calculatedValue += HYPERLINK_MARGIN_WIDTH;
			} else if (fInteractive) {
				calculatedValue += BUTTON_MARGIN_WIDTH;
			}
			return Math.min(halfContentWidth, calculatedValue);
		}
	}

	/**
	 * Return the minimum height for the displayed String.  This height
	 * is the height required when the minimum width of the line is known
	 * @return int The line height in pixels
	 */
	public int getMinimumHeight() {
		if (usesButton()) return fStringButton.fHeight;
		return getDrawnRowHeight() + TOTAL_BORDER_HEIGHT;
	}

	int getType() {
		return Item.TYPE_STRING;
	}

	/**
	 * Set the text for the StringItem
	 * @param String text The new text for the item
	 */
	public void setText(String text) {
		fTextComponent.setText(text, true);
		fPrefContentWidth = -1;
		fTotalMarginHeight = -1;
		fInvalidParsedStrings = true;
		invalidate();
	}

	void setFont(Font font) {
		fFont = font;
		fLineHeight = fFont._getHeight();
		fPrefContentWidth = -1;
		fTotalMarginHeight = -1;
		fInvalidParsedStrings = true;
		invalidate();
	}

	void sizeChanged(int w, int h) {
		if (w != fCachedWidth || h != fCachedHeight) {
			fInvalidParsedStrings = true;
			fCachedWidth = w;
			fCachedHeight = h;
		}
		if (fInvalidParsedStrings) {
			fInvalidParsedStrings = false;
			int rowHeight = getDrawnRowHeight();
			int maxDisplayableRows = (fHeight - TOTAL_BORDER_HEIGHT) / rowHeight;
			if (fStringItem.fAppearanceMode == Item.BUTTON ||
			   (fStringItem.fAppearanceMode == Item.PLAIN && fInteractive)) {
				maxDisplayableRows = Math.max(1, maxDisplayableRows);
			}
			int lineWidth = fWidth - TOTAL_BORDER_WIDTH;
			if (fStringItem.fAppearanceMode == Item.HYPERLINK) {
				lineWidth -= HYPERLINK_MARGIN_WIDTH;
			} else if (fInteractive) {
				lineWidth -= BUTTON_MARGIN_WIDTH;
			}
			fTextComponent.parse(fWidth, lineWidth, maxDisplayableRows, false);
			fTextHeight = fTextComponent.fFormattedTextLength * rowHeight;
		}
		fButtonWidth = 0;
		fHyperlinkWidth = 0;
		if (fStringItem.fAppearanceMode == Item.HYPERLINK) {
			fHyperlinkWidth = (fTextComponent.fStrippedText == null) ? w : Math.min(w, fFont.getStringWidth(fTextComponent.fStrippedText) + TOTAL_BORDER_WIDTH + HYPERLINK_MARGIN_WIDTH);
		} else if (usesButton()) {
			fButtonWidth = (fTextComponent.fStrippedText == null) ? w : Math.min(w, fFont.getStringWidth(fTextComponent.fStrippedText) + TOTAL_BORDER_WIDTH + BUTTON_MARGIN_WIDTH);
			fStringButton.setBounds(fX, fY, fButtonWidth, fStringButton.fHeight);
		}
	}

	/**
	 * Paint the text of the StringItem
	 * @param Graphics g The Graphics component on which to draw the text
	 */
	protected void paint(Graphics g) {
		if (usesButton()) {
			fStringButton.paint(g);
			return;
		}

		g.setFont(fFont);
		g.setColor(DisplayPeer.COLOR_DISPLAYABLE_BACKGROUND_RGB);
		g.fillRect(0, 0, fWidth, fHeight);
		boolean hasFocus = hasFocus();
		if (fStringItem.fAppearanceMode == Item.HYPERLINK){
			if ((hasFocus || fHighlighted) && fInteractive) {
				paintBorder(g, (hasFocus || fHighlighted) && fInteractive, 0, 0, fWidth - 1, fHeight - 1);
			}
			if (fPressed) {
				g.setColor(DisplayPeer.COLOR_HIGHLIGHTED_HYPERLINK_RGB);
			} else {
				g.setColor(DisplayPeer.COLOR_HYPERLINK_RGB);
			}
		} else {
 			if (hasFocus && fStringItem.fCommands != null) paintBorder(g, true, 0, 0, fWidth - 1, fHeight - 1);
			g.setColor(DisplayPeer.COLOR_FOREGROUND_RGB);
		}

		if (fTextComponent.fFormattedText == null) return;

		int yOffset;
		if ((fStringItem.fLayout & Item.LAYOUT_VCENTER) == Item.LAYOUT_VCENTER) {
			yOffset = (fHeight - TOTAL_BORDER_HEIGHT - fTextHeight) / 2;
		} else if ((fStringItem.fLayout & Item.LAYOUT_BOTTOM) > 0) {
			yOffset = fHeight - fTextHeight - BORDER_HEIGHT;
		} else {
			yOffset = BORDER_HEIGHT;
		}

		int underlineOffset = fFont._getBaselinePosition() + 1;
		int width = fWidth - TOTAL_BORDER_WIDTH;
		int marginWidth = (fStringItem.fAppearanceMode == Item.HYPERLINK) ? HYPERLINK_MARGIN_WIDTH / 2 : 0;
		int rowHeight = getDrawnRowHeight();
		for (int i = 0; i < fTextComponent.fFormattedTextLength; i++) {
			int xOffset;
			/* Calculate the xOffset to be used for drawing the current line of text. */
			if ((fStringItem.fLayout & Item.LAYOUT_CENTER) == Item.LAYOUT_CENTER) {
				xOffset = (width - (fFont.getStringWidth(fTextComponent.fFormattedText[i]) + 2 * marginWidth)) / 2;
			} else if ((fStringItem.fLayout & Item.LAYOUT_RIGHT) == Item.LAYOUT_RIGHT) {
				xOffset = width - (fFont.getStringWidth(fTextComponent.fFormattedText[i]) + 2 * marginWidth);
			} else {
				xOffset = BORDER_WIDTH + marginWidth;
			}
			g.drawStringWithTabExpansion(fTextComponent.fFormattedText[i], xOffset, yOffset, 0);
			/* Underline the text if the StringItem has an Item.HYPERLINK appearance mode. */
			if (fStringItem.fAppearanceMode == Item.HYPERLINK && fTextComponent.fFormattedText[i].length() > 0) {
				g.drawLine(xOffset, yOffset + underlineOffset, xOffset + fFont.getStringWidth(fTextComponent.fFormattedText[i]), yOffset + underlineOffset);
			}
			yOffset += rowHeight;
		}
	}

	boolean traverse(int direction, int viewWidth, int viewHeight, int[] visibleRectangle, int x, int y) {
		if (usesButton()) {
			fStringButton.traverse(direction, viewWidth, viewHeight, visibleRectangle, x, y);
		}

		/*
		 * If the whole item is visible:
		 * - If the item either has focus or is not interactive, return false.
		 * - Otherwise repaint to show focus.
		 */
		if (visibleRectangle[2] == fWidth && visibleRectangle[3] == fHeight) {
			if (hasFocus())	return false;
			if (fInteractive) {
				fHighlighted = true;
				repaint();
				return true;
			} else if (fStringItem.fCommands != null) {
				return true;
			}
			return false;
		}

		if (!hasFocus()) {
			switch (direction) {
				case Canvas.LEFT:
				case Canvas.UP:
					if (visibleRectangle[2] < viewWidth) visibleRectangle[2] = viewWidth;
					if (visibleRectangle[3] < viewHeight) visibleRectangle[3] = viewHeight;
					visibleRectangle[0] = fWidth - viewWidth;
					visibleRectangle[1] = fHeight - viewHeight;
					return true;
				case Canvas.DOWN:
				case Canvas.RIGHT:
				case CustomItem.NONE:
					if (visibleRectangle[2] < viewWidth) visibleRectangle[2] = viewWidth;
					if (visibleRectangle[3] < viewHeight) visibleRectangle[3] = viewHeight;
					visibleRectangle[0] = 0;
					visibleRectangle[1] = 0;
					return true;
			}
		}

		switch (direction) {
			case Canvas.LEFT:
			case Canvas.UP:
				/*
				 * If the upper left corner of the rectangle is at zero, the
				 * rectangle can't be scrolled anymore.
				 */
				if (visibleRectangle[1] == 0) return false;
				visibleRectangle[1] = Math.max(visibleRectangle[1] - viewHeight / 2, 0);
				visibleRectangle[3] = viewHeight;
				return true;
			case Canvas.RIGHT:
			case Canvas.DOWN:
				/*
				 * If the bottom of the rectangle is the viewHeight, the
				 * rectangle can't be scrolled anymore.
				 */
				if (visibleRectangle[1] + visibleRectangle[3] == fHeight) return false;
				visibleRectangle[1] = Math.min(visibleRectangle[1] + viewHeight / 2, fHeight - viewHeight);
				visibleRectangle[3] = viewHeight;
				return true;
			case CustomItem.NONE:
				return true;
		}

		return false;
	}

	void traverseOut() {
		fHighlighted = false;
		if (usesButton()) fStringButton.traverseOut();
		super.traverseOut();
	}

	int getDrawnRowHeight() {
		if (fStringItem.fAppearanceMode == Item.BUTTON || fStringItem.fAppearanceMode == Item.HYPERLINK || fInteractive) {
			return PushButton.TOTAL_TEXT_V_MARGIN + fLineHeight;
		}
		if (fTotalMarginHeight == -1) fTotalMarginHeight = MARGIN_HEIGHT;
		return fLineHeight + fTotalMarginHeight;
	}

	boolean keyPressed(int keyCode) {
		if (fInteractive && CanvasPeer.getGameAction(keyCode) == Canvas.FIRE) {
			fPressed = true;
			if (fStringItem.fAppearanceMode == Item.HYPERLINK) {
				fStringItem.fireCommand(fStringItem.fDefaultCommand);
				repaint();
			} else if (usesButton()) {
				fStringButton.keyPressed(keyCode);
			}
			return true;
		}
		return false;
	}

	boolean keyReleased(int keyCode) {
		if (fInteractive && CanvasPeer.getGameAction(keyCode) == Canvas.FIRE) {
			fPressed = false;
			if (fStringItem.fAppearanceMode == Item.HYPERLINK) {
				repaint();
			} else if (usesButton()) {
				fStringButton.keyReleased(keyCode);
			}
			return true;
		}
		return false;
	}

	boolean pointerPressed(int x, int y) {
		if (fInteractive && contains(x, y)) {
			fPressed = true;
			if (fStringItem.fAppearanceMode == Item.HYPERLINK) {
				repaint();
			} else if (usesButton()) {
				fStringButton.pointerPressed(x, y);
			}
			return true;
		}
		return false;
	}

	boolean pointerDragged(int x, int y) {
		if (fInteractive && fPressed != contains(x, y)) {
			fPressed = !fPressed;
			if (fStringItem.fAppearanceMode == Item.HYPERLINK) {
				repaint();
			} else if (usesButton()) {
				fStringButton.fHighlighted = !fStringButton.fHighlighted;
				fStringButton.pointerDragged(x, y);
			}
			return true;
		}
		return false;
	}

	boolean pointerReleased(int x, int y) {
		if (fPressed) {
			fPressed = false;
			if (fStringItem.fAppearanceMode == Item.HYPERLINK){
				repaint();
				fStringItem.fireCommand(fStringItem.fDefaultCommand);
			} else if (usesButton()) {
				fStringButton.pointerReleased(x, y);
			}
			return true;
		}
		return false;
	}

	void updateCommands() {
		boolean interactive = fStringItem.fCommandListener != null &&
				fStringItem.fDefaultCommand != null;

		if (interactive != fInteractive && fStringItem.fAppearanceMode == Item.PLAIN) {
			fInvalidParsedStrings = true;
			invalidate();
		}

		fInteractive = interactive;
		if (usesButton()) fStringButton.setEnabled(fInteractive);
	}

	boolean contains(int x, int y) {
		if (isHyperlink()) return (x >= 0 && x < fHyperlinkWidth) && (y >= 0 && y < fHeight);
		if (!fInteractive) return super.contains(x, y);
		return (x >= 0 && x < fButtonWidth) && (y >= 0 && y < fHeight);
	}

	boolean isEditable() {
		return fInteractive;
	}

	boolean isHyperlink() {
		return fStringItem.fAppearanceMode == Item.HYPERLINK;
	}

	void clearCachedSizes() {
		fPrefContentWidth = -1;
		fCachedWidth = -1;
		fCachedHeight = -1;
	}

	public void buttonFired(ButtonWidget button) {
		fStringItem.fireCommand(fStringItem.fDefaultCommand);
	}
}
