package javax.microedition.lcdui;

import com.ibm.ive.midp.*;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */
class TextImpl {

	// These constants are used in the state machine for determining
	// whether a string passes the DECIMAL contraint.
	//
	// @see TextImpl.checkDecimal(String)
	private final static int STATE_START       = 0;
	private final static int STATE_AFTER_MINUS = 1;
	private final static int STATE_FRACTION    = 2;
	private final static int STATE_NUMBER      = 3;

	String fContents;
	private int fMaxSize;
	private int fConstraints;

	TextPeer fPeer;

	private static final int ALL_CONTRAINTS =
		TextField.CONSTRAINT_MASK
		| TextField.PASSWORD
		| TextField.INITIAL_CAPS_SENTENCE
		| TextField.INITIAL_CAPS_WORD
		| TextField.NON_PREDICTIVE
		| TextField.SENSITIVE
		| TextField.UNEDITABLE
		;

	public TextImpl(String contents, int maxSize, int constraints) {
		if (contents == null) contents = ""; //$NON-NLS-1$
		if (maxSize <= 0) throw new IllegalArgumentException(MidpMsg.getString("TextImpl.constructor.MaxSize")); //$NON-NLS-1$
		fMaxSize = maxSize;

		setConstraints(constraints);
		setString(contents);
	}

	public void delete(int offset, int length) {
		// StringBuffer.delete does not throw a StringIndexOutOfBoundsException
		// if the end is greater than length()
		if (offset + length > size()) throw new StringIndexOutOfBoundsException();

		StringBuffer buffer = new StringBuffer(getString());
		buffer.delete(offset, offset + length);
		setString(buffer.toString());
	}

	public int getCaretPosition() {
		if (fPeer != null) return fPeer.getCaretPosition();
		return 0;
	}

	public int getChars(char[] chars) {
		if (chars == null) throw new NullPointerException();

		String contents = getString();

		if (chars.length < contents.length()) throw new ArrayIndexOutOfBoundsException();
		contents.getChars(0, contents.length(), chars, 0);
		return contents.length();
	}

	public int getConstraints() {
		return fConstraints;
	}

	public int getMaxSize() {
		return fMaxSize;
	}

	public String getString() {
		return fPeer != null ? fPeer.getText() : fContents;
	}

	public void insert(char[] chars, int offset, int length, int position) {
		if (chars == null) throw new NullPointerException();

		try {
			String value = String.valueOf(chars, offset, length);
			insert(value, position);
		} catch (StringIndexOutOfBoundsException exp1) {
			throw new ArrayIndexOutOfBoundsException(exp1.getMessage());
		}
	}

	public void insert(String text, int position) {
		if (text == null) throw new NullPointerException();

		// if the position is greater than or equal to the length then
		// append should occur
		if (position >= size()) {
			setString(getString() + text);
			return;
		}

		// If the position is less then 0 then 0 should be used
		if (position < 0) position = 0;
		StringBuffer buffer = new StringBuffer(getString());
		buffer.insert(position, text);
		setString(buffer.toString());
	}

	public void setChars(char[] chars, int offset, int length) {
		if (chars == null) {
			setString(null);
		} else {
			try {
				String newString = String.valueOf(chars, offset, length);
				setString(newString);
			} catch (StringIndexOutOfBoundsException exp1) {
				throw new ArrayIndexOutOfBoundsException(exp1.getMessage());
			}
		}
	}

	public void setConstraints(int constraints) {
		// Check the contraints
		switch (constraints & TextField.CONSTRAINT_MASK) {
			case TextField.ANY:
			case TextField.EMAILADDR:
			case TextField.URL:
			case TextField.DECIMAL:
			case TextField.NUMERIC:
			case TextField.PHONENUMBER: {
				break;
			}
			default: throw new IllegalArgumentException();
		}

		if ((constraints & ~ALL_CONTRAINTS) != 0) throw new IllegalArgumentException();

		fConstraints = constraints;

		// setContraints is called before setString so if the
		// constructor is calling this method then the contents could
		// still be null
		String contents = getString();
		if (getString() == null) return;

		// If the current text does not work with the current
		// contraints then the text field should be set to empty
		try {
			checkConstraints(contents);
			if (fPeer != null) fPeer.setConstraints(fConstraints);
		} catch (IllegalArgumentException exp1) {
			setString(""); //$NON-NLS-1$
		}
	}

	public void setInitialInputMode(String characterSubset) {
		// TODO: implment
	}

	public int setMaxSize(int size) {
		if (size <= 0) throw new IllegalArgumentException();

		// Attemp truncation of the string if needed
		String contents = getString();
		if (size < contents.length()) {
			String test = contents.substring(0,size);
			setString(test);
			fMaxSize = size;
		}
		fMaxSize = size;

		// Keep peer in sync
		if (fPeer != null) fPeer.setMaxSize(size);
		return size;
	}

	public void setString(String text) {
		if (text == null) {
			fContents = "";	//$NON-NLS-1$
		} else {
			if (text.length() > fMaxSize) throw new IllegalArgumentException(MidpMsg.getString("TextImpl.setString.Length"));				 //$NON-NLS-1$
			checkConstraints(text);
			fContents = text;
		}

		// Keep peer in sync
		if (fPeer != null) fPeer.setText(fContents);
	}

	public int size() {
		if (fPeer != null) return fPeer.size();
		return fContents.length();
	}

	private void checkConstraints(String text) {
		// The empty string is always allowed
		if (text.length() == 0) return;

		switch (fConstraints & TextField.CONSTRAINT_MASK) {
			// These two contraints let any text pass
			case TextField.ANY:
			case TextField.PHONENUMBER:
				break;
			case TextField.EMAILADDR:
				//if (checkEmail(text) == 0) throw new IllegalArgumentException();
				break;
			case TextField.URL:
				//if (!checkURL(text)) throw new IllegalArgumentException();
				break;
			case TextField.DECIMAL:
				checkDecimal(text);
				break;
			case TextField.NUMERIC:
				checkNumeric(text);
				break;
		}
	}

	/**
	 * Checks whether the character is QCHAR as specified in RFC 821.
	 * QCHAR = Any of 128 ASCII characters except \r, \n, ", \
	 * @param ch
	 * @return 	true if character is a QCHAR
	 * 			false otherwise
	 */
	boolean isQChar(char ch) {
		return (!(ch > 128 || ch == '\"' || ch == '\n' || ch == '\r' || ch == '\\'));
	}

	/**
	 * Checks whether a character is SPECIAL character according to
	 * RFC 821.
	 * SPECIAL = {"<" | ">" | "(" | ")" | "[" | "]" | "\" | "."
					  | "," | ";" | ":" | "@"  """ | the control
					  characters (ASCII codes 0 through 31 inclusive and
					  127) }
	 * @param ch
	 * @return	true if character is a Special character
	 * 			false otherwise
	 */
	boolean isSpecialChar(char ch) {
		return ((ch >= 127 || (ch <= 0 && ch >= 31) || ch == '<' || ch == '>' || ch == '(' || ch == ')' || ch == '[' || ch == ']' || ch == '\\' || ch == '.' || ch == ',' || ch == ';' || ch == ':' || ch == '@' || ch == '\"'));
	}

	/**
	 * This implementation is part of ABNF format specification of RFC 821.
	 * RFC 821 specifies the complete ABNF for SMTP commands. But this implementation
	 * does implement only mailbox specification which specifies the syntax of an Email Address.
	 *
	 * @return  0 Invalid Email address
	 * 			1 Incomplete but valid email characters are present
	 * 			2 Complete Email address
	 */
	int checkEmail(String emailAddress) {
		String localName, domain =  null;
		int len = emailAddress.length();
		int index = emailAddress.indexOf('@');
		if (index == -1) {
			localName = emailAddress;
		} else {
			if (index == 0) return 0; // Invalid to have at start

			localName = emailAddress.substring(0, index);
			if (index + 1 < len) domain = emailAddress.substring(index + 1);
		}

		// check for validity of localName
		if (localName.charAt(0) == '\"') { // Quoted String.
			//Invalid as domain is present and this string starts with " and doesn't end with "
			if (!localName.endsWith("\"") && domain != null) return 0;

			index = 1;
			int lengthOfQString = localName.length();
			if (localName.charAt(lengthOfQString - 1) == '\"') lengthOfQString--; // Skipping the last quotation mark

			while (index < lengthOfQString) {
				char ch = localName.charAt(index);
				index++;
				if (ch == '\\') {
					if (index < lengthOfQString) {
						ch = localName.charAt(index);
						index++;
					} else {
						/*
						 * If \ is present and no other character present then it is
						 * invalid. Otherwise it might be incomplete
						 */
						return domain != null ? 0 : 1;
					}
				} else { // It must be a qchar
					if (!isQChar(ch)) return 0; // Invalid char present
				}
			}
		} else { // Dot String
			index = 0;
			int lengthOfDotString = localName.length();
			if (localName.charAt(lengthOfDotString - 1) == '.' && domain != null) return 0;

			while (index < lengthOfDotString) {
				char ch = localName.charAt(index);
				index++;
				if (ch == '\\') {
					if (index < lengthOfDotString) {
						ch = localName.charAt(index);
						index++;
					} else {
						/*
						 * If \ is present and no other character present then it is
						 * invalid. Otherwise it might be incomplete
						 */
						return domain != null ? 0 : 1;
					}
				} else { // It must be qchar
					if (isSpecialChar(ch)) return 0; // Invalid char present
				}
			}
		}
		if (domain != null) { // Check whether it is a valid domain
			index = 0;
			int lengthOfDomain = domain.length();
			while (index < lengthOfDomain) {
				char ch = domain.charAt(index);
				if (isAlpha(ch)) {
					index++;
					while (index < lengthOfDomain) {
						ch = domain.charAt(index);
						if (ch == '.') break; // Can be next domain element
						index++;
						if (!isAlpha(ch) && !isDigit(ch) && ch != '-') return 0; // Invalid char in domain name
					}
					if (ch != '.') { // Completed the domain length
						/*
						 * if ch ends with -, then it is incomplete. Otherwise, it is
						 * a complete E-mail address
						 */
						return ch == '-' ? 1 : 2;
					}
				} else if (ch == '#') { // The Number follows
					index++;
					while (index < lengthOfDomain) {
						ch = domain.charAt(index);
						if (ch =='.') break; // Can be next domain element present
						if (!isDigit(ch)) return 0; // Invalid char where number digits must be present
						index++;
					}
					if (ch != '.') return 2; // End of the domain is reached
				} else if (ch == '[') {
					index++;
					int num = -1;
					while (index < lengthOfDomain) {
						ch = domain.charAt(index);
						index++;
						if (ch == ']') {
							if (index < lengthOfDomain) { // Next char must be dot since next domain element is present.
								ch = domain.charAt(index);
								if (ch != '.') return 0;
							} else {
								return 2;
							}
						}
						if (ch == '.') {
							if (num < 0 || num > 255) return 0; // dots are present at wrong places.
							num = -1;
							continue;
						}
						if (!isDigit(ch)) return 0; // Invalid char other than digit is found

						if (num == -1) {
							num = ch - '0';
						} else {
							num = num * 10 + ch - '0';
						}
					}
				} else {
					return 0;
				} // Invalid char present
				index++;
			}
		}
		return 1; // In-complete Email Address.
	}

	/**
	 * checks whether the character is an alphabet or not
	 * @param c character to be checked
	 * @return	true if c is an alphabet
	 * 			false otherwise
	 */
	boolean isAlpha(char c) {
		return (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z');
	}

	/**
	 * checks whether the character is a Digit or not
	 * @param c character to be checked
	 * @return	true if c is a digit
	 * 			false otherwise
	 */
	boolean isDigit(char c) {
		return c >= '0' && c <= '9';
	}

	/**
	 * checks whether the character is Sub-Delimiters for URL
	 * Sub-Delimiters = "!" / "$" / "&" / "'" / "(" / ")" / "*" / "+" / "," / ";" / "="
	 * @param c character to be checked
	 * @return	true if c is a sub-delimiter
	 * 			false otherwise
	 */
	boolean isSubDelimiters(char c) {
		return c == '!' || c == '$' || c == '\'' || c == '*' || c == '(' || c == ')' || c == '+' || c == ',' || c == ';' || c == '=';
	}

	/**
	 * checks whether the character is a unreserved character according to RFC 3986
	 * Unreserved = ALPHA / DIGIT / "-" / "." / "_" / "~"
	 * @param c
	 * @return	true if c is a unreserved character
	 * 			false otherwise
	 */
	boolean isUnreserved(char c) {
		return isAlpha(c) || isDigit(c) || c == '-' || c =='_' || c == '~' || c =='.';
	}

	/**
	 * Checks whether the query and fragment are valid or not
	 * according to specification of Fragment and Query parts in
	 * syntax of URI in RFC 3986.
	 * @param query 	To check whether it is a valid query part
	 * @param fragment	To Check whether fragment part is a valid fragment part or not
	 * @return	true if both are valid parts
	 * 			false if any one is not valid.
	 */
	boolean isValidQueryAndFragment(String query, String fragment) {
		if ((query == null || query.length() == 0) && (fragment == null || fragment.length() == 0)) return true;

		int val = isPChars(query);
		if (val == 0 || (val == 1 && fragment != null)) return false; // If Fragment part is present, then query cant be incomplete.

		// Valid Query is there. Now process the Fragment
		if (fragment != null) {
			return isPChars(fragment) != 0;
		}
		return true;
	}

	/**
	 * Checks whether a character is a HEX-DIGIT - {0..9, A..F}
	 * @param ch
	 * @return	true if it is a hexa-digit
	 * 			false otherwise
	 */
	boolean isHexDigit(char ch) {
		return (isDigit(ch) || (ch >= 'a' && ch <= 'f') || (ch >= 'A' || ch <= 'F'));
	}

	/**
	 * checks whether the text string contains only pchars
	 * pchar = pct-encoded / unreserved / sub-delimiters / ':' / '@'
	 * @param text
	 * @return 	0 	If it contains invalid character other than pchar
	 * 			1	Partially complete - PCT ENCODED may be incomplete
	 * 			2 	string completely consists of pchars only.
	 */
	int isPChars(String text) {
		if (text == null || text.length() == 0) return 1; // Incomplete

		int index = 0;
		int len = text.length();
		while (index < len) {
			char ch = text.charAt(index);
			if (ch == '%') { // May be PCT Encoded
				if (index + 2 >= len) return 1; // may be partially empty

				index++;
				ch = text.charAt(index);
				char nextCh = text.charAt(index + 1);
				if (!isHexDigit(ch) || !isHexDigit(nextCh)) return 0; // Invalid pchar
				index++;
			} else {
				if (!isUnreserved(ch) && !isSubDelimiters(ch) && ch != ':' && ch != '@') return 0; // Invalid pchar
			}
			index++;
		}
		return 2; // Complete pchar string
	}

	/**
	 * checkURL method is used to check the URL syntax. If checks whether
	 * the text in the TextField/TextBox is valid URL conforming to specification
	 * of RFC 3986. It is implementation of ABNF form of URI syntax in RFC 3986.
	 * @param string - Text
	 * @return 	true if it is a partial or complete URL
	 * 			false if contains invalid character or syntax
	 */
	boolean checkURL(String text) {
		String query = null, fragment = null;
		boolean isAuthority = false;

		if (text.length() == 0) return true; // Empty string is valid URL

		int index = 0;
		int length = text.length();
		char c;
		if (text.indexOf(' ') != -1) { // Check if the text contains space
			StringBuffer textBuffer = new StringBuffer();
			// Convert SPACE into Percent-Encoded form so as to allow space in the URL field
			 while (index < length) {
				 c = text.charAt(index);
				 if (c == ' ') {
					 // %20 is Percent Encoded format for space
					 textBuffer.append("%20"); //NON-NLS-$1
				 } else {
					 textBuffer.append(c);
				 }
				 index++;
			 }
			 text = textBuffer.toString();
			 textBuffer = null;
		}

		index = text.indexOf('#'); // Checking whether the Fragment Part is there
		if (index != -1) {
			fragment = text.substring(index + 1); // Extract the fragment part
			text = text.substring(0, index);
		}

		index = text.indexOf('?'); // Checking whether the Query part is there
		if (index != -1) { // Extracting the Query Part of URL
			query = text.substring(index + 1);
			text = text.substring(0, index);
		}

		// Checking whether the fragment and query parts are valid.
		boolean validQueryFragment = isValidQueryAndFragment(query, fragment);
		if (!validQueryFragment) return false;

		// Process now to check for Scheme and hier-part
		index = 0;
		c = text.charAt(index);
		length = text.length();

		boolean isURI = false;
		boolean isURIReference = false;
		if (isAlpha(c)) { // URL scheme is present
			index = 1;
			isURI = true;
			do {
				if (index == length) return true; // Scheme incomplete. But valid URL

				c = text.charAt(index++);
				if (c == ':') break; // Scheme ended

				if (!(isAlpha(c) || isDigit(c) || c == '+' || c == '-' || c == '.')) return false; // SCHEME contains invalid character
			} while (index < length);

			if (index == length) return true; // Scheme incomplete. But valid URL
			c = text.charAt(index);
		}
		if (c == '/') { // If / is there it should contain hierarchical part
			index++;
			if (index == length) return true; // Scheme is complete. But next parts are not full.

			if (text.charAt(index) == '/') {
				isURIReference = true;
				isAuthority = true;
				index++;
			}
		}
		// If not URI or URI-Reference it is invalid scheme
		if (!isURI & !isURIReference) return false;

		if (index == length) return true;

		String remaining = text.substring(index);
		if (isAuthority) { // Authority part is there
			String abEmpty = null, authority = null;
			int indexAbEmpty = remaining.indexOf('/');
			if (indexAbEmpty != -1) { // abEmpty part is there.
				abEmpty = remaining.substring(indexAbEmpty + 1);
				authority = remaining.substring(0, indexAbEmpty);
			} else { // path-abempty is absent
				authority = remaining;
			}

			// Process the authority part
			if (authority != null && authority.length() != 0) {
				String userName = null;
				int uNameInd = authority.indexOf('@');
				if (uNameInd != -1) {
					userName = authority.substring(0, uNameInd);
					authority = authority.substring(uNameInd + 1);
				}

				if (userName != null && userName.length() != 0) {
					if (userName.indexOf('@') != -1) return false;

					int val = isPChars(userName);
					if (val == 0 || (val == 1 && (authority != null || abEmpty != null || query != null || fragment != null))) return false;
				}

				// Process the authority
				if (authority != null && authority.length() != 0) {
					if (authority.indexOf('@') != -1) return false;

					int val = isPChars(authority);
					if (val == 0 || (val == 1 && (abEmpty != null || query != null || fragment != null))) return false;
				}
			}
			// Process the abEmpty.
			if (abEmpty != null && abEmpty.length() != 0) {
				int len = abEmpty.length();
				index = abEmpty.indexOf('/');
				String segment = null;
				while (index < len && index != -1) {
					segment = abEmpty.substring(0, index);
					int val = isPChars(segment);
					if (val == 0 || (val == 1 && (query != null || fragment != null))) return false;
					abEmpty = abEmpty.substring(index + 1);
					index = abEmpty.indexOf('/');
				}
				int val = isPChars(abEmpty);
				if (val == 0 || (val == 1 && (query != null || fragment != null))) return false;
			}
			return true;
		} else {
			// path-abempty or path-absolute or path-noscheme checking goes here.
			// This is general URI syntax construct.
			int len = remaining.length();
			index = remaining.indexOf('/');
			String segment = null;
			while (index < len && index != -1) {
				segment = remaining.substring(0, index);
				int val = isPChars(segment);
				if (val == 0 || (val == 1 && (query != null || fragment != null))) return false;
				remaining = remaining.substring(index + 1);
				index = remaining.indexOf('/');
			}
			int val = isPChars(remaining);
			if (val == 0 || (val == 1 && (query != null || fragment != null))) return false;

			return true;
		}
	}

	private void checkDecimal(String text) {
		int state = STATE_START;

		boolean hadWholePart = false;
		boolean hadFractionPart = false;

		for (int i=0;i<text.length();i++) {
			char c = text.charAt(i);
			switch (state) {
				case STATE_START: {
					if (c == '-') {
						state = STATE_AFTER_MINUS;
						break;
					}
				}
			    case STATE_NUMBER:
				case STATE_AFTER_MINUS: {
					if (c == '.') {
						state = STATE_FRACTION;
						break;
					}
					if (Character.isDigit(c)) {
						state = STATE_NUMBER;
						hadWholePart = true;
						break;
					}
					throw new IllegalArgumentException();
				}
				case STATE_FRACTION: {
					if (Character.isDigit(c)) {
						hadFractionPart = true;
						break;
					}
					throw new IllegalArgumentException();
				}
			}
		}

		if (hadFractionPart || hadWholePart) return;
		throw new IllegalArgumentException();
	}

	private void checkNumeric(String text) {
		try {
			Integer.parseInt(text);
		} catch (NumberFormatException exp1) {
			throw new IllegalArgumentException();
		}
	}

	void replace(int start, int end, String text) {
		StringBuffer buffer = new StringBuffer(getString());
		buffer.delete(start, end);
		buffer.insert(start,text);
		String newString = buffer.toString();
		checkConstraints(newString);
		setString(newString);
	}
}

