package javax.microedition.lcdui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2005, 2006  All Rights Reserved
 */

import com.ibm.ive.midp.*;

abstract class Window extends Composite {

	Graphics fGraphics;
	Display fDisplay;
	WindowContent fContentComponent;
	boolean fInvalidated;
	boolean fDisposed = true;
	boolean fAllowScroll = true;

	int fUpdateType;
	boolean fUpdateScheduled;

	final static int ITEMS_APPENDED = 1;
	final static int ITEMS_INSERTED = 2;
	final static int ITEMS_SET = 4;
	final static int ITEMS_DELETED = 8;
	final static int ITEM_VALUE_CHANGED = 16;
	final static int LAYOUT_CHANGED = 32;
	final static int COMMANDS_CHANGED = 64;
	final static int TICKER_UPDATED = 128;
	final static int TITLE_UPDATED = 256;
	final static int ITEMS_CHANGED = ITEMS_APPENDED | ITEMS_INSERTED | ITEMS_SET | ITEMS_DELETED;

	Runnable fUpdateRunnable = new Runnable() {
		public void run() {
			update();
		}
	};

	Window(Composite parent, Display display) {
		super(parent);
		fDisplay = display;
		fExpandVertically = true;
		fShown = false;
	}

	int getWindowHandle() {
		return fDisplay.fPeer.fHandle;
	}

	Image getOffscreenBuffer() {
		return fDisplay.fPeer.fOffscreenBuffer;
	}

	void createGraphics() {
		fGraphics = new Graphics(this);
	}

	public Graphics getGraphics() {
		return fGraphics;
	}

	DisplayablePeer getDisplayablePeer() {
		return null;
	}

	Window getWindow() {
		return this;
	}

	public GraphicsThreadsafe getDisplayGraphics() {
		GraphicsThreadsafe g = new GraphicsThreadsafe(this);
		fContentComponent.setOriginAndClip(g);
		return g;
	}

	void setInitialFocus() {
		if (fFocusComponent != null) {
			fFocusComponent.traverse(CustomItem.NONE, fFocusComponent.fWidth, fFocusComponent.fHeight, getVisibleRect(fFocusComponent), 0, 0);
			return;
		}

		if (fContentComponent != null) {
			fAllowScroll = false;
			traverseComponent(fContentComponent, Canvas.DOWN, fContentComponent.fWidth, fContentComponent.fHeight, getVisibleRect(fContentComponent), 0, 0);
			fAllowScroll = true;
		}
	}

	void show() {
		synchronized (Device.gRepaintLock) {
			fShown = false;
			fDisposed = false;
			getDisplayPeer().addWindow(this);

			/*
			 * We defer the layout code until we're actually ready to show the displayable.
			 * The reason for this is that resize events can be generated on some platforms
			 * when widgets are created. So, we defer laying out the widgets as long as we
			 * can in order to minimize the number of times we have to layout the widgets.
			 */
			create();

			/*
			 * This is the internal showNotify. It should not result in a call to Canvas showNotify.
			 * Otherwise, deadlocks can occur.
			 */
			showNotify();
			fShown = true;

			/*
			 * If there is a pending update for the current update Runnable,
			 * queue it so that it doesn't get lost
			 */
			if (Device.gDisplayableUpdateNeeded) {
				Device.postRunnable(Device.gDisplayableUpdateRunnable);
				Device.gDisplayableUpdateNeeded = false;
			}

			Device.gDisplayableUpdateRunnable = fUpdateRunnable;

			/*
			 * We must call update directly rather than scheduling an update normally
			 * because otherwise there is a critical section between the time a displayable
			 * is created and when it is laid out. For example, midlets calling Canvas.getHeight()
			 * could get an invalid height if the call is made between the displayable
			 * show() and update() calls.
			 */
			fUpdateType |= LAYOUT_CHANGED;
			fUpdateScheduled = true;
			update();
		}
	}

	void scheduleUpdate(int type) {
//		if ((type & LAYOUT_CHANGED)  > 0) new Exception().printStackTrace();
		if ((fUpdateType & type) == type) return;
		fUpdateType |= type;
		if (fUpdateScheduled) return;

		/*
		 * Another window could be in the foreground. If so, the Device.gUpdateRunnable
		 * will not be pointing to this window and so we could update the wrong window.
		 *
		 * We need to be able to update a window that's not in the foreground in the case
		 * that user code modifies a displayable which has a popup window.
		 */
		fUpdateScheduled = true;
		if (isShown() && Device.gDisplayableUpdateRunnable == fUpdateRunnable) {
			Device.scheduleDisplayableUpdate();
		} else {
			Device.postRunnable(fUpdateRunnable);
		}
	}

	void updateLayout() {
		synchronized (Device.gDisplayableLock) {
			if ((fUpdateType & LAYOUT_CHANGED) != 0) {
				fUpdateType &= ~LAYOUT_CHANGED;
				fInvalidated = false;
				layout();
				setInitialFocus();
				repaint();
			}
		}
	}

	void update() {
		if (!fUpdateScheduled) return;
		fUpdateScheduled = false;
		updateLayout();
	}

	void paint(Graphics g) {
		g.setColor(getBackgroundColor());
		g.fillRect(0, 0, fWidth, fHeight);
		super.paint(g);
	}

	void hide() {
		super.hide();
		hideNotify();
		getDisplayPeer().removeWindow(this);
	}

	void dispose() {
		synchronized (Device.gRepaintLock) {
			fDisposed = true;
			super.dispose();
			if (fGraphics != null) {
				fGraphics.dispose();
				fGraphics = null;
			}
		}
	}

	Display getDisplay() {
		return fDisplay;
	}

	int getOriginX() {
		return fX;
	}

	int getOriginY() {
		return fY;
	}

	boolean hasFocus() {
		return true; // Displayables always have focus when they're created
	}

	boolean isShown() {
		return fShown && !fInvalidated && !fDisplay.fPeer.isObscured();
	}

	boolean isDisposed() {
		return fDisposed;
	}

	boolean isInvalidated() {
		return fInvalidated;
	}

	boolean keyPressed(int keyCode) {
		switch (keyCode) {
			case Canvas.KEYCODE_LEFT:
			case Canvas.KEYCODE_RIGHT:
			case Canvas.KEYCODE_UP:
			case Canvas.KEYCODE_DOWN:
				traverse(CanvasPeer.getGameAction(keyCode));
		}

		if (fFocusComponent != null) return fFocusComponent.keyPressed(keyCode);
		return false;
	}

	boolean keyRepeated(int keyCode) {
		switch (keyCode) {
			case Canvas.KEYCODE_LEFT:
			case Canvas.KEYCODE_RIGHT:
			case Canvas.KEYCODE_UP:
			case Canvas.KEYCODE_DOWN:
				traverse(CanvasPeer.getGameAction(keyCode));
		}

		if (fFocusComponent != null) return fFocusComponent.keyRepeated(keyCode);
		return false;
	}

	boolean keyReleased(int keyCode) {
		if (fFocusComponent != null) return fFocusComponent.keyReleased(keyCode);
		return false;
	}

	Displayable getDisplayable() {
		return null;
	}

	abstract boolean dispatchEvent(Event e);
	abstract void resize(int w, int h);
	abstract void create();
}
