package javax.microedition.media;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

/**
 * Player is the interface that defines the operations to play some media.
 */
public interface Player extends Controllable {

	/**
	 * State indicating the Player is closed.
	 */
	static public final int  CLOSED       = 0;

	/**
	 * State indicating the Player is prefetched.
	 */
	static public final int  PREFETCHED   = 300;

	/**
	 * State indicating the Player is realized.
	 */
	static public final int  REALIZED     = 200;

	/**
	 * State indicating the Player is started.
	 */
	static public final int  STARTED      = 400;

	/**
	 * Value indicating the time is unknown.
	 */
	static public final long TIME_UNKNOWN = -1;

	/**
	 * State indicating the Player is unrealized.
	 */
	static public final int  UNREALIZED   = 100;

	/**
	 * Associates a PlayerListener to the Player.
	 *
	 * @param playerListener The PlayerListener to disassociate from the Player.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 */
	public void addPlayerListener(PlayerListener playerListener);

	/**
	 * Moves the Player to the CLOSED state.
	 */
	public void close();

	/**
	 * Deallocates the Player.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 */
	public void deallocate();

	/**
	 * Returns the content type of the Player.
	 *
	 * @return The current content type.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED or UNREALIZED state.
	 */
	public String getContentType();

	/**
	 * Returns the duration of the media.
	 *
	 * @return The duration of the media.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 */
	public long getDuration();

	/**
	 * Returns the media time of the media.
	 *
	 * @return The media time of the media.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 */
	public long getMediaTime();

	/**
	 * Returns the state of the Player.
	 *
	 * @return The state of the Player.
	 */
	public int getState();

	/**
	 * Moves the Player to the PREFETCHED state.
	 *
	 * @throws MediaException        Thrown if the Player cannot be moved to the PREFETCHED state.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 *
	 * @throws SecurityException     Thrown if the caller does not have permission to prefetch the Player.
	 */
	public void prefetch() throws MediaException ;

	/**
	 * Moves the Player to the REALIZED state.
	 *
	 * @throws MediaException        Thrown if the Player cannot be moved to the REALIZED state.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 *
	 * @throws SecurityException     Thrown if the caller does not have permission to realize the Player.
	 */
	public void realize() throws MediaException ;

	/**
	 * Disassociates the PlayerListener from the Player.
	 *
	 * @param playerListener The PlayerListener to disassociate from the Player.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 */
	public void removePlayerListener(PlayerListener playerListener);

	/**
	 * Sets the number of times the Player will loop the media.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the STARTED or CLOSED state.
	 */
	public void setLoopCount(int count);

	/**
	 * Sets the Player's media time.
	 *
	 * @param mediaTime The new media time to set for the player.
	 *
	 * @return The actual media time set.
	 *
	 * @throws MediaException        Thrown if the Player's media time cannot be set.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the UNREALIZED or CLOSED state.
	 */
	public long setMediaTime(long mediaTime) throws MediaException ;

	/**
	 * Starts the player.
	 *
	 * @throws MediaException        Thrown if the Player cannot be moved to the STARTED state.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 */
	public void start() throws MediaException ;

	/**
	 * Stops the player.
	 *
	 * @throws MediaException        Thrown if the Player cannot be moved to the STOPPED state.
	 *
	 * @throws IllegalStateException Thrown if the Player is in the CLOSED state.
	 *
	 * @throws SecurityException     Thrown if the caller does not have permission to start the Player.
	 */
	public void stop() throws MediaException ;
}
