#undef CSR_WIFI_HIP_NOISY
/*
 * ---------------------------------------------------------------------------
 *  FILE:     card_sdio_intr.c
 *
 *  PURPOSE:
 *      Interrupt processing for the UniFi SDIO driver.
 *
 *      We may need another signal queue of responses to UniFi to hold
 *      bulk data commands generated by read_to_host_signals().
 *
 * Copyright (C) 2005-2009 by Cambridge Silicon Radio Ltd.
 *
 * Refer to LICENSE.txt included with this source code for details on
 * the license terms.
 * ---------------------------------------------------------------------------
 */
#include "driver/unifi.h"
#include "driver/conversions.h"
#include "card.h"
#include "xbv.h"


/* 
 * If the SDIO link is idle for this time (in milliseconds),
 * signal UniFi to go into Deep Sleep.
 * Valid return value of unifi_bh().
 */
#define UNIFI_DEFAULT_HOST_IDLE_TIMEOUT 5
/* 
 * If the UniFi has not woken up for this time (in milliseconds),
 * signal the bottom half to take action.
 * Valid return value of unifi_bh().
 */
#define UNIFI_DEFAULT_WAKE_TIMEOUT      1000


static CsrInt32 process_bh(card_t *card);
static CsrInt32 handle_host_protocol(card_t *card);

static CsrInt32 flush_fh_buffer(card_t *card);

static CsrInt32 check_fh_sig_slots(card_t *card, CsrUint16 needed);

static CsrInt32 read_to_host_signals(card_t *card);
static CsrInt32 process_to_host_signals(card_t *card);

static CsrInt32 process_bulk_data_command(card_t *card,
                                          const CsrUint8 *cmdptr,
                                          CsrInt16 cmd, CsrUint16 len);
static CsrInt32 process_clear_slot_command(card_t *card,
                                           const CsrUint8 *cmdptr);
static CsrInt32 process_fh_cmd_queue(card_t *card);
static CsrInt32 process_fh_traffic_queue(card_t *card);
static void restart_packet_flow(card_t *card);
static CsrInt32 process_clock_request(card_t *card);

#ifdef CSR_WIFI_HIP_NOISY
CsrInt16 dump_fh_buf = 0;
#endif /* CSR_WIFI_HIP_NOISY */

#ifdef CSR_WIFI_HIP_DEBUG_OFFLINE

/*
 * The unifi_debug_output buffer can be used to debug the HIP behaviour offline
 * i.e. without using the tracing functions that change the timing.
 *
 * Call unifi_debug_log_to_buf() with printf arguments to store a string into
 * unifi_debug_output. When unifi_debug_buf_dump() is called, the contents of the
 * buffer are dumped with dump_str() which has to be implemented in the
 * OS layer, during the porting exercise. The offset printed, holds the
 * offset where the last character is (always a zero).
 *
 */

#define UNIFI_DEBUG_GBUFFER_SIZE       4096
static char unifi_debug_output[UNIFI_DEBUG_GBUFFER_SIZE];
static char *unifi_dbgbuf_ptr = unifi_debug_output;
static char *unifi_dbgbuf_start = unifi_debug_output;

static void
append_char(char c)
{
    /* write char and advance pointer */
    *unifi_dbgbuf_ptr++ = c;
    /* wrap pointer at end of buffer */
    if ((unifi_dbgbuf_ptr - unifi_debug_output) >= UNIFI_DEBUG_GBUFFER_SIZE)
    {
        unifi_dbgbuf_ptr = unifi_debug_output;
    }

    /* put end-of-buffer marker after this char */
    /* DON'T increment pointer, the NUL must be overwritten by next char */
    *unifi_dbgbuf_ptr = '\0';
} /* append_char() */

static void
append_to_buf(const char *str)
{
    const char *p = str;
    while (*p) {
        append_char(*p);
        p++;
    }
    /* Update start-of-buffer pointer */
    unifi_dbgbuf_start = unifi_dbgbuf_ptr + 1;
    if ((unifi_dbgbuf_start - unifi_debug_output) >= UNIFI_DEBUG_GBUFFER_SIZE)
    {
        unifi_dbgbuf_start = unifi_debug_output;
    }
} /* append_to_buf() */

void unifi_debug_log_to_buf(const char *fmt, ...)
{
#define DEBUG_BUFFER_SIZE       80
    static char s[DEBUG_BUFFER_SIZE];
    va_list args;

    va_start(args, fmt);
    vsnprintf(s, DEBUG_BUFFER_SIZE, fmt, args);
    va_end(args);

    append_to_buf(s);
} /* unifi_debug_log_to_buf() */

void unifi_debug_buf_dump(void) {
    int offset = unifi_dbgbuf_ptr - unifi_debug_output;

    unifi_error(NULL, "HIP debug buffer offset=%d\n", offset);
    dump_str(unifi_debug_output + offset, UNIFI_DEBUG_GBUFFER_SIZE - offset);
    dump_str(unifi_debug_output, offset);
} /* unifi_debug_buf_dump() */

#endif /* CSR_WIFI_HIP_DEBUG_OFFLINE */


static INLINE CsrUint16 q_data_slots_used(const q_t *q)
{
    CsrUint16 i, data_slots_used = 0;
    
    for(i = 0; i < UNIFI_WME_NO_OF_QS; i++) {
        data_slots_used += q_slots_used(&q[i]);
    }
    return data_slots_used;
}

/*
 * ---------------------------------------------------------------------------
 *  unifi_sdio_interrupt_handler
 *
 *      This function should be called by the OS-dependent code to handle
 *      an SDIO interrupt from the UniFi.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      None.
 *
 *  Notes: This function may be called in DRS context. In this case,
 *         tracing with the unifi_trace(), etc, is not allowed.
 * ---------------------------------------------------------------------------
 */
void
unifi_sdio_interrupt_handler(card_t *card)
{
    /* 
     * Set the flag to say reason for waking was SDIO interrupt.
     * Then ask the OS layer to run the unifi_bh to give attention to the UniFi.
     */
    card->bh_reason_unifi = 1;
    unifi_run_bh(card->ospriv);

} /*  sdio_interrupt_handler() */


/*
 * ---------------------------------------------------------------------------
 *  unifi_configure_low_power_mode
 *
 *      This function should be called by the OS-dependent when
 *      the deep sleep signaling needs to be enabled or disabled.
 * 
 *  Arguments:
 *      card            Pointer to card context structure.
 *      low_power_mode  Disable/Enable the deep sleep signaling
 *      periodic_wake_mode UniFi wakes host periodically.
 *
 *  Returns:
 *      0 on success or a CSR error code.
 * ---------------------------------------------------------------------------
 */
CsrInt32
unifi_configure_low_power_mode(card_t *card,
                               enum unifi_low_power_mode low_power_mode,
                               enum unifi_periodic_wake_mode periodic_wake_mode)
{
    card->low_power_mode = low_power_mode;
    card->periodic_wake_mode = periodic_wake_mode;

    unifi_trace(card->ospriv, UDBG1,
                "unifi_configure_low_power_mode: new mode = %s, wake_host = %s\n",
                (low_power_mode == UNIFI_LOW_POWER_DISABLED) ? "disabled" : "enabled",
                (periodic_wake_mode == UNIFI_PERIODIC_WAKE_HOST_DISABLED) ? "FALSE" : "TRUE");
    return 0;
} /* unifi_configure_low_power_mode() */


/*
 * ---------------------------------------------------------------------------
 *  unifi_force_low_power_mode
 *
 *      This function should be called by the OS-dependent when
 *      UniFi needs to be set to the low power mode (e.g. on suspend)
 * 
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      0 on success or an error code.
 * ---------------------------------------------------------------------------
 */
CsrInt32
unifi_force_low_power_mode(card_t *card)
{

    if (card->low_power_mode == UNIFI_LOW_POWER_DISABLED) {
        unifi_error(card->ospriv, "Attempt to set mode to TORPID when lower power mode is disabled\n");
        return -CSR_EINVAL;
    }

    return unifi_set_host_state(card, UNIFI_HOST_STATE_TORPID);

} /* unifi_force_low_power_mode() */


/*
 * ---------------------------------------------------------------------------
 *  unifi_bh
 *
 *      This function should be called by the OS-dependent code when
 *      host and/or UniFi has requested an exchange of messages.
 * 
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      0 on success or a CSR error code.
 * ---------------------------------------------------------------------------
 */
CsrInt32
unifi_bh(card_t *card, CsrUint32 *remaining)
{
    CsrInt16 reason_unifi, reason_host;
    CsrInt32 r;
    const enum unifi_low_power_mode low_power_mode = card->low_power_mode;

    reason_unifi = reason_host = 0;
    /* Disable the SDIO interrupts while doing SDIO ops */
    r = CsrSdioInterruptDisable(card->sdio_if);
    if (r == -CSR_ENODEV) {
        goto exit;
    }
    if (r) {
        unifi_error(card->ospriv, "Failed to disable SDIO interrupts. unifi_bh queues error.\n");
        goto exit;
    }

    /* Process request to raise the maximum SDIO clock */
    r = process_clock_request(card);
    if (r) {
        unifi_error(card->ospriv, "Error setting maximum SDIO clock\n");
        goto exit;
    }

    /* 
     * Why was the BH thread woken?
     * If it was an SDIO interrupt, UniFi is awake and we need to process it.
     * If it was a host process queueing data, then we need to awaken UniFi.
     *
     * Priority of flags is top down.
     *
     * ----------------------------------------------------------+
     *    \state|   AWAKE      |    DROWSY      |    TORPID      |
     * flag\    |              |                |                |
     * ---------+--------------+----------------+----------------|
     *          | do the host  | go to AWAKE and| go to AWAKE and|
     *   unifi  | protocol     | do the host    | do the host    |
     *          |              | protocol       | protocol       |
     * ---------+--------------+----------------+----------------|
     *          | do the host  |                |                |
     *   host   | protocol     |  do nothing    | go to DROWSY   |
     *          |              |                |                |
     * ---------+--------------+----------------+----------------|
     *          |              |                | should not     |
     *  timeout | go to TORPID | error, unifi   | occur          |
     *          |              | didn't wake up | do nothing     |
     * ----------------------------------------------------------+
     * 
     * Note that if we end up in the AWAKE state we always do the host protocol.
     */

#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE
    card->cmd_prof.cmd52_count = 0;
    card->cmd_prof.cmd53_count = 0;
    card->cmd_prof.tx_count = 0;
    card->cmd_prof.tx_cfm_count = 0;
    card->cmd_prof.rx_count = 0;
    card->cmd_prof.bh_count = 0;
    card->cmd_prof.protocol_count = 0;
    card->cmd_prof.process_count = 0;
#endif

    do {
        /*
         * Copy and clear the reason flags before we process them
         * to avoid race condition.
         */
        reason_unifi = card->bh_reason_unifi;
        reason_host  = card->bh_reason_host;
        card->bh_reason_unifi = 0;
        card->bh_reason_host  = 0;

        r = 0;
        switch (card->host_state)
        {
          case UNIFI_HOST_STATE_AWAKE:
            if (reason_unifi || reason_host) {
                unifi_trace(card->ospriv, UDBG5, "UNIFI_HOST_STATE_AWAKE: Process b-h.\n");
                (*remaining) = 0;
                break;
            }
            /*
             * if the state is AWAKE and we did not receive a request and the timeout
             * is zero, it means that the host and UniFi are idle long enough to let
             * UniFi go to sleep.
             */
            if (((*remaining) == 0) && (low_power_mode == UNIFI_LOW_POWER_ENABLED)) {
                /* Time to let UniFi go to sleep */
                unifi_trace(card->ospriv, UDBG5, "UNIFI_HOST_STATE_AWAKE: Set state to TORPID.\n");
                r = unifi_set_host_state(card, UNIFI_HOST_STATE_TORPID);
            } else {
                /* There is nothing to process, return immediately. */
                r = CsrSdioInterruptEnable(card->sdio_if);
                if (r == -CSR_ENODEV) {
                    goto exit;
                }
                r = 0;
                goto exit;
            }

            break;

          case UNIFI_HOST_STATE_DROWSY:
            if (reason_unifi) {
                /*
                 * If UniFi raised SDIO interrupt, then move to directly to AWAKE
                 * and do the host interface protocol.
                 */
                unifi_trace(card->ospriv, UDBG5, "UNIFI_HOST_STATE_DROWSY: Set state to AWAKE.\n");
                r = unifi_set_host_state(card, UNIFI_HOST_STATE_AWAKE);
                (*remaining) = 0;
                break;
            }

            if ((*remaining) == 0) {

                unifi_error(card->ospriv, "UniFi did not wake up on time...\n");

                /* Check if Function1 has gone away. */
                r = unifi_check_io_status(card);
                if (r == -CSR_ENODEV) {
                    goto exit;
                }
                if (r < 0) {
                    unifi_error(card->ospriv, "Failed to read SDIO_IO_ENABLE to check for spontaneous reset\n");
                }
                else
                {
                    /* See if we missed an SDIO interrupt */
                    r = CardPendingInt(card);
                    if (r == 1) {
                        unifi_error(card->ospriv, "There is an unhandled pending interrupt.\n");
                    }
                }

                /* Need to reset and reboot */
                r = -CSR_EIO;
                goto exit;
            } else {
                /* There is nothing to process. */
                r = 0;
            }
            break;

          case UNIFI_HOST_STATE_TORPID:
            if (reason_unifi) {
                /*
                 * If UniFi raised SDIO interrupt, then move to directly to AWAKE
                 * and do the host interface protocol.
                 */
                r = CardPendingInt(card);
                if (r == 1) {
                    unifi_trace(card->ospriv, UDBG5, "UNIFI_HOST_STATE_TORPID: Set state to AWAKE.\n");
                    r = unifi_set_host_state(card, UNIFI_HOST_STATE_AWAKE);
                    (*remaining) = 0;
                    break;
                }
                else {
                    if (r < 0) {
                        goto exit;
                    }
                    reason_unifi = 0;
                }
            }

            /* If the interrupt was spurious, a host request needs to wake up UniFi. */
            if (reason_host && (!reason_unifi))
            {
                unifi_trace(card->ospriv, UDBG5, "UNIFI_HOST_STATE_TORPID: Set state to DROWSY.\n");
                r = unifi_set_host_state(card, UNIFI_HOST_STATE_DROWSY);
                /* 
                 * Need a timeout to catch the case that UniFi dies while
                 * asleep, in which case it will never wake up.
                 */
                if (r == 0) {
                    /* set the return value to UNIFI_DEFAULT_WAKE_TIMEOUT to capture a wake error.*/
                    (*remaining) = UNIFI_DEFAULT_WAKE_TIMEOUT;
                }
            } else {
                /* There is nothing to process. */
                r = 0;
            }
            break;

          default:
            unifi_error(card->ospriv, "Bad state in bh: %d\n", card->host_state);
            break;
        }

        /* Check for an error from one of the above unifi_set_host_state() calls */
        if (r) {
            goto exit;
        }

        /* Break out and return, need to wait for the wake-up interrupt */
        if (card->host_state == UNIFI_HOST_STATE_DROWSY) {
            break;
        }

        /* UniFi is awake, do the host protocol */
        if (card->host_state == UNIFI_HOST_STATE_AWAKE) {
#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE
            card->cmd_prof.protocol_count ++;
#endif
            r = process_bh(card);
            if (r < 0) {
                goto exit;
            }
        }

#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE
        card->cmd_prof.bh_count ++;
#endif
    } while (card->bh_reason_unifi || card->bh_reason_host);

#if defined (CSR_WIFI_HIP_DEBUG_OFFLINE) && defined (CSR_WIFI_HIP_DATA_PLANE_PROFILE)
    unifi_debug_log_to_buf("runs=%d proc=%d prot=%d\n",
                           card->cmd_prof.bh_count,
                           card->cmd_prof.process_count,
                           card->cmd_prof.protocol_count);
#endif

    /*
     * If host is now idle, schedule a timer for the delay before we
     * let UniFi go into deep sleep.
     * If the timer goes off, we will move to TORPID state.
     * If UniFi raises an interrupt in the meantime, we will cancel
     * the timer and start a new one when we become idle.
     */
    if (card->host_state == UNIFI_HOST_STATE_AWAKE) {
        if ((low_power_mode == UNIFI_LOW_POWER_ENABLED) &&
            (q_data_slots_used(card->fh_traffic_queue) == 0))
        {
            if (card->ta_sampling.traffic_type != unifi_TrafficPeriodic) {
                /* return the UNIFI_DEFAULT_HOST_IDLE_TIMEOUT, so we can go to sleep. */
                unifi_trace(card->ospriv, UDBG5, "Traffic is not periodic, set timer for TORPID.\n");
                (*remaining) = UNIFI_DEFAULT_HOST_IDLE_TIMEOUT;
            } else {
                unifi_trace(card->ospriv, UDBG5, "Traffic is periodic, set unifi to TORPID immediately.\n");
                r = unifi_set_host_state(card, UNIFI_HOST_STATE_TORPID);
            }
        }
    }

    r = CsrSdioInterruptEnable(card->sdio_if);

    if (r == -CSR_ENODEV) {
    }
    if (r) {
        unifi_error(card->ospriv, "Failed to enable SDIO interrupt\n");
    }

exit:
    CsrSdioInterruptAcknowledge(card->sdio_if);

    unifi_trace(card->ospriv, UDBG4, "New state=%d\n", card->host_state);

    if (r) {
        unifi_error(card->ospriv,
                    "unifi_bh: state=%d, clock=%dHz, interrupt=%d host=%d, power_save=%s\n",
                    card->host_state, card->sdio_clock_speed,
                    reason_unifi, reason_host,
                    (low_power_mode == UNIFI_LOW_POWER_DISABLED) ? "disabled" : "enabled");
    }
    return r;
} /* unifi_bh() */


/*
 * ---------------------------------------------------------------------------
 *  process_clock_request
 *
 *      Handle request from the OS layer to increase the SDIO clock speed.
 *      The fast clock is limited until the firmware has indicated that it has
 *      completed initialisation to the OS layer. 
 * 
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      0 on success or CSR error code.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
process_clock_request(card_t *card)
{
    CsrInt32 r = 0;

    if (!card->request_max_clock) {
        return 0;   /* No pending request */
    }

    /*
     * The SDIO clock speed request from the OS layer is only acted upon if
     * the UniFi is awake. If it was in any other state, the clock speed will
     * transition through SAFE to MAX while the host wakes it up, and the
     * final speed reached will be UNIFI_SDIO_CLOCK_MAX.
     * This assumes that the SME never requests low power mode while the f/w
     * initialisation takes place.
     */
    if (card->host_state == UNIFI_HOST_STATE_AWAKE) {
        unifi_trace(card->ospriv, UDBG1, "Set SDIO max clock\n");
        r = csr_sdio_set_max_clock_speed(card->sdio_if, UNIFI_SDIO_CLOCK_MAX);
    } else {
        unifi_trace(card->ospriv, UDBG1, "Will set SDIO max clock after wakeup\n");
    }

    /* Cancel the request now that it has been acted upon, or is about to be
     * by the wakeup mechanism
     */
    card->request_max_clock = 0;
    
    return r;
}

/*
 * ---------------------------------------------------------------------------
 *  process_bh
 *
 *      Exchange messages with UniFi
 * 
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      0 on success or CSR error code.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
process_bh(card_t *card)
{
    CsrInt32 r, more = 0;

    /* Process the reasons (interrupt, signals) */
    do {
        /*
         * Run in a while loop, to save clearing the interrupts
         * every time around the outside loop.
         */
        do {
            r = handle_host_protocol(card);
            if (r < 0) {
                return r;
            }
            more = r;

#if defined (CSR_WIFI_HIP_DEBUG_OFFLINE) && defined (CSR_WIFI_HIP_DATA_PLANE_PROFILE)
            unifi_debug_log_to_buf("c52=%d c53=%d tx=%d txc=%d rx=%d\n",
                                   card->cmd_prof.cmd52_count,
                                   card->cmd_prof.cmd53_count,
                                   card->cmd_prof.tx_count,
                                   card->cmd_prof.tx_cfm_count,
                                   card->cmd_prof.rx_count);
            card->cmd_prof.cmd52_count = card->cmd_prof.cmd53_count = 0;
            card->cmd_prof.tx_count = card->cmd_prof.tx_cfm_count = card->cmd_prof.rx_count = 0;
#endif

        } while (more);

        /* Acknowledge the h/w interrupt */
        r = CardClearInt(card);
        if (r) {
            unifi_error(card->ospriv, "Failed to acknowledge interrupt.\n");
            return r;
        }

        /*
         * UniFi may have tried to generate an interrupt during the
         * CardClearInt() was running. So, we need to run the host
         * protocol again, to check if there are any pending requests.
         */
        r = handle_host_protocol(card);
        if (r < 0) {
            return r;
        }
        more = r;

#if defined (CSR_WIFI_HIP_DEBUG_OFFLINE) && defined (CSR_WIFI_HIP_DATA_PLANE_PROFILE)
        unifi_debug_log_to_buf("c52=%d c53=%d tx=%d txc=%d rx=%d\n",
                               card->cmd_prof.cmd52_count,
                               card->cmd_prof.cmd53_count,
                               card->cmd_prof.tx_count,
                               card->cmd_prof.tx_cfm_count,
                               card->cmd_prof.rx_count);
        card->cmd_prof.cmd52_count = card->cmd_prof.cmd53_count = 0;
        card->cmd_prof.tx_count = card->cmd_prof.tx_cfm_count = card->cmd_prof.rx_count = 0;
#endif

    } while (more);

    return 0;
} /* process_bh() */


/*
 * ---------------------------------------------------------------------------
 *  handle_host_protocol
 *
 *      This function implements the Host Interface Protocol (HIP) as
 *      described in the Host Interface Protocol Specification.
 * 
 *  Arguments:
 *      card            Pointer to card context structure.
 *
 *  Returns:
 *      0 on success or CSR error code.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
handle_host_protocol(card_t *card)
{
    CsrInt32 processed_something = 0;
    CsrInt32 r;

#ifdef CSR_WIFI_HIP_NOISY
    unifi_error(card->ospriv, "   ========================     \n");
#endif /* CSR_WIFI_HIP_NOISY */

#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE
    card->cmd_prof.process_count ++;
#endif

    card->generate_interrupt = 0;


    /*
     * (Re)fill the T-H signal buffer
     */
    r = read_to_host_signals(card);
    if (r < 0) {
        unifi_error(card->ospriv, "Error occured reading to-host signals\n");
        return r;
    }
    if (r > 0) {
        processed_something = 1;
    }

    /* 
     * Process any to-host signals.
     * Perform any requested CMD53 transfers here, but just queue any
     * bulk data command responses.
     */
    r = process_to_host_signals(card);
    if (r < 0) {
        unifi_error(card->ospriv, "Error occured processing to-host signals\n");
        return r;
    }

    /* Now send any signals in the F-H queues */
    /* Give precedence to the command queue */
    r = process_fh_cmd_queue(card);
    if (r < 0) {
        unifi_error(card->ospriv, "Error occured processing from-host signals\n");
        return r;
    }
    if (r > 0) {
        processed_something = 1;
    }

    r = process_fh_traffic_queue(card);
    if (r < 0) {
        unifi_error(card->ospriv, "Error occured processing from-host data signals\n");
        return r;
    }
    if (r > 0) {
        processed_something = 1;
    }

    /* Flush out the batch of signals to the UniFi. */
    r = flush_fh_buffer(card);
    if (r < 0) {
        unifi_error(card->ospriv, "Failed to copy from-host signals to UniFi\n");
        return r;
    }


    /*
     * Send the host interrupt to say the queues have been modified.
     */
    if (card->generate_interrupt) {
        r = CardGenInt(card);
        if (r < 0) {
            unifi_error(card->ospriv, "Failed to notify UniFi that queues have been modified.\n");
            return r;
        }
    }

    /* See if we can re-enable transmission now */
    restart_packet_flow(card);


    /* 
     * Don't put the thread sleep if we just interacted with the chip,
     * there might be more to do if we look again.
     */
    return processed_something;
} /* handle_host_protocol() */



/*
 * ---------------------------------------------------------------------------
 *  get_chunks_for
 *
 *      Rounds the given signal length in bytes to a whole number
 *      of sig_frag_size.
 * 
 *  Arguments:
 *      card            Pointer to card context structure.
 *      len             The signal length in bytes to convert
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static INLINE CsrUint16
get_chunks_for(const card_t *card, CsrUint16 len)
{
    CsrUint16 mul = card->config_data.sig_frag_size;

    return (len + (mul - 1)) / mul;
} /* get_chunks_for() */


/*
 * ---------------------------------------------------------------------------
 *  read_shared_count
 *
 *      Read signal count locations, checking for an SDIO error.  The
 *      signal count locations only contain a valid number if the
 *      highest bit isn't set.
 *
 *  Arguments:
 *      card            Pointer to card context structure.
 *      addr            Shared-memory address to read.
 *
 *  Returns:
 *      Value read from memory (0-127) or CSR negative error code.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
read_shared_count(card_t *card, CsrUint32 addr)
{
    CsrUint8 b;
    /* I've increased this count, because I have seen cases where
     * there were three reads in a row with the top bit set.  I'm not
     * sure why this might have happened, but I can't see a problem
     * with increasing this limit.  It's better to take a while to
     * recover than to fail. */
#define SHARED_READ_RETRY_LIMIT 10
    CsrInt32 r, i;

    /* 
     * Get the to-host-signals-written count.
     * The top-bit will be set if the firmware was in the process of
     * changing the value, in which case we read again.
     */
    /* Limit the number of repeats so we don't freeze */
    for (i=0; i<SHARED_READ_RETRY_LIMIT; i++) {
        r = unifi_read_8_or_16(card, addr, &b);
        if (r) {
            return r;
        }
        if (!(b & 0x80)) {
            /* There is a chance that the MSB may have contained invalid data
             * (overflow) at the time it was read. Therefore mask off the MSB.
             * This avoids a race between driver read and firmware write of the
             * word, the value we need is in the lower 8 bits anway.
             */
            return (CsrInt32)(b & 0xff);
        }
    }

    return -CSR_EIO;
} /* read_shared_count() */



/*
 * ---------------------------------------------------------------------------
 *  read_to_host_signals
 *
 *      Read everything pending in the UniFi TH signal buffer.
 *      Only do it if the local buffer is empty.
 *
 *  Arguments:
 *      card    Pointer to card context struct
 *
 *  Returns:
 *      0 if there were no signals pending,
 *      1 if we read at least one signal
 *      CSR error code if an error occurred.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
read_to_host_signals(card_t *card)
{
    CsrInt32 count_thw, count_thr;
    CsrInt32 unread_chunks, unread_bytes;
    CsrInt32 r;

    /* Read any pending signals or bulk data commands */
    r = read_shared_count(card, card->sdio_ctrl_addr+4);
    if (r < 0) {
        unifi_error(card->ospriv, "Failed to read to-host sig written count\n");
        return r;
    }
    count_thw = r;
    card->to_host_signals_w = count_thw; /* diag */

    count_thr = card->to_host_signals_r;

    if (count_thw == count_thr) {
        return 0;
    }

    unread_chunks = 
        (((count_thw - count_thr) + 128) % 128) - card->th_buffer.count;

    if (unread_chunks == 0) {
        return 0;
    }

    unread_bytes = card->config_data.sig_frag_size * unread_chunks;


    r = unifi_bulk_rw(card,
                      card->config_data.tohost_sigbuf_handle,
                      card->th_buffer.ptr,
                      unread_bytes,
                      UNIFI_SDIO_READ);
    if (r) {
        unifi_error(card->ospriv, "Failed to read ToHost signal\n");
        return r;
    }

    card->th_buffer.ptr += unread_bytes;
    card->th_buffer.count += (CsrUint16)unread_chunks;

    return 1;

} /* read_to_host_signals() */


/*
 * ---------------------------------------------------------------------------
 *  update_to_host_signals_r
 *
 *      Advance the shared-memory count of chunks read from the to-host
 *      signal buffer.
 *      Raise a UniFi internal interrupt to tell the firmware that the
 *      count has changed.
 * 
 *  Arguments:
 *      card            Pointer to card context struct
 *      pending         Number of chunks remaining
 *
 *  Returns:
 *      0 on success or CSR error code
 * ---------------------------------------------------------------------------
 */
static CsrInt32
update_to_host_signals_r(card_t *card, CsrInt16 pending)
{
    CsrInt32 r;

    card->to_host_signals_r =
        (card->to_host_signals_r + (card->th_buffer.count - pending)) % 128;
    card->th_buffer.count = pending;

    /* Update the count of signals read */
    r = unifi_write_8_or_16(card, card->sdio_ctrl_addr+6,
                            (CsrUint8)card->to_host_signals_r);
    if (r) {
        unifi_error(card->ospriv, "Failed to update to-host signals read\n");
        return r;
    }

    r = CardGenInt(card);
    if (r) {
        unifi_error(card->ospriv, "Failed to notify UniFi that we processed to-host signals.\n");
        return r;
    }

    card->generate_interrupt = 0;

    return 0;
} /* update_to_host_signals_r() */



/*
 * ---------------------------------------------------------------------------
 *  read_unpack_cmd
 *
 *      Converts a wire-formatted command to the host bulk_data_cmd_t structure.
 * 
 *  Arguments:
 *      ptr             Pointer to the command
 *      bulk_data_cmd   Pointer to the host structure
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void
read_unpack_cmd(const CsrUint8 *ptr, bulk_data_cmd_t *bulk_data_cmd)
{
    CsrInt16 index = 0;
    bulk_data_cmd->cmd_and_len = COAL_GET_UINT16_FROM_LITTLE_ENDIAN(ptr + index);
    index += SIZEOF_UINT16;
    bulk_data_cmd->data_slot = COAL_GET_UINT16_FROM_LITTLE_ENDIAN(ptr + index);
    index += SIZEOF_UINT16;
    bulk_data_cmd->offset = COAL_GET_UINT16_FROM_LITTLE_ENDIAN(ptr + index);
    index += SIZEOF_UINT16;
    bulk_data_cmd->buffer_handle = COAL_GET_UINT16_FROM_LITTLE_ENDIAN(ptr + index);
    index += SIZEOF_UINT16;
} /* read_unpack_cmd */


/*
 * ---------------------------------------------------------------------------
 *  process_to_host_signals
 *
 *      Read and dispatch signals from the UniFi
 *
 *  Arguments:
 *      card    Pointer to card context struct
 *
 *  Returns:
 *      0 if there were no signals pending,
 *      1 if we read at least one signal
 *      CSR error code if there was an error
 *
 *  Notes:
 *      Since bulk data transfers can take a long time, if we wait until
 *      all are done before we acknowledge the signals, the UniFi runs out
 *      of buffer space. Therefore we keep a count of the bytes transferred
 *      in bulk data commands, and update the to-host-signals-read count
 *      if we've done a large transfer.
 *
 *      All data in the f/w is stored in a little endian format, without any
 *      padding bytes. Every read from the memory has to be transformed in
 *      host (cpu specific) format, before we can process it. Therefore we 
 *      use read_unpack_cmd() and read_unpack_signal() to convert the raw data
 *      contained in the card->th_buffer.buf to host structures.
 *      Important: UDI clients use wire-formatted structures, so we need to 
 *      indicate all data, as we have read it from the device.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
process_to_host_signals(card_t *card)
{
    CsrInt16 pending;
    CsrInt16 remaining;
    CsrUint8 *bufptr;
    bulk_data_param_t data_ptrs;
    CsrInt16 cmd;
    CsrUint16 sig_len;
    CsrInt16 i;
    CsrUint16 chunks_in_buf;
    CsrUint16 bytes_transferred = 0;
    CsrInt32 r=0;

    pending = card->th_buffer.count;

    /* Are there new to-host signals? */
    unifi_trace(card->ospriv, UDBG4, "handling %d to-host chunks\n", pending);

    if (!pending) {
        return 0;
    }

    /*
     * This is a pointer to the raw data we have read from the f/w.
     * Can be a signal or a command. Note that we need to convert
     * it to a host structure before we process it.
     */
    bufptr = card->th_buffer.buf;

    while (pending > 0)
    {
        CsrInt16 f_flush_count = 0;

        /*
         * Command and length are common to signal and bulk data msgs.
         * If command == 0 (i.e. a signal), len is number of bytes
         * *following* the 2-byte header.
         */
        cmd = bufptr[1] >> 4;
        sig_len = bufptr[0] + ((bufptr[1] & 0x0F) << 8);

#ifdef CSR_WIFI_HIP_NOISY
        unifi_error(card->ospriv, "Received UniFi msg cmd=%d, len=%d\n",
                    cmd, sig_len);
#endif /* CSR_WIFI_HIP_NOISY */

        if ((sig_len == 0) &&
            ((cmd != SDIO_CMD_CLEAR_SLOT) && (cmd != SDIO_CMD_PADDING))) {
            unifi_error(card->ospriv, "incomplete signal or command: has size zero\n");
            return -CSR_EIO;
        }
        /* 
         * Make sure the buffer contains a complete message.
         * Signals may occupy multiple chunks, bulk-data commands occupy
         * one chunk.
         */
        if (cmd == SDIO_CMD_SIGNAL) {
            chunks_in_buf = get_chunks_for(card, sig_len + 2);
        } else {
            chunks_in_buf = 1;
        }

        if (chunks_in_buf > (CsrUint16)pending) {
            unifi_error(card->ospriv, "incomplete signal: need %d chunks, got %d\n",
                        chunks_in_buf, pending);
            unifi_error(card->ospriv, " thsw=%d, thsr=%d\n",
                        card->to_host_signals_w,
                        card->to_host_signals_r);
            return -CSR_EIO;
        }


        switch (cmd) {
          case SDIO_CMD_SIGNAL:
            /* This is a signal. Read the rest of it and then handle it. */

            for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++) {
                /* Retrieve dataRefs[i].DataLength */
                CsrUint16 data_len = GET_PACKED_DATAREF_LEN(bufptr+2, i);

                /* 
                 * The bulk data length in the signal can not be greater than
                 * the maximun length allowed by the SDIO config structure.
                 */
                if (data_len > card->config_data.data_slot_size) {
                    unifi_error(card->ospriv,
                                "Bulk Data length (%d) exceeds Maximum Bulk Data length (%d)\n",
                                data_len, card->config_data.data_slot_size);
                    return -CSR_EIO;
                }

                /* 
                 * Len here might not be the same as the length in the
                 * bulk data slot.  The slot length will always be even,
                 * but len could be odd.
                 */
                if (data_len != 0)
                {
                    /* Retrieve dataRefs[i].SlotNumber */
                    CsrInt16 slot = GET_PACKED_DATAREF_SLOT(bufptr+2, i);

                    if (slot >= card->config_data.num_tohost_data_slots) {
                        unifi_error(card->ospriv, "!!!bad slot number in to-host signal: %d, sig 0x%X\n", 
                                    slot, cmd);
                        return -CSR_EIO;
                    }

                    data_ptrs.d[i].os_data_ptr = card->to_host_data[slot].os_data_ptr;
                    data_ptrs.d[i].os_net_buf_ptr = card->to_host_data[slot].os_net_buf_ptr;
                    data_ptrs.d[i].net_buf_length = card->to_host_data[slot].net_buf_length;
                    data_ptrs.d[i].data_length = data_len;

                }
                else
                {
                    unifi_init_bulk_data(&data_ptrs.d[i]);
                }
            }

            /*
             * Log the signal to the UDI, before call unifi_receive_event() as 
             * it can modify the bulk data.
             */
            if (card->udi_hook) {
                (*card->udi_hook)(card->ospriv, bufptr+2, sig_len,
                                  &data_ptrs, UDI_LOG_TO_HOST);
            }

#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE
            if (GET_SIGNAL_ID(bufptr+2) == CSR_MA_UNITDATA_CONFIRM_ID) {
                card->cmd_prof.tx_cfm_count ++;
            } else if (GET_SIGNAL_ID(bufptr+2) == CSR_MA_UNITDATA_INDICATION_ID) {
                card->cmd_prof.rx_count ++;
            }
#endif

            /* Pass event to OS layer */
            unifi_receive_event(card->ospriv, bufptr+2, sig_len, &data_ptrs);

            /* Initialise the to_host data, so it can be re-used. */
            for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++) { 
                /* The slot is only valid if the length is non-zero. */
                if (GET_PACKED_DATAREF_LEN(bufptr+2, i) != 0) {
                    CsrInt16 slot = GET_PACKED_DATAREF_SLOT(bufptr+2, i);
                    if (slot < card->config_data.num_tohost_data_slots) {
                        unifi_init_bulk_data(&card->to_host_data[slot]);
                    }
                }
            }

            /* 
             * If we have previously transferred a lot of data, ack
             * the signals read so far, so f/w can reclaim the buffer
             * memory sooner.
             */
            if (bytes_transferred >= TO_HOST_FLUSH_THRESHOLD) {
                f_flush_count = 1;
            }
            break;


          case SDIO_CMD_CLEAR_SLOT:
            /* This is a clear slot command. */
            if (sig_len != 0) {
                unifi_error(card->ospriv, "process_to_host_signals: clear slot, bad data len: 0x%X at offset %d\n",
                            sig_len, bufptr - card->th_buffer.buf);
                return -CSR_EIO;
            }

            r = process_clear_slot_command(card, bufptr);
            if (r < 0) {
                unifi_error(card->ospriv, "Failed to process clear slot\n");
                return r;
            }
            break;

          case SDIO_CMD_TO_HOST_TRANSFER:
          case SDIO_CMD_FROM_HOST_TRANSFER:
          case SDIO_CMD_FROM_HOST_AND_CLEAR:
          case SDIO_CMD_OVERLAY_TRANSFER:
            /* This is a bulk data command. */
            if (sig_len & 1) {
                unifi_error(card->ospriv, "process_to_host_signals: bulk data, bad data len: 0x%X at offset %d\n",
                            sig_len, bufptr - card->th_buffer.buf);
                return -CSR_EIO;
            }

            r = process_bulk_data_command(card, bufptr, cmd, sig_len);
            if (r < 0) {
                unifi_error(card->ospriv, "Failed to process bulk cmd\n");
                return r;
            }
            /* Count the bytes transferred */
            bytes_transferred += sig_len;

            if (cmd == SDIO_CMD_FROM_HOST_AND_CLEAR) {
                f_flush_count = 1;
            }
            break;

          case SDIO_CMD_PADDING:
            break;
 
          default:
            unifi_error(card->ospriv, "Unrecognised to-host command: %d\n", cmd);
            break;
        }

        bufptr += chunks_in_buf * card->config_data.sig_frag_size;
        pending -= chunks_in_buf;

        /* 
         * Write out the host signal count when a significant
         * number of bytes of bulk data have been transferred or
         * when we have performed a CopyFromHostAndClear.
         */
        if (f_flush_count) {
            r = update_to_host_signals_r(card, pending);
            if (r) {
                return r;
            }
            bytes_transferred = 0;
        }

    }

    if (pending) {
        unifi_warning(card->ospriv, "proc_th_sigs: %d unprocessed\n", pending);
    }

    /* If we processed any signals, write the updated count to UniFi */
    if (card->th_buffer.count != pending)
    {
        r = update_to_host_signals_r(card, pending);
        if (r) {
            return r;
        }
    }

    /*
     * Reset the buffer pointer, copying down any un-processed signals.
     * This can happen if we enable the optimisation in read_to_host_signals()
     * that limits the length to whole blocks.
     */
    remaining = card->th_buffer.ptr - bufptr;
    if (remaining < 0) {
        unifi_error(card->ospriv, "Processing TH signals overran the buffer\n");
        return -CSR_EIO;
    }
    if (remaining > 0) {
        /* Use a safe copy because source and destination may overlap */
        CsrUint8 *d = card->th_buffer.buf;
        CsrUint8 *s = bufptr;
        CsrInt32 n = remaining;
        while (n--) {
            *d++ = *s++;
        }
    }
    card->th_buffer.ptr = card->th_buffer.buf + remaining;


    /* If we reach here then we processed something */
    return 1;
} /* process_to_host_signals() */



/*
 * ---------------------------------------------------------------------------
 *  process_overlay_cmd
 *
 *      Process a f/w overlay request from the UniFi.  This function finds
 *      the sectionof the FWOV from the XBV and returns it to the caller.
 *      The caller must then send it to UniFi.
 *
 *  Arguments:
 *   card       Pointer to card context struct
 *   bdslot     A Bulk Data slot to be filled with the correct section of 
 *              the fimware overlay.
 *   fw_offset  The offset into the firmware overlay section requested by 
 *              the firmware
 *   req_len    The length of firmware data reqquested by the firmware.
 *
 *  Returns:
 *      0 on success, CSR error code on error
 * ---------------------------------------------------------------------------
 */
static CsrInt32
process_overlay_cmd(card_t *card,
                    bulk_data_desc_t *bdslot, CsrUint32 fw_offset, CsrUint32 req_len)
{
    void *dlpriv;
    CsrUint32 file_read_len;
    CsrInt32 r;
    const struct FWOV *fwov = &card->fwov;

    func_enter();

    /* Start reading the f/w file */
    dlpriv = unifi_dl_fw_read_start(card, UNIFI_FW_STA);
    if (dlpriv == NULL) {
        unifi_error(card->ospriv, "f/w file read failed\n");
        func_exit();
        return -CSR_EINVAL;
    }

    /*
     * The request length might be rounded-up to the UNIFI_IO_SDIO_SIZE size
     * which means that we my don't have enough data in the file to give to
     * the f/w. In this case we need to pad the buffer.
     */
    if (fw_offset + req_len > fwov->dl_size) {
        if (req_len % card->sdio_io_block_size == 0) {
            /* Firmware file not big enough, the buffer will be padded. */
            if (fw_offset >= fwov->dl_size) {
                unifi_error(card->ospriv, "offset (%u) exceeds section size (%u)\n",
                            fw_offset, fwov->dl_size);
                unifi_fw_read_stop(card->ospriv, dlpriv);
                func_exit();
                return -CSR_EINVAL;
            }
            file_read_len = fwov->dl_size - fw_offset;
        } else {
            unifi_error(card->ospriv, "Request size (%u) is not an I/O block size (%u) multiple\n",
                        req_len, card->sdio_io_block_size);
            unifi_error(card->ospriv, "and FWOV section not big enough\n");
            unifi_fw_read_stop(card->ospriv, dlpriv);
            func_exit();
            return -CSR_EINVAL;
        }
    } else {
        /* We have enough data in the f/w file. */
        file_read_len = req_len;
    }

    /* Allocate a buffer with the request length. */
    r = unifi_net_data_malloc(card->ospriv, bdslot, req_len);
    if (r != 0) {
        unifi_error(card->ospriv, "Failed to allocate f/w overlay buffer\n");
        unifi_fw_read_stop(card->ospriv, dlpriv);
        func_exit();
        return -CSR_EIO;
    }

    /* Read as much data as we can from the f/w file. */
    r = unifi_fw_read(card->ospriv, dlpriv, fwov->dl_offset + fw_offset,
                      (void*)bdslot->os_data_ptr, file_read_len);
    if ((r < 0) || (r != file_read_len)) {
        unifi_error(card->ospriv, "Failed (fw_offset=%u, len=%u, ret:%d)\n",
                    fw_offset, file_read_len, r);
        unifi_net_data_free(card->ospriv, bdslot);
        unifi_fw_read_stop(card->ospriv, dlpriv);
        func_exit();
        return -CSR_EINVAL;
    }

    unifi_fw_read_stop(card->ospriv, dlpriv);

    func_exit();
    return 0;
}



/*
 * ---------------------------------------------------------------------------
 *  process_clear_slot_command
 *
 *      Process a clear slot command fom the UniFi.
 *
 *  Arguments:
 *   card       Pointer to card context struct
 *   bdcmd      Pointer to bulk-data command msg from UniFi
 *
 *  Returns:
 *      0 on success, CSR error code on error
 * ---------------------------------------------------------------------------
 */
static CsrInt32
process_clear_slot_command(card_t *card, const CsrUint8 *cmdptr)
{
    CsrUint16 data_slot;
    CsrInt16 slot;

    data_slot = COAL_GET_UINT16_FROM_LITTLE_ENDIAN(cmdptr + SIZEOF_UINT16);

    unifi_trace(card->ospriv, UDBG4, "Processing clear slot cmd, slot=0x%X\n",
                data_slot);

    slot = data_slot & 0x7FFF;

#ifdef CSR_WIFI_HIP_NOISY
    unifi_error(card->ospriv, "CMD clear data slot 0x%04x\n", data_slot);
#endif /* CSR_WIFI_HIP_NOISY */

    if (data_slot & SLOT_DIR_TO_HOST) {
        if (slot >= card->config_data.num_tohost_data_slots) {
            unifi_error(card->ospriv, 
                        "Invalid to-host data slot in SDIO_CMD_CLEAR_SLOT: %d\n",
                        slot);
            return -CSR_EIO;
        }
        /* clear to-host data slot */
        unifi_warning(card->ospriv, "Unexpected clear to-host data slot cmd: 0x%04x\n",
                      data_slot);
    } else {
        if (slot >= card->config_data.num_fromhost_data_slots) {
            unifi_error(card->ospriv, 
                        "Invalid from-host data slot in SDIO_CMD_CLEAR_SLOT: %d\n",
                        slot);
            return -CSR_EIO;
        }

        /* Set length field in from_host_data array to 0 */
        CardClearFromHostDataSlot(card, slot);
    }

    return 0;
} /* process_clear_slot_command() */



/*
 * ---------------------------------------------------------------------------
 *  process_bulk_data_command
 *
 *      Process a bulk data request from the UniFi.
 *
 *  Arguments:
 *   card       Pointer to card context struct
 *   bdcmd      Pointer to bulk-data command msg from UniFi
 *   cmd, len   Decoded values of command and length from the msg header
 *              Cmd will only be one of:
 *                      SDIO_CMD_TO_HOST_TRANSFER
 *                      SDIO_CMD_FROM_HOST_TRANSFER
 *                      SDIO_CMD_FROM_HOST_AND_CLEAR
 *                      SDIO_CMD_OVERLAY_TRANSFER
 *
 *  Returns:
 *      0 on success, CSR error code on error
 * ---------------------------------------------------------------------------
 */
static CsrInt32
process_bulk_data_command(card_t *card, const CsrUint8 *cmdptr,
                          CsrInt16 cmd, CsrUint16 len)
{
    bulk_data_desc_t *bdslot;
    bulk_data_cmd_t bdcmd;
    CsrInt16 offset;
    CsrInt16 slot;
    CsrInt16 dir;
    CsrInt32 r;
    bulk_data_desc_t overlay_bdslot;

    read_unpack_cmd(cmdptr, &bdcmd);

    unifi_trace(card->ospriv, UDBG4, "Processing bulk data cmd %d %s, len=%d, slot=0x%X\n",
          cmd, lookup_bulkcmd_name(cmd), len, bdcmd.data_slot);

    /* 
     * Round up the transfer length if required.
     * This is useful to force all transfers to be a multiple of the SDIO block
     * size, so the SDIO driver won't try to use a byte-mode CMD53. These are
     * broken on some hardware platforms.
     */
    if (card->sdio_io_block_pad) {
        len = (len + card->sdio_io_block_size - 1) & ~(card->sdio_io_block_size - 1);
        unifi_trace(card->ospriv, UDBG4, "Rounded bulk data length up to %d\n", len);
    }

    slot = bdcmd.data_slot & 0x7FFF;

    if (cmd == SDIO_CMD_OVERLAY_TRANSFER) {
        CsrUint32 fw_offset;
        /* In the case of SDIO_CMD_OVERLAY_TRANSFER, the bulk data command fields are different. */
        fw_offset = COAL_GET_UINT32_FROM_LITTLE_ENDIAN(cmdptr + SIZEOF_UINT16);

        r = process_overlay_cmd(card, &overlay_bdslot, fw_offset, len);
        if (r) {
            return r;
        }

        bdslot = &overlay_bdslot;
        offset = 0;
    } else {
        if (bdcmd.data_slot & SLOT_DIR_TO_HOST) {
            /* Request is for to-host bulk data */

            /* Check sanity of slot number */
            if (slot >= card->config_data.num_tohost_data_slots) {
                unifi_error(card->ospriv, 
                    "Invalid to-host data slot in SDIO bulk xfr req: %d\n",
                    slot);
                return -CSR_EIO;
            }

            /* Allocate memory for card->to_host_data[slot] bulk data here. */
            r = unifi_net_data_malloc(card->ospriv, &card->to_host_data[slot], len);
            if (r != 0) {
                unifi_error(card->ospriv, "Failed to allocate t-h bulk data\n");
                return -CSR_EIO;
            }

            bdslot = &card->to_host_data[slot];
        } else {
            /* Request is for from-host bulk data */

            if (slot >= card->config_data.num_fromhost_data_slots) {
                unifi_error(card->ospriv, 
                    "Invalid from-host data slot in SDIO bulk xfr req: %d\n",
                    slot);
                return -CSR_EIO;
            }
            bdslot = &card->from_host_data[slot];
        }
        offset = bdcmd.offset;
    }
    /* Do the transfer */
    dir = (cmd == SDIO_CMD_TO_HOST_TRANSFER) ?
        UNIFI_SDIO_READ : UNIFI_SDIO_WRITE;

    unifi_trace(card->ospriv, UDBG4, "Bulk %s len=%d, handle %d -- slot=%d %p+0x%X\n",
          lookup_bulkcmd_name(cmd),
          len,
          bdcmd.buffer_handle,
          slot, bdslot->os_data_ptr, offset);
#ifdef CSR_WIFI_HIP_NOISY
    unifi_error(card->ospriv, "Bulk %s len=%d, handle %d -- slot=%d %p+0x%X\n",
                lookup_bulkcmd_name(cmd),
                len,
                bdcmd.buffer_handle,
                slot, bdslot->os_data_ptr, offset);
#endif /* CSR_WIFI_HIP_NOISY */

    r = unifi_bulk_rw(card,
                      bdcmd.buffer_handle,
                      (void*)(bdslot->os_data_ptr + offset),
                      len,
                      dir);

    if (cmd == SDIO_CMD_OVERLAY_TRANSFER) {
        unifi_net_data_free(card->ospriv, &overlay_bdslot);
        return r;
    }
    if (r == -CSR_ENODEV) return r;
    if (r) {
#if 1
        unifi_error(card->ospriv, "Bulk data transfer failed:"
                    "Bulk %s len=%d, handle %d -- slot=%d %p+0x%X\n",
                    lookup_bulkcmd_name(cmd),
                    len,
                    bdcmd.buffer_handle,
                    slot, bdslot->os_data_ptr, offset);
#endif
        return r;
    }

    bdslot->data_length = len;

    if (cmd == SDIO_CMD_FROM_HOST_AND_CLEAR) {
        if (slot >= card->config_data.num_fromhost_data_slots) {
            unifi_error(card->ospriv, 
                    "Invalid from-host data slot in SDIO_CMD_FROM_HOST_AND_CLEAR: %d\n",
                    slot);
            return -CSR_EIO;
        }

        /* Set length field in from_host_data array to 0 */
        CardClearFromHostDataSlot(card, slot);
    }

    return 0;
} /* process_bulk_data_command() */



/*
 * ---------------------------------------------------------------------------
 *  check_fh_sig_slots
 *
 *      Check whether there are <n> free signal slots available on UniFi.
 *      This takes into account the signals already batched since the
 *      from_host_signal counts were last read.
 *      If the from_host_signal counts indicate not enough space, we read
 *      the latest count from UniFi to see if some more have been freed.
 *
 *  Arguments:
 *      None.
 *
 *  Returns:
 *      0, otherwise CSR error code on error.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
check_fh_sig_slots(card_t *card, CsrUint16 needed)
{
    CsrUint32 count_fhw, count_fhr;
    CsrUint32 occupied_fh, space_fh, slots_fh;

    count_fhw = card->from_host_signals_w;
    count_fhr = card->from_host_signals_r;
    slots_fh = card->config_data.num_fromhost_sig_frags;

    /* Only read the space in from-host queue if necessary */
    occupied_fh = (count_fhw - count_fhr) % 128;

    if (slots_fh < occupied_fh) {
        space_fh = 0;
    } else {
        space_fh = slots_fh - occupied_fh;
    }

    if ((occupied_fh != 0) && (space_fh < needed))
    {
        CsrInt32 r;
        r = read_shared_count(card, card->sdio_ctrl_addr+2);
        if (r < 0) {
            unifi_error(card->ospriv, "Failed to read from-host sig read count\n");
            return r;
        }
        count_fhr = r;
        card->from_host_signals_r = count_fhr; /* diag */

        occupied_fh = (count_fhw - count_fhr) % 128;
        space_fh = slots_fh - occupied_fh;
    }

    return (CsrInt32)space_fh;
} /* check_fh_sig_slots() */



static INLINE CsrUint16
free_chunks_in_fh_buffer(card_t *card)
{
    return get_chunks_for(card,
                          (card->fh_buffer.buf + UNIFI_FH_BUF_SIZE) - card->fh_buffer.ptr);
}


static INLINE CsrUint16
round_up_needed_chunks(card_t *card, CsrUint16 needed_chunks)
{
    CsrUint16 chunks_per_block;
    CsrUint16 chunks_in_last_block;

    if (!card->sdio_io_block_pad) {
        return needed_chunks;       /* Not padding */
    }
    
    /*
     * If we are padding the From-Host signals to the SDIO block size,
     * we need to round up the needed_chunks to the SDIO block size.
     */

    /* chunks in a block size */
    chunks_per_block = card->sdio_io_block_size / card->config_data.sig_frag_size;
    /* chunks out of the block size */
    chunks_in_last_block = needed_chunks % chunks_per_block;

    /* Increase the needed_chunks to be a multiple of chunks_per_block */
    if (chunks_in_last_block != 0) {
        return needed_chunks + (chunks_per_block - chunks_in_last_block);
    } else {
        return needed_chunks;
    }
}


static INLINE CsrUint16
round_up_space_chunks(card_t *card, CsrUint16 space_chunks)
{
    CsrUint16 chunks_per_block;

    if (!card->sdio_io_block_pad) {
        return space_chunks;       /* Not padding */
    }

    /* chunks in a block size */
    chunks_per_block = card->sdio_io_block_size / card->config_data.sig_frag_size;

    if ((space_chunks / chunks_per_block) == 0) {
        return 0;
    }

    return ((space_chunks / chunks_per_block) * chunks_per_block);
}





/*
 * ---------------------------------------------------------------------------
 *  process_fh_cmd_queue
 *
 *      Take one signal off the from-host queue and copy it to the UniFi.
 *      Does nothing if the UniFi has no slots free.
 *
 *  Arguments:
 *      card       Pointer to card context struct
 *
 *  Returns:
 *      0 if there is nothing on the queue to process
 *      1 if a signal was successfully processed
 *      CSR error code if an error occurred.
 *
 *  Notes:
 *      The from-host queue contains signal requests from the network driver
 *      and any UDI clients interspersed. UDI clients' requests have been stored
 *      in the from-host queue using the wire-format structures, as they arrive.
 *      All other requests are stored in the from-host queue using the host
 *      (cpu specific) structures. We use the is_packed member of the card_signal_t
 *      structure that describes the queue to make the distiction.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
process_fh_cmd_queue(card_t *card)
{
    q_t *sigq = &card->fh_command_queue;
    
    CsrInt32 processed = 0;
    CsrUint16 pending_sigs;
    CsrUint16 pending_chunks;
    CsrUint16 needed_chunks;
    CsrInt32 space_chunks;
    CsrUint16 q_index;

    /* Get the number of pending signals. */
    pending_sigs = q_slots_used(sigq);
    unifi_trace(card->ospriv, UDBG5, "proc_fh: %d pending\n", pending_sigs);
    if (pending_sigs == 0)
    {
        /* Nothing to do */
        return 0;
    }

    /* Work out how many chunks we have waiting to send */
    for (pending_chunks = 0, q_index = q_next_r_slot(sigq);
         q_index != q_next_w_slot(sigq);
         q_index = q_wrap(sigq, q_index + 1))
    {
        card_signal_t *csptr = q_slot_data(sigq, q_index);

        /*
         * Note that get_chunks_for() needs the size of the packed 
         * (wire-formatted) structure
         */
        pending_chunks += get_chunks_for(card, csptr->signal_length + 2);
    }

    /*
     * Check whether UniFi has space for all the buffered bulk-data
     * commands and signals as well.
     */
    needed_chunks = pending_chunks + card->fh_buffer.count;

    /* Round up to the block size if necessary */
    needed_chunks = round_up_needed_chunks(card, needed_chunks);

    space_chunks = check_fh_sig_slots(card, needed_chunks);
    if (space_chunks < 0) {
        /* Error */
        unifi_error(card->ospriv, "Failed to read fh sig count\n");
        return space_chunks;
    }

#ifdef CSR_WIFI_HIP_NOISY
    unifi_error(card->ospriv, "proc_fh: %d chunks free, need %d\n",
                space_chunks, needed_chunks);
#endif /* CSR_WIFI_HIP_NOISY */


    /*
     * Coalesce as many from-host signals as possible
     * into a single block and write using a single CMD53 
     */
    if (needed_chunks > (CsrUint16)space_chunks) {

        /* Round up to the block size if necessary */
        space_chunks = round_up_space_chunks(card, (CsrUint16)space_chunks);

        /*
         * If the f/w has less free chunks than those already pending
         * return immediately.
         */
        if ((CsrUint16)space_chunks <= card->fh_buffer.count) {
            /*
             * No room in UniFi for any signals after the buffered bulk
             * data commands have been sent.
             */
            unifi_error(card->ospriv, "not enough room to send signals, need %d chunks, %d free\n",
                        card->fh_buffer.count, space_chunks);
            card->generate_interrupt = 1;
            return 0;
        }
        pending_chunks = (CsrUint16)(space_chunks - card->fh_buffer.count);
    }

    while (pending_sigs-- && pending_chunks > 0)
    {
        card_signal_t *csptr;
        CsrInt16 i;
        CsrUint16 sig_chunks, total_length;
        CsrUint16 num_data_refs;
        bulk_data_param_t bulkdata;
        CsrUint8 *packed_sigptr;
        CsrUint16 signal_length = 0;

        /* Retrieve the entry at the head of the queue */
        q_index = q_next_r_slot(sigq);

        /* Get a pointer to the containing card_signal_t struct */
        csptr = q_slot_data(sigq, q_index);

        /* Get the new length of the packed signal */
        signal_length = csptr->signal_length;

        if ((signal_length & 1) || (signal_length > UNIFI_PACKED_SIGBUF_SIZE)) {
            unifi_error(card->ospriv, "process_fh_queue: Bad len: %d\n", signal_length);
            return -CSR_EIO;
        }

        /* Need space for 2-byte SDIO protocol header + signal */
        sig_chunks = get_chunks_for(card, signal_length + 2);
        if (free_chunks_in_fh_buffer(card) < sig_chunks) {
            /* No more room */
            unifi_notice(card->ospriv, "proc_fh_cmd_q: no room in fh buffer for %s, deferring\n",
                         lookup_signal_name(GET_SIGNAL_ID(csptr->sigbuf)));
            break;
        }

        /* Count the used data slots. */
        num_data_refs = 0;
        for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; i++) {
            if (csptr->bulkdata[i].data_length != 0) {
                num_data_refs++;
            }
        }

        /* Check that there are enough empty data slots. */
        if ((num_data_refs != 0) &&
             (CardGetFreeFromHostDataSlots(card) < num_data_refs))
        {
            unifi_notice(card->ospriv, "proc_fh_cmd_q: no fh data slots for %s, deferring\n",
                         lookup_signal_name(GET_SIGNAL_ID(csptr->sigbuf)));
            break;
        }

        packed_sigptr = csptr->sigbuf;

        /* 
         * Claim data slots for the bulk data. We already know slots
         * will be available.  This should be the only place where we
         * poke values into SlotNumber and DataLength.
         */
        for (i = 0; i < UNIFI_MAX_DATA_REFERENCES; ++i)
        {
            CsrUint32 datalen = csptr->bulkdata[i].data_length;

            bulkdata.d[i].data_length = datalen;
            if (datalen == 0) {
                /* Zero-out the DATAREF in the signal */
                SET_PACKED_DATAREF_SLOT(packed_sigptr, i, 0);
                SET_PACKED_DATAREF_LEN(packed_sigptr, i, 0);

                unifi_init_bulk_data(&bulkdata.d[i]);
            }
            else
            {
                /* Claim and set up a from-host data slot. */
                CsrInt32 slotnum = CardWriteBulkData(card, &csptr->bulkdata[i]);
    
                if (slotnum < 0) {
                    unifi_error(card->ospriv, "Failed to get from host data slot\n");
                    return -CSR_EIO;
                }
                if ((slotnum > 65535) || (datalen > 65535)) {
                    unifi_error(card->ospriv, "Bad data slot (%u:%u)\n", slotnum, datalen);
                    return -CSR_EIO;
                }
    
                /* Fill in the slot number in the SIGNAL structure. */
                SET_PACKED_DATAREF_SLOT(packed_sigptr, i, slotnum);
                SET_PACKED_DATAREF_LEN(packed_sigptr, i, datalen);
                bulkdata.d[i].os_data_ptr = card->from_host_data[slotnum].os_data_ptr;
                bulkdata.d[i].data_length = card->from_host_data[slotnum].data_length;

            }
        }

        unifi_trace(card->ospriv, UDBG2, "Sending signal 0x%X %s\n",
              GET_SIGNAL_ID(packed_sigptr),
              lookup_signal_name(GET_SIGNAL_ID(packed_sigptr)));
#ifdef CSR_WIFI_HIP_NOISY
        unifi_error(card->ospriv, "Sending signal 0x%X %s\n",
                    GET_SIGNAL_ID(packed_sigptr),
                    lookup_signal_name(GET_SIGNAL_ID(packed_sigptr)));
#endif /* CSR_WIFI_HIP_NOISY */

        
        /* Append packed signal to F-H buffer */
        total_length = sig_chunks * card->config_data.sig_frag_size;

        card->fh_buffer.ptr[0] = signal_length & 0xff;
        card->fh_buffer.ptr[1] =
                ((signal_length >> 8) & 0xf) | (SDIO_CMD_SIGNAL << 4);

        CsrMemCpy(card->fh_buffer.ptr + 2, packed_sigptr, signal_length);
        CsrMemSet(card->fh_buffer.ptr + 2 + signal_length, 0,
               total_length - (2 + signal_length));
        
#ifdef CSR_WIFI_HIP_NOISY
        unifi_error(card->ospriv, "proc_fh: fh_buffer %d bytes \n",
                    signal_length+2);
        dump(card->fh_buffer.ptr, signal_length+2);
        unifi_trace(card->ospriv, UDBG1, " \n");
#endif /* CSR_WIFI_HIP_NOISY */

        card->fh_buffer.ptr += total_length;
        card->fh_buffer.count += sig_chunks;

#ifdef CSR_WIFI_HIP_NOISY
        unifi_error(card->ospriv, "Added %d to fh buf, len now %d, count %d\n",
                    signal_length,
                    card->fh_buffer.ptr - card->fh_buffer.buf,
                    card->fh_buffer.count);
#endif /* CSR_WIFI_HIP_NOISY */

        processed++;
        pending_chunks -= sig_chunks;

        /* Log the signal to the UDI. */
        /* UDI will get the packed structure */
        /* Can not log the unpacked signal, unless we reconstruct it! */
        if (card->udi_hook) {
            (*card->udi_hook)(card->ospriv, packed_sigptr, signal_length,
                              &bulkdata, UDI_LOG_FROM_HOST);
        }

        /* Remove entry from q */
        csptr->signal_length = 0;
        q_inc_r(sigq);
    }

    return processed;
} /* process_fh_cmd_queue() */



/*
 * ---------------------------------------------------------------------------
 *  process_fh_traffic_queue
 *
 *      Take signals off the from-host queue and copy them to the UniFi.
 *      Does nothing if the UniFi has no slots free.
 *
 *  Arguments:
 *      card       Pointer to card context struct
 *      sigq       Pointer to the traffic queue
 *
 *  Returns:
 *      0 if there is nothing on the queue to process
 *      1 if a signal was successfully processed
 *      CSR error code if an error occurred.
 *
 *  Notes:
 *      The from-host queue contains signal requests from the network driver
 *      and any UDI clients interspersed.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
process_fh_traffic_queue(card_t *card)
{
    q_t *sigq = card->fh_traffic_queue;

    CsrInt32 processed = 0;
    CsrInt16 n, restrict_queues = 0;
    CsrInt32 q_no;
    CsrUint16 pending_sigs = 0;
    CsrUint16 pending_chunks = 0;
    CsrUint16 needed_chunks;
    CsrInt32 space_chunks;
    CsrUint16 q_index;

    /* calculate how many signals are in queues and how many chunks are needed. */
    for (n = UNIFI_WME_NO_OF_QS - 1; n >= 0; n--) {
    /* Get the number of pending signals. */
        pending_sigs += q_slots_used(&sigq[n]);
        unifi_trace(card->ospriv, UDBG5, "proc_fh%d: %d pending\n", n, pending_sigs);
    
        /* Work out how many chunks we have waiting to send */
        for (q_index = q_next_r_slot(&sigq[n]);
             q_index != q_next_w_slot(&sigq[n]);
             q_index = q_wrap(&sigq[n], q_index + 1))
        {
            card_signal_t *csptr = q_slot_data(&sigq[n], q_index);
    
            /*
             * Note that get_chunks_for() needs the size of the packed 
             * (wire-formatted) structure
             */
            pending_chunks += get_chunks_for(card, csptr->signal_length + 2);
        }
        
    }

    /* If there are no pending signals, just return */
    if (pending_sigs == 0)
    {
        /* Nothing to do */
        return 0;
    }

    /*
     * Check whether UniFi has space for all the buffered bulk-data
     * commands and signals as well.
     */
    needed_chunks = pending_chunks + card->fh_buffer.count;

    /* Round up to the block size if necessary */
    needed_chunks = round_up_needed_chunks(card, needed_chunks);

    space_chunks = check_fh_sig_slots(card, needed_chunks);
    if (space_chunks < 0) {
        /* Error */
        unifi_error(card->ospriv, "Failed to read fh sig count\n");
        return space_chunks;
    }

#ifdef CSR_WIFI_HIP_NOISY
    unifi_error(card->ospriv,
                "process_fh_traffic_queue: %d chunks free, need %d\n",
                space_chunks, needed_chunks);
    read_fhsr(card);            /* debugging only */
#endif /* CSR_WIFI_HIP_NOISY */

    /* Coalesce as many from-host signals as possible
       into a single block and write using a single CMD53 */
    if (needed_chunks > (CsrUint16)space_chunks) {

        /* Round up to the block size if necessary */
        space_chunks = round_up_space_chunks(card, (CsrUint16)space_chunks);

        if ((CsrUint16)space_chunks <= card->fh_buffer.count) {
            /*
             * No room in UniFi for any signals after the buffered bulk
             * data commands have been sent.
             */
            unifi_error(card->ospriv, "not enough room to send signals, need %d chunks, %d free\n",
                        card->fh_buffer.count, space_chunks);
            card->generate_interrupt = 1;
            return 0;
        }

        pending_chunks = (CsrUint16)space_chunks - card->fh_buffer.count;
        /* We do not have enough available chunks, we are going to restrict signals per queue */
        restrict_queues = 1;
    }

    /*
     * If there is a stopped queue, process it first.
     * However, if we are going to send all the signals,
     * there is no need to give it priority.
     */
    if (card->paused && restrict_queues) {
        q_no = card->paused_queue;
    } else {
        q_no = UNIFI_WME_NO_OF_QS - 1;
    }

    /*
     * pending_sigs will be exhausted if there are is no restriction to the pending
     * signals per queue. pending_chunks may be exhausted if there is a restriction.
     * q_no check will be exhausted if there is a restriction and our round-robin
     * algorith fails to fill all chunks.
     */
    do {
        card_signal_t *csptr;
        CsrUint16 sig_chunks, total_length;
        CsrUint16 num_data_refs;
        bulk_data_param_t bulkdata;
        CsrUint8 *packed_sigptr;
        CsrUint16 signal_length = 0;
        CsrInt32 i;

        /* if this queue is empty go to next one. */
        if (q_slots_used(&sigq[q_no]) == 0) {
            /*
             * If we have just proccessed the paused queue,
             * jump to the highest priority.
             */
            if (card->paused && restrict_queues && (q_no == card->paused_queue)) {
                q_no = UNIFI_WME_NO_OF_QS - 1;
                restrict_queues = 0;
            } else {
                q_no--;
            }
            continue;
        }

        /* Retrieve the entry at the head of the queue */
        q_index = q_next_r_slot(&sigq[q_no]);

        /* Get a pointer to the containing card_signal_t struct */
        csptr = q_slot_data(&sigq[q_no], q_index);

        /* Get the new length of the packed signal */
        signal_length = csptr->signal_length;

        if ((signal_length & 1) || (signal_length > UNIFI_PACKED_SIGBUF_SIZE)) {
            unifi_error(card->ospriv, "process_fh_traffic_queue: Bad len: %d\n", signal_length);
            return -CSR_EIO;
        }

        /* Need space for 2-byte SDIO protocol header + signal */
        sig_chunks = get_chunks_for(card, signal_length + 2);
        if (free_chunks_in_fh_buffer(card) < sig_chunks) {
            /* No more room */
            unifi_notice(card->ospriv, "process_fh_traffic_queue: no more chunks.\n");
            break;
        }

        /* Count the used data slots.
         * Note that the traffic queue has only one valid bulk data buffer.
         */
        num_data_refs = 0;
        if (csptr->bulkdata[0].data_length != 0) {
            num_data_refs++;
        }

        /*
         * Check that there are enough empty data slots.
         * We reserve 2 bulk data slots exclusively for the command queue.
         * The original check is:
         * num_data_refs > CardGetFreeFromHostDataSlots(card) - 2
         * but CardGetFreeFromHostDataSlots might return a number
         * less than 2 which means that the result may wrap around.
         */
        if ((num_data_refs != 0) &&
            (CardGetFreeFromHostDataSlots(card) < (num_data_refs + 2))) {
            break;
        }

        packed_sigptr = csptr->sigbuf;
        /* 
         * Claim data slots for the bulk data. We already know slots
         * will be available.  This should be the only place where we
         * poke values into SlotNumber and DataLength.
         * Note that the traffic queue has only one valid bulk data buffer.
         */
        {
            CsrUint32 datalen = csptr->bulkdata[0].data_length;
            
            /*
             * The bulkdata structure is used by the udi_hook callback
             * to log the signal to the logging registered clients.
             * The compiler may not initialise the structure to zero.
             * Since this is a data signal and we use only the first bulk
             * data ref, we set the length of the rest bulk data refs to zero.
             */
            for (i = 1; i < UNIFI_MAX_DATA_REFERENCES; i++)
            {
                unifi_init_bulk_data(&bulkdata.d[i]);
            }
    
            bulkdata.d[0].data_length = datalen;
            if (datalen == 0) {
                /* Zero-out the DATAREF in the signal */
                SET_PACKED_DATAREF_SLOT(packed_sigptr, 0, 0);
                SET_PACKED_DATAREF_LEN(packed_sigptr, 0, 0);

                unifi_init_bulk_data(&bulkdata.d[0]);
            }
            else
            {
                /* Claim and set up a from-host data slot. */
                CsrInt32 slotnum = CardWriteBulkData(card, &csptr->bulkdata[0]);
    
                if (slotnum < 0) {
                    unifi_error(card->ospriv, "Failed to get from host data slot\n");
                    return -CSR_EIO;
                }
                if ((slotnum > 65535) || (datalen > 65535)) {
                    unifi_error(card->ospriv, "Bad data slot (%u:%u)\n", slotnum, datalen);
                    return -CSR_EIO;
                }
    
                /* Fill in the slot number in the SIGNAL structure. */
                SET_PACKED_DATAREF_SLOT(packed_sigptr, 0, slotnum);
                SET_PACKED_DATAREF_LEN(packed_sigptr, 0, datalen);
                bulkdata.d[0].os_data_ptr = card->from_host_data[slotnum].os_data_ptr;
                bulkdata.d[0].data_length = card->from_host_data[slotnum].data_length;
                bulkdata.d[0].os_net_buf_ptr = card->from_host_data[slotnum].os_net_buf_ptr;
                bulkdata.d[0].net_buf_length = card->from_host_data[slotnum].net_buf_length;

            }
        }

#ifdef CSR_WIFI_HIP_DATA_PLANE_PROFILE
        card->cmd_prof.tx_count ++;
#endif
        unifi_trace(card->ospriv, UDBG3, "Sending signal 0x%X %s\n",
              GET_SIGNAL_ID(packed_sigptr),
              lookup_signal_name(GET_SIGNAL_ID(packed_sigptr)));
#ifdef CSR_WIFI_HIP_NOISY
        unifi_error(card->ospriv, "Sending signal 0x%X %s\n",
                    GET_SIGNAL_ID(packed_sigptr),
                    lookup_signal_name(GET_SIGNAL_ID(packed_sigptr)));
#endif /* CSR_WIFI_HIP_NOISY */

        /* Append packed signal to F-H buffer */
        total_length = sig_chunks * card->config_data.sig_frag_size;

        card->fh_buffer.ptr[0] = signal_length & 0xff;
        card->fh_buffer.ptr[1] =
                ((signal_length >> 8) & 0xf) | (SDIO_CMD_SIGNAL << 4);

        CsrMemCpy(card->fh_buffer.ptr + 2, packed_sigptr, signal_length);
        CsrMemSet(card->fh_buffer.ptr + 2 + signal_length, 0,
               total_length - (2 + signal_length));
        
#ifdef CSR_WIFI_HIP_NOISY
        unifi_error(card->ospriv, "proc_fh: fh_buffer %d bytes \n",
                    signal_length+2);
        dump(card->fh_buffer.ptr, signal_length+2);
        unifi_trace(card->ospriv, UDBG1, " \n");
#endif /* CSR_WIFI_HIP_NOISY */

        card->fh_buffer.ptr += total_length;
        card->fh_buffer.count += sig_chunks;

#ifdef CSR_WIFI_HIP_NOISY
        unifi_error(card->ospriv, "Added %d to fh buf, len now %d, count %d\n",
                    signal_length,
                    card->fh_buffer.ptr - card->fh_buffer.buf,
                    card->fh_buffer.count);
#endif /* CSR_WIFI_HIP_NOISY */

        processed++;
        pending_sigs--;
        pending_chunks -= sig_chunks;

        /* Log the signal to the UDI. */
        /* UDI will get the packed structure */
        /* Can not log the unpacked signal, unless we reconstruct it! */
        if (card->udi_hook) {
            (*card->udi_hook)(card->ospriv, packed_sigptr, signal_length,
                              &bulkdata, UDI_LOG_FROM_HOST);
        }

        /* Remove entry from q */
        csptr->signal_length = 0;
        /* Note that the traffic queue has only one valid bulk data buffer. */
        csptr->bulkdata[0].data_length = 0;

        q_inc_r(&sigq[q_no]);
    } while ((pending_sigs > 0) && (pending_chunks > 0) && (q_no >= 0));

    return processed;
} /* process_fh_traffic_queue() */


/*
 * ---------------------------------------------------------------------------
 *  flush_fh_buffer
 *
 *      Write out the cache from-hosts signals to the UniFi.
 * 
 *  Arguments:
 *      card       Pointer to card context struct
 *
 *  Returns:
 *      0 if there is nothing in the buffer to process
 *      CSR error code if an SDIO error occurred.
 * ---------------------------------------------------------------------------
 */
static CsrInt32
flush_fh_buffer(card_t *card)
{
    CsrInt32 r;
    CsrUint16 len;
    CsrUint16 sig_units;
    CsrUint16 data_round;
    CsrUint16 chunks_in_last_block;
    CsrUint16 padding_chunks;
    CsrUint16 i;

    len = card->fh_buffer.ptr - card->fh_buffer.buf;

#ifdef CSR_WIFI_HIP_NOISY
    unifi_error(card->ospriv, "fh_buffer is at %p, ptr= %p\n",
                card->fh_buffer.buf, card->fh_buffer.ptr);
#endif /* CSR_WIFI_HIP_NOISY */

    if (len == 0) {
        return 0;
    }

#ifdef CSR_WIFI_HIP_NOISY
    if (dump_fh_buf) {
        dump(card->fh_buffer.buf, len);
        dump_fh_buf = 0;
    }
#endif /* CSR_WIFI_HIP_NOISY */

    if (card->sdio_io_block_pad) {
        /* Both of these are powers of 2 */
        sig_units = card->config_data.sig_frag_size;
        data_round = card->sdio_io_block_size;

        if (data_round > sig_units) {
            chunks_in_last_block = (len % data_round) / sig_units;

            if (chunks_in_last_block != 0) {
                padding_chunks = (data_round / sig_units) - chunks_in_last_block;

                CsrMemSet(card->fh_buffer.ptr, 0, padding_chunks * sig_units);
                for (i = 0; i < padding_chunks; i++) {
                    card->fh_buffer.ptr[1] = SDIO_CMD_PADDING << 4;
                    card->fh_buffer.ptr += sig_units;
                }

                card->fh_buffer.count += padding_chunks;
                len += padding_chunks * sig_units;
            }
        }
    }

    r = unifi_bulk_rw(card,
                      card->config_data.fromhost_sigbuf_handle,
                      card->fh_buffer.buf,
                      len, UNIFI_SDIO_WRITE);
    if (r == -CSR_ENODEV) return r;
    if (r < 0) {
        unifi_error(card->ospriv, "Failed to write fh signals: %u bytes, error %d\n", len, r);
        return r;
    }

    /* Update from-host-signals-written signal count */
    card->from_host_signals_w =
        (card->from_host_signals_w + card->fh_buffer.count) % 128u;
    r = unifi_write_8_or_16(card, card->sdio_ctrl_addr+0,
                            (CsrUint8)card->from_host_signals_w);
    if (r < 0) {
        unifi_error(card->ospriv, "Failed to write fh signal count %u with error %d\n",
                    card->from_host_signals_w, r);
        return r;
    }
    card->generate_interrupt = 1;

    /* Reset the fh buffer pointer */
    card->fh_buffer.ptr = card->fh_buffer.buf;
    card->fh_buffer.count = 0;

#ifdef CSR_WIFI_HIP_NOISY
    unifi_error(card->ospriv, "END flush: fh len %d, count %d\n",
                card->fh_buffer.ptr - card->fh_buffer.buf,
                card->fh_buffer.count);
#endif /* CSR_WIFI_HIP_NOISY */

    return 0;
} /* flush_fh_buffer() */


/*
 * ---------------------------------------------------------------------------
 *  restart_packet_flow
 *
 *      This function is called before the bottom-half thread sleeps.
 *      It checks whether both data and signal resources are available and
 *      then calls the OS-layer function to re-enable packet transmission.
 *
 *  Arguments:
 *      card       Pointer to card context struct
 *
 *  Returns:
 *      None.
 * ---------------------------------------------------------------------------
 */
static void
restart_packet_flow(card_t *card)
{
    /* 
     * We only look at the fh_traffic_queue, because that is where packets from
     * the network stack are placed.
     */
    if (card->paused &&
        (q_slots_free(&card->fh_traffic_queue[0]) >= RESUME_XMIT_THRESHOLD) &&
        (q_slots_free(&card->fh_traffic_queue[1]) >= RESUME_XMIT_THRESHOLD) &&
        (q_slots_free(&card->fh_traffic_queue[2]) >= RESUME_XMIT_THRESHOLD) &&
        (q_slots_free(&card->fh_traffic_queue[3]) >= RESUME_XMIT_THRESHOLD)
       )
    {
        /*
         * Clear the paused flag *before* calling
         * unifi_restart_xmit().  The calling sequence down through
         * unifi_restart_xmit() could possibly call all the way to
         * unifi_send_signal() and unifi_pause_xmit().  When that
         * happens, paused will have been set again, so we don't then
         * want to clear it.
         */
        card->paused = 0;
        card->paused_queue = 0;
        unifi_restart_xmit(card->ospriv);
    }
} /* restart_packet_flow() */
