/***********************************************************************

SlapEnable.c

Purpose: 	Enables a Palm application to be slap compatible, receiving
			Slap events.
			
How to use:
			Slap will send your application a block of text when the user
			selects your application from the popup application list. It
			is up to you to interpret the text somehow and add it to your
			application, and then go to the data as if you just received
			a goto command from the system.
			
			To begin receiving the Slap command, call SlapInit in your
			sysAppLaunchCmdSyncNotify AND sysAppLaunchCmdSystemReset launchcode
			handlers. This will notify Slap to begin sending you commands.
			
			Then, create a launch code handler for the SlapLaunchCode. This 
			handler will receive the block of text from Slap, process it,
			and then go to it. The SlapLaunchCode is similar to a
			launchGoTo, in that it will start your application and give you
			your own globals.
			
			To extract the Slap text from the SlapLaunchCode launch parameters,
			call SlapGetText. Once you are done with the text from Slap, call
			SlapReleaseParams.
			
			Remeber that after you have interpreted the data, you should
			immediately go to it and show it on the screen. You will have
			globals, so you can probably just call whatever routine you normally
			call to handle a launchGoTo.

Copyright  2002 Hands High Software
All Rights Reserved
************************************************************************/

#include "PalmOS.h"
#include "SlapEnable.h"

/***********************************************************************

SlapInit

Purpose: 		Initialize slap. This basically makes sure feature memory
				is set up that contains the creator id of your program. This
				will tell slap to send you commands. You should call
				this from the sysAppLaunchCmdSyncNotify AND sysAppLaunchCmdSystemReset
				so that your application will be setup right after it is installed,
				or right after a system reset.

Preconditions: 	

Input:			appCreator		Your application's creator code.

Output:			none

Returns:		none

************************************************************************/

void SlapInit(UInt32 appCreator)
{
	DWord	romVersion;
	Err		err;
	MemPtr	p;
	UInt32  *pCreators;
	UInt16	count;
	UInt16	i;
	Boolean found = false;
	
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);

	if (romVersion < (sysMakeROMVersion(3,1,0,0,0))) {
		return; // no feature ptr stuff available
	}
	
	err = FtrGet (SlapCreator, SlapFeatureNum, (UInt32*)&pCreators);

	if (err) { // no feature at all!
		err = FtrPtrNew(SlapCreator, SlapFeatureNum, sizeof (appCreator), &p);
		if (!err) {
			DmWrite (p, 0, &appCreator, sizeof (appCreator));
		}
	} else {	// found a feature, now search it
		
		count = MemPtrSize (pCreators) / 4;
		
		for (i = 0; i < count; i++) {
			if (pCreators[i] == appCreator) {
				found = true;
				break;
			}
		}
		
		if (!found) {
			// add the creator to the end of the feature memory
			err = FtrPtrResize (SlapCreator, SlapFeatureNum, (count + 1) * sizeof (appCreator), &p);
			if (!err) {
				DmWrite (p, count * sizeof (appCreator), &appCreator, sizeof (appCreator));
			}
		}
	}
} /* SlapInit */

/***********************************************************************

MemHandle SlapGetText

Purpose: 		Get the text of the slap data. This will be coming from
				the slap database. It should be treated as read only.
				
				Be sure to call SlapReleaseParams when you are done.

Preconditions: 	

Input:			pParams		Slap launch params

Output:			none

Returns:		pointer to character string, or null if an error occurs.

************************************************************************/

char * SlapGetText(SlapLaunchParamsPtr pParams)
{
	MemHandle h;
	MemPtr p = NULL;
	
	pParams->dbRef = DmOpenDatabase (pParams->slapDBCard, pParams->slapDBLocalID, dmModeReadOnly);
	if (pParams->dbRef) {
		h = DmQueryRecord (pParams->dbRef, pParams->slapRecord);
		if (h) {
			p = MemHandleLock (h);
		}
	}
	
	return (char*)p;
} /* MemHandle SlapGetText */


/***********************************************************************

SlapReleaseParams

Purpose: 		Call this after you are done using the text.

Preconditions: 	

Input:			none

Output:			none

Returns:		none

************************************************************************/

void SlapReleaseParams(SlapLaunchParamsPtr pParams)
{
	MemHandle h;
	
	if (pParams) {
		if (pParams->dbRef) {
			h = DmQueryRecord (pParams->dbRef, pParams->slapRecord);
			if (h) {
				MemHandleUnlock (h);
			}
		}
		DmCloseDatabase (pParams->dbRef);
	}
} /* SlapReleaseParams */
