package com.ibm.ive.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2005  All Rights Reserved
 */

/**
 * A simple blocking event queue.
 */
public class BlockingQueue {

	private FastVector fVector;
	private boolean fIsClosed;

	/**
	 * Create a new queue.
	 */
	public BlockingQueue() {
		super();
		fVector = new FastVector();
		fIsClosed = false;
	}

	/**
	 * Add an object to the queue.
	 */
	public void postObject(Object object) {
		synchronized(this) {
			if (fIsClosed) return;
			fVector.addElement(object);
			notifyAll();
		}
	}

	/**
	 * Return the next object in the queue.
	 */
	public Object getNextObject() {

		synchronized(this) {
			if (fIsClosed) return null;
			if (fVector.size() == 0) {
				try {
					wait();
				} catch (InterruptedException e) {
				}
			}

			if (fIsClosed) return null;
			if (fVector.size() == 0) return null;

			Object result = fVector.elementAt(0);
			fVector.removeElementAt(0);
			return result;
		}
	}

	/**
	 * Return the next object in the queue.
	 * Waits specified time, or till till the queue is closed.
	 */
	public Object getNextObjectTimeout(int timeout) {
		synchronized(this) {
			if (fIsClosed) return null;
			if (fVector.size() == 0) {
				try {
					wait(timeout);
				} catch (InterruptedException e) {
				}
			}

			if (fIsClosed) return null;
			if (fVector.size() == 0) return null;

			Object result = fVector.elementAt(0);
			fVector.removeElementAt(0);
			return result;
		}
	}

	/**
	 * Return the next object in the queue.
	 */
	public Object getNextObjectNoWait() {
		synchronized(this) {
			if (fIsClosed) return null;
			if (fVector.size() == 0) return null;
			Object result = fVector.elementAt(0);
			fVector.removeElementAt(0);

			return result;
		}
	}

	/**
	 * Returns true if the queue is closed.
	 */
	public boolean isClosed() {
		synchronized(this) {
			return fIsClosed;
		}
	}

	/**
	 * Close the queue.
	 * All future postObject() calls are no-ops.
	 * All future getNextObject() calls return null.
	 */
	public void close() {
		synchronized(this) {
			fIsClosed = true;
			notifyAll();
		}
	}
}
