package com.ibm.ive.midp;

import java.util.*;

/*
 * Licensed Materials - Property of IBM.
 * (c) Copyright IBM Corp 2000, 2005.
 */

public class FastVector {

	private static final int DEFAULT_START_SIZE = 4;
	private static final int REDUCTION_THRESHOLD = DEFAULT_START_SIZE * 8;
	private static final boolean PERFORM_ERROR_CHECKING = false;

	private Object[] elements;
	private int size = 0;

	public FastVector() {
		this(DEFAULT_START_SIZE);
	}

	public FastVector(int startSize) {
		elements = new Object[startSize];
	}

	public boolean contains(Object o) {
		return indexOf(o) != -1;
	}

	public int size() {
		return size;
	}

	public Object elementAt(int pos) {
		if (PERFORM_ERROR_CHECKING) {
			if (pos < 0 || pos >= size) throw new ArrayIndexOutOfBoundsException();
		}
		return elements[pos];
	}

	public void insertElementAt(Object o, int pos) {
		if (PERFORM_ERROR_CHECKING) {
			if (pos < 0 || pos > size) throw new ArrayIndexOutOfBoundsException();
		}
		doubleSizeIfNecessary();
		/* if we're not inserting at the end, make room for the new element */
		if (pos != size) System.arraycopy(elements, pos, elements, pos+1, size-pos);
		size++;
		elements[pos] = o;
	}

	public void setElementAt(Object o, int pos) {
		if(PERFORM_ERROR_CHECKING) {
			if(pos < 0 || pos >= size) throw new ArrayIndexOutOfBoundsException();
		}
		elements[pos] = o;
	}

	public void addElement(Object o) {
		doubleSizeIfNecessary();
		elements[size++] = o;
	}

	public FastVector clone() {
		FastVector copy = new FastVector(size);
		System.arraycopy(elements, 0, copy.elements, 0, size);
		copy.size = size;
		return copy;
	}

	public int indexOf(Object o) {
		if (o != null) {
			for (int i = 0; i < size; i++)
				if (o.equals(elements[i])) return i;
		} else {
			for (int i = 0; i < size; i++)
				if (elements[i] == null) return i;
		}
		return -1;
	}

	public void removeElementAt(int pos) {
		if (PERFORM_ERROR_CHECKING) {
			if (pos < 0 || pos >= size) throw new ArrayIndexOutOfBoundsException();
		}
		/* if we're not removing the last element, shift the rest of the elements back one */
		if (pos != size-1) System.arraycopy(elements, pos+1, elements, pos, size-pos-1);
		elements[--size] = null;
		reduceSizeIfNecessary();
	}

	public boolean removeElement(Object o) {
		int index;
		if ((index = indexOf(o)) == -1) return false;
		removeElementAt(index);
		return true;
	}

	public void setSize(int newSize) {
		if (PERFORM_ERROR_CHECKING) {
			if (newSize < 0) throw new ArrayIndexOutOfBoundsException();
		}
		if (size > newSize) {
			for (int i = newSize; i < size; i++) elements[i] = null;
		} else if (size < newSize && elements.length < newSize) {
			Object[] newArray = new Object[newSize];
			System.arraycopy(elements, 0, newArray, 0, size);
			elements = newArray;
		}
		size = newSize;
	}

	public void addAll(FastVector other) {
		int oldSize = size;
		setSize(size + other.size);
		for (int i = 0; i < other.size; i++) {
			elements[oldSize++] = other.elements[i];
		}
	}

	private void doubleSizeIfNecessary() {
		if (elements.length == 0) {
			elements = new Object[DEFAULT_START_SIZE];
		} else if (size == elements.length) {
			Object[] newArray = new Object[size << 1];
			System.arraycopy(elements, 0, newArray, 0, size);
			elements = newArray;
		}
	}

	private void reduceSizeIfNecessary() {
		if (size < (elements.length / REDUCTION_THRESHOLD)) {
			/* create an array twice the size of what is currently needed */
			Object[] newArray = new Object[size << 1];
			System.arraycopy(elements, 0, newArray, 0, size);
			elements = newArray;
		}
	}

	public void copyInto(Object[] objArray) {
		System.arraycopy(elements, 0, objArray, 0, size);
	}

	public Object lastElement() {
		if (size == 0) return null;
		return elements[size-1];
	}

	public void removeAllElements() {
		for (; size > 0; size--) {
			elements[size-1] = null;
		}

		if (elements.length > REDUCTION_THRESHOLD) elements = new Object[DEFAULT_START_SIZE];
	}

	public boolean equals(FastVector v) {
		if (v == null || v.size != size) return false;
		for (int i = 0; i < size; i++) {
			if (v.elements[i] == null) {
				if (elements[i] == null) continue;

				return false;
			}
			if (!v.elements[i].equals(elements[i])) return false;
		}
		return true;
	}

	public Enumeration elements() {
		return new Enumeration() {
			int pos = 0;

			public boolean hasMoreElements() {
				return pos < size;
			}

			public Object nextElement() {
				if (pos < size) return elements[pos++];
				throw new NoSuchElementException();
			}
		};
	}
}
