package com.ibm.ive.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2005  All Rights Reserved
 */

import java.io.*;
import java.util.*;

import com.ibm.oti.vm.*;

/**
 * This class retrieves strings from a resource bundle
 * and returns them, formatting them with MessageFormat
 * when required.
 * <p>
 * It is used by the system classes to provide national
 * language support, by looking up messages in the
 * <code>
 *    com.ibm.oti.util.ExternalMessages
 * </code>
 * resource bundle. Note that if this file is not available,
 * or an invalid key is looked up, or resource bundle support
 * is not available, the key itself will be returned as the
 * associated message. This means that the <em>KEY</em> should
 * a reasonable human-readable (english) string.
 *
 * @author		OTI
 * @version		initial
 */

public class PermissionMsg {

	// Properties holding the system messages.
	static private Hashtable messages = null;

	static {
	  // Attempt to load the messages.
	  try {
		messages = loadMessages("com/ibm/ive/midp/PermissionMessages"); //$NON-NLS-1$
	  } catch (IOException e) {}
	}

/**
 * Retrieves a message which has no arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		msg String
 *					the key to look up.
 * @return		String
 *					the message for that key in the system
 *					message bundle.
 */
static public String getString (String msg) {
	if (messages == null) return msg;
	String resource = (String) messages.get(msg);
	if (resource == null) return msg;
	return resource;
}

/**
 * Retrieves a message which takes 1 argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		msg String
 *					the key to look up.
 * @param		arg Object
 *					the object to insert in the formatted output.
 * @return		String
 *					the message for that key in the system
 *					message bundle.
 */
static public String getString (String msg, Object arg) {
	return getString(msg, new Object[] {arg});
}

/**
 * Retrieves a message which takes 1 integer argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		msg String
 *					the key to look up.
 * @param		arg int
 *					the integer to insert in the formatted output.
 * @return		String
 *					the message for that key in the system
 *					message bundle.
 */
static public String getString (String msg, int arg) {
	return getString(msg, new Object[] {Integer.toString(arg)});
}

/**
 * Retrieves a message which takes 1 character argument.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		msg String
 *					the key to look up.
 * @param		arg char
 *					the character to insert in the formatted output.
 * @return		String
 *					the message for that key in the system
 *					message bundle.
 */
static public String getString (String msg, char arg) {
	return getString(msg, new Object[] {String.valueOf(arg)});
}

/**
 * Retrieves a message which takes 2 arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		msg String
 *					the key to look up.
 * @param		arg1 Object
 *					an object to insert in the formatted output.
 * @param		arg2 Object
 *					another object to insert in the formatted output.
  * @return		String
 *					the message for that key in the system
 *					message bundle.
 */
static public String getString (String msg, Object arg1, Object arg2) {
	return getString(msg, new Object[] {arg1, arg2});
}

/**
 * Retrieves a message which takes several arguments.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		msg String
 *					the key to look up.
 * @param		args Object[]
 *					the objects to insert in the formatted output.
 * @return		String
 *					the message for that key in the system
 *					message bundle.
 */
static public String getString (String msg, Object[] args) {
	String format = msg;

	if (messages != null) {
		format = (String)messages.get(msg);
		if (format==null) format = msg;
	}

	return format(format, args);
}

/**
 * Retrieves indirectly a message which takes one or more arguments, by looking
 * up a message and using that as a key into the actual message to retrieve.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		msg String
 *					the key to look up in a first message. The retrieved message
 * 				becomes the key into actual message.
 * @param		args Object[]
 *					the objects to insert in the formatted output.
 * @return		String
 *					the message for that key in the system
 *					message bundle.
 */
static public String getIndirectString(String msg, Object[] args) {
	if (messages==null)
		return msg;
	String key = (String)messages.get(msg);
	if (key==null)
		return msg;
	return getString(key, args);
}

/**
 * Generates a formatted text string given a source string
 * containing "argument markers" of the form "{argNum}"
 * where each argNum must be in the range 0..9. The result
 * is generated by inserting the toString of each argument
 * into the position indicated in the string.
 * <p>
 * To insert the "{" character into the output, use a single
 * backslash character to escape it (i.e. "\{"). The "}"
 * character does not need to be escaped.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		format String
 *					the format to use when printing.
 * @param		args Object[]
 *					the arguments to use.
 * @return		String
 *					the formatted message.
 */
static private String format (String format, Object[] args) {
	StringBuffer answer = new StringBuffer();
	String[] argStrings = new String[args.length];
	for (int i = 0; i < args.length; ++i) {
		if (args[i] == null)
			argStrings[i] = "<null>"; //$NON-NLS-1$
		else
			argStrings[i] = args[i].toString();
	}
	int lastI = 0;
	for (int i = format.indexOf('{', 0); i >= 0; i = format.indexOf('{', lastI)) {
		if (i != 0 && format.charAt(i-1) == '\\') {
			// It's escaped, just print and loop.
			if (i != 1)
				answer.append(format.substring(lastI,i-1));
			answer.append('{');
			lastI = i+1;
		} else {
			// It's a format character.
			if (i > format.length()-3) {
				// Bad format, just print and loop.
				answer.append(format.substring(lastI, format.length()));
				lastI = format.length();
			} else {
				int argnum = (byte) Character.digit(format.charAt(i+1), 10);
				if (argnum < 0 || format.charAt(i+2) != '}') {
					// Bad format, just print and loop.
					answer.append(format.substring(lastI, i+1));
					lastI = i+1;
				} else {
					// Got a good one!
					answer.append(format.substring(lastI, i));
					if (argnum >= argStrings.length)
						answer.append("<missing argument>"); //$NON-NLS-1$
					else
						answer.append(argStrings[argnum]);
					lastI = i + 3;
				}
			}
		}
	}
	if (lastI < format.length())
		answer.append(format.substring(lastI, format.length()));
	return answer.toString();
}

// Loads properties from the specified resource. The properties are of
// the form <code>key=value</code>, one property per line.
static private Hashtable loadMessages(String resourceName) throws IOException {
	String resName;
	InputStream resourceStream;

	String language = System.getProperty("user.language"); //$NON-NLS-1$
	if (language == null) language = "en"; //$NON-NLS-1$
	String region = System.getProperty("user.region"); //$NON-NLS-1$
	if (region == null) region = "US"; //$NON-NLS-1$
	String variant = System.getProperty("user.variant"); //$NON-NLS-1$

	if (variant != null) {
		resName = resourceName + "_" + language + "_" + region + "_" + variant + ".properties"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
		resourceStream = VM.getResourceAsStream(resName,false);
		if (resourceStream != null) return load(resourceStream);
	}

	resName = resourceName + "_" + language + "_" + region + ".properties"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
	resourceStream = VM.getResourceAsStream(resName,false);
	if (resourceStream != null) return load(resourceStream);

	resName = resourceName + "_" + language + ".properties"; //$NON-NLS-1$ //$NON-NLS-2$
	resourceStream = VM.getResourceAsStream(resName,false);
	if (resourceStream != null) return load(resourceStream);

	return load(VM.getResourceAsStream(resourceName + ".properties",false)); //$NON-NLS-1$
}

static private Hashtable load(InputStream resourceStream) throws IOException {
	final int
		NONE = 0, SLASH = 1, UNICODE = 2,
		CONTINUE = 3, DONE = 4, IGNORE = 5;

	Hashtable messages = new Hashtable();
	int mode = NONE, unicode = 0, count = 0, nextChar;
	char[] buf = new char[40];
	int offset = 0, keyLength = -1;
	boolean firstChar = true;

	/*
	 * Avoid doing expensive single byte read operations by
	 * reading the entire contents in large chunks.
	 */
	InputStreamReader reader = new InputStreamReader(resourceStream);
	final int initialSize = 5000;
	char[] buffer = new char[initialSize];
	StringBuffer stringBuffer = new StringBuffer(initialSize);
	int readerCount = 1;
	while(readerCount > 0){
		readerCount = reader.read(buffer);
		if(readerCount > 0)
			stringBuffer.append(new String(buffer, 0, readerCount));
	}
	buffer = stringBuffer.toString().toCharArray();
	int totalCount = buffer.length;
	int indexCount = 0;

//	while ((nextChar = resourceStream.read()) != -1) {
	while (indexCount < totalCount) {
		nextChar = buffer[indexCount++];

		if (offset == buf.length) {
			char[] newBuf = new char[buf.length * 2];
			System.arraycopy(buf, 0, newBuf, 0, offset);
			buf = newBuf;
		}
		if (mode == UNICODE) {
			int digit = Character.digit((char)nextChar, 16);
			if (digit >= 0) {
				unicode = (unicode << 4) + digit;
				if (++count < 4) continue;
			}
			mode = NONE;
			buf[offset++] = (char)unicode;
			if (nextChar != '\n') continue;
		}
		if (mode == SLASH) {
			mode = NONE;
			switch (nextChar) {
				case '\r':
					mode = CONTINUE; // Look for a following \n
					continue;
				case '\n':
					mode = IGNORE; // Ignore whitespace on the next line
					continue;
				case 'b':
					nextChar = '\b';
					break;
				case 'f':
					nextChar = '\f';
					break;
				case 'n':
					nextChar = '\n';
					break;
				case 'r':
					nextChar = '\r';
					break;
				case 't':
					nextChar = '\t';
					break;
				case 'u':
					mode = UNICODE;
					unicode = count = 0;
					continue;
			}
		} else {
			switch (nextChar) {
				case '#':
				case '!':
					if (firstChar) {
//						while ((nextChar = resourceStream.read()) != -1)
						while (indexCount < totalCount) {
							nextChar = buffer[indexCount++];
							if (nextChar == '\r' || nextChar == '\n') break;
						}
						continue;
					}
					break;
				case '\n':
					if (mode == CONTINUE) { // Part of a \r\n sequence
						mode = IGNORE; // Ignore whitespace on the next line
						continue;
					}
					// fall into the next case
				case '\r':
					mode = NONE;
					firstChar = true;
					if (keyLength >= 0) {
						String temp = new String(buf, 0, offset);
						messages.put(temp.substring(0,keyLength), temp.substring(keyLength));
					}
					keyLength = -1;
					offset = 0;
					continue;
				case '\\':
					mode = SLASH;
					continue;
				case ':':
				case '=':
					if (keyLength == -1) { // if parsing the key
						keyLength = offset;
						continue;
					}
					break;
			}
			if (isWhitespace((char)nextChar)) {
				if (mode == CONTINUE) mode = IGNORE;
				// if key length == 0 or value length == 0
				if (offset == 0 || offset == keyLength || mode == IGNORE) continue;
				if (keyLength == -1) { // if parsing the key
					mode = DONE;
					continue;
				}
			}
			if (mode == IGNORE || mode == CONTINUE) mode = NONE;
		}
		firstChar = false;
		if (mode == DONE) {
			keyLength = offset;
			mode = NONE;
		}
		buf[offset++] = (char)nextChar;
	}
	if (keyLength >= 0) {
		String temp = new String(buf, 0, offset);
		messages.put(temp.substring(0,keyLength), temp.substring(keyLength));
	}
	resourceStream.close();
	return messages;
}

public static boolean isWhitespace(char c) {
	// Optimized case for ASCII
	if ((c >= 0x1c && c <= 0x20) || (c >= 0x9 && c <= 0xd)) return true;
	if (c < 0x2000) return false;
	return c <= 0x200b || c == 0x2028 || c == 0x2029 || c == 0x3000;
}

}
