package com.ibm.ive.midp;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import javax.microedition.io.*;
import javax.microedition.lcdui.*;
import javax.microedition.midlet.*;

import com.ibm.ive.midp.ams.*;
import com.ibm.ive.midp.ams.ui.*;
import com.ibm.oti.midlet.help.MidletLoader;
import com.ibm.ive.midp.util.MIDletManager;

/*
 * Handles calls to MIDlet.platformRequest() based on the Device and it's
 * settings.
 */
public abstract class PlatformRequest implements Runnable {

	private static final int TYPE_INVALID = -1;
	private static final int TYPE_DIAL = 0;
	private static final int TYPE_JAR = 1;
	private static final int TYPE_JAD = 2;
	private static final int TYPE_HTTP_URL = 3;
	private static final int TYPE_MAILTO = 4;
	private static final int TYPE_UNKNOWN = 5;

	public static Object gInstallLock = new Object();

	private static FastVector gPending;
	private boolean fMustExit;
	private boolean fCancelled;

	protected String fUrl;

	PlatformRequest(String url, boolean mustExit) throws Exception {
		super();
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
		fMustExit = mustExit;
		fCancelled = false;
		fUrl = url;
	}

	/*
	 * Cancel any pending requests and remove them from gPending.
	 */
	public static void cancelPendingRequests() {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
		if (gPending == null || gPending.size() == 0) return;

		synchronized (gPending) {
			int size = gPending.size();
			for (int i = 0; i < size; i++) {
				((PlatformRequest) gPending.elementAt(0)).cancel();
				gPending.removeElementAt(0);
			}
		}
	}

	/*
	 * Create a new PlatformRequest. If url is invalid, throw a
	 * ConnectionNotFoundException. If url is valid, add the new PlatformRequest
	 * to gPending. Return true if the MIDlet suite must exit first before this
	 * request can be satisfied.
	 */
	public static boolean newRequest(MIDlet midlet, String url) throws ConnectionNotFoundException {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();

		int type = getType(url);
		if (type == TYPE_INVALID) throw new ConnectionNotFoundException();
		if (gPending == null) gPending = new FastVector();

		try {
			PlatformRequest request = null;

			switch (type) {
				case TYPE_DIAL:
					// First check whether the device can make phone calls
					if (!Device.isPhone()) throw new ConnectionNotFoundException();
					request = new DialRequest(url);
					break;
				case TYPE_JAR:
					request = new AmsInstallRequest(midlet, url);
					break;
				case TYPE_JAD:
					request = new AmsInstallRequest(midlet, url);
					break;
				case TYPE_HTTP_URL:
					// First check whether the device can open platform browser
					if (!Device.canUsePlatformBrowser()) throw new ConnectionNotFoundException();
					request = new HttpRequest(url);
					break;
				case TYPE_MAILTO:
					// First check whether the device can open platform mail client
					if (!Device.canUsePlatformMailClient()) throw new ConnectionNotFoundException();
					request = new MailToRequest(url);
					break;
				case TYPE_UNKNOWN: {
					// determine handler of specialized platform requests
					String className = midlet.getAppProperty("PlatformRequestHandler"); //$NON-NLS-1$
					// TODO: the following line should be removed, but I don't
					// know where the property is set
					className = "com.ibm.ive.midp.ext.PalmPlatformRequestHandler"; //$NON-NLS-1$
					IPlatformRequestHandler handler = null;
					if (className != null) {
						// get the class
						try {
							Class handlerClass = Class.forName(className.trim());
							handler = (IPlatformRequestHandler) handlerClass.newInstance();
						} catch (ClassNotFoundException e) {
							e.printStackTrace();
						} catch (IllegalAccessException e) {
							e.printStackTrace();
						} catch (InstantiationException e) {
							e.printStackTrace();
						}
					}
					if (handler == null || !handler.handlesURL(url)) {
						// the user either doesn't have a handler, or the
						// handler doesn't support this url
						throw new ConnectionNotFoundException();
					}
					request = new UserRequest(url, handler);
					break;
				}
			}

			synchronized (gPending) {
				gPending.addElement(request);
			}

			Display.getDisplay(midlet).callSerially(request);
			return request.fMustExit;
		} catch (Exception e) {
			throw new ConnectionNotFoundException();
		}
	}

	/*
	 * Return true if the platform can process this request.
	 */
	public static int getType(String url) {
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();
		if (url == null) return TYPE_INVALID;

		String lcURL = url.toLowerCase();

		if (lcURL.endsWith(".jad")) { //$NON-NLS-1$
			return TYPE_JAD;
		} else if (lcURL.endsWith(".jar")) { //$NON-NLS-1$
			return TYPE_JAR;
		} else if (lcURL.startsWith("tel:")) { //$NON-NLS-1$
			return TYPE_DIAL;
		} else if (lcURL.startsWith("http://")) { //$NON-NLS-1$
			return TYPE_HTTP_URL;
		} else if (lcURL.startsWith("mailto:")) { //$NON-NLS-1$
			return TYPE_MAILTO;
		}
		return TYPE_UNKNOWN;
	}

	public void run() {
		synchronized (this) {
			if (!fCancelled) {
				performRequest();
				// if (!performRequest()) System.out.println("request failed");
			}
		}

		synchronized (gPending) {
			if (gPending.contains(this)) {
				gPending.removeElement(this);
			}
		}
	}

	synchronized void cancel() {
		synchronized (this) {
			fCancelled = true;
		}
	}

	protected abstract boolean performRequest();
	static native void callHiddenPauseSetCurrent(Display display);
	static native void callHiddenUnpauseSetCurrent(Display display);
}

class DialRequest extends PlatformRequest {

	String fNumber;

	DialRequest(String url) throws Exception {
		super(url, false);
		fNumber = parseNumber(url);
		// System.out.println("DialRequest created "+fNumber);
	}

	private String parseNumber(String url) {
		return url.substring(4, url.length());
	}

	protected boolean performRequest() {
		// System.out.println("DialRequest dialing "+fNumber);
		return Device.dialPhone(fNumber);
	}
}

class AmsInstallRequest extends PlatformRequest {

	protected MIDlet fMidlet;

	AmsInstallRequest(MIDlet midlet, String url) throws Exception {
		super(url, false);
		fMidlet = midlet;

		// TODO: if we are updating the current suite, pass true to super()
	}

	protected boolean performRequest() {
		try {
			startInstallProcess(fUrl);
			return true;
		} catch (AmsException e) {
			System.err.println(MidpMsg.getString("PlatformRequest.performRequest.InstallError")); //$NON-NLS-1$
			e.printStackTrace();
			return false;
		}
		// return Device.platformRequest(fUrl);
	}

	protected void startInstallProcess(final String url) throws AmsException {
		/*
		 * Run the install process in a seperate thread so that the ui is not
		 * blocked and the user has the ability to press cancel.
		 */
		Thread installThread = new Thread() {
			public void run () {
				synchronized (gInstallLock) {
					Display display = Display.getDisplay(fMidlet);
					/*
					 * To avoid any other displayable to be set as current
					 * when the install screens come up.
					 */
					callHiddenPauseSetCurrent(display);
					Displayable current = display.getCurrent();
					if (current == null) current = new Form("");
					DisplayBasedInstallListener listener = new DisplayBasedInstallListener(current, current, display);
					InstallProcess process = new InstallProcess(url, MidpConstants.INSTALL_MIDLET, listener, true);
					// Show the progress page to the user
					listener.showProgressUI(MidpMsg.getString("PlatformRequest.startInstallProcess.Installing", url)); //$NON-NLS-1$
					/* The midlet installed should be added to AMS Record Store.
					 * At this point it is pointing to the Midlets Record Store.
					 */
					MidletLoader.init("AMSMidlet", "IBM");
					process.run();
					MidletLoader.init(MIDletManager.gMidletSuiteName, MIDletManager.gVendorName);
					callHiddenUnpauseSetCurrent(display);
				}
			}
		};
		installThread.start();
	}
}

class HttpRequest extends PlatformRequest {
	HttpRequest(String url) throws Exception {
		super(url, false);
	}

	protected boolean performRequest() {
		return Device.openPlatformBrowser(fUrl);
	}
}

class MailToRequest extends PlatformRequest {
	MailToRequest(String url) throws Exception {
		super(url, false);
	}

	protected boolean performRequest() {
		return Device.openPlatformMailClient(fUrl);
	}
}

class UserRequest extends PlatformRequest {

	IPlatformRequestHandler fRequestHandler;

	UserRequest(String url, IPlatformRequestHandler handler) throws Exception {
		super(url, false);
		fRequestHandler = handler;
	}

	protected boolean performRequest() {
		try {
			fRequestHandler.handleRequest(fUrl);
		} catch (Throwable e) {
			// do not remove this stackTrace. It will let
			// the developer that handles this platformRequest know
			// that something bad happened
			e.printStackTrace();
			return false;
		}
		return true;
	}
}
