package com.ibm.ive.midp.ams.ui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006 All Rights Reserved
 */

import com.ibm.ive.midp.*;
import com.ibm.ive.midp.ams.*;
import javax.microedition.lcdui.*;
import javax.microedition.midlet.*;
import com.ibm.oti.midlet.help.*;
import com.ibm.ive.midp.util.*;
import com.ibm.oti.vm.VM;

public class MidletList extends List implements CommandListener {

	Display fDisplay;
	MIDlet famsMidlet;

	LaunchConfiguration fConfiguration;

	boolean fDeleteCommandShowing;
	boolean fLaunchCommandShowing;
	boolean fDetailsCommandShowing;
	boolean fUpdateCommandShowing;
	boolean fPermissionCommandShowing;

	AboutPage fAboutPage;

	public MidletList(Display display, MIDlet ams) {
		super(MidpMsg.getString("MidletList.constructor.title"), Choice.IMPLICIT); //$NON-NLS-1$
		if (!com.ibm.oti.vm.VM.callerIsBootstrap()) throw new SecurityException();

		fDisplay = display;
		famsMidlet = ams;
		setCommandListener(this);
		fAboutPage = new AboutPage();
		addCommand(CommandFactory.getCommand(CommandFactory.INSTALL_COMMAND));
		rebuildPage();
		addCommand(CommandFactory.getCommand(CommandFactory.ABOUT_COMMAND));
		addCommand(CommandFactory.getCommand(CommandFactory.EXIT_COMMAND));
	}

	public void rebuildPage() {
		// Remove any midlets in the current list
		deleteAll();

		int size = 0;

		// Add each midlet available to the list
		synchronized (MidletStorage.gLock) {
			size = MidletStorage.size();
			for (int i = 0; i < size; i++) {
				MidletEntry midlet = MidletStorage.get(i);
				String icon = midlet.getIcon();
				Image iconImage = null;
				if (icon != null) {
					try {
						byte[] buf = VM.fileFromZip(midlet.getJarFile(), icon);
						iconImage = Image.createImage(buf, 0, buf.length);
					} catch (RuntimeException e) {
						e.printStackTrace();
					}
				}
				append(getMidletString(midlet), iconImage);
			}
		}

		if (size > 0) {
			if (!fLaunchCommandShowing) {
				addCommand(CommandFactory.getCommand(CommandFactory.LAUNCH_COMMAND));
				fLaunchCommandShowing = true;
			}
			if (!fDetailsCommandShowing) {
				addCommand(CommandFactory.getCommand(CommandFactory.DETAILS_COMMAND));
				fDetailsCommandShowing = true;
			}
			if (!fDeleteCommandShowing) {
				addCommand(CommandFactory.getCommand(CommandFactory.DELETE_COMMAND));
				fDeleteCommandShowing = true;
			}
			if (!fUpdateCommandShowing) {
				addCommand(CommandFactory.getCommand(CommandFactory.UPDATE_COMMAND));
				fUpdateCommandShowing = true;
			}
			if (!fPermissionCommandShowing) {
				addCommand(CommandFactory.getCommand(CommandFactory.PERMISSION_COMMAND));
				fPermissionCommandShowing = true;
			}
			setTicker(null);
		} else {
			if (fDeleteCommandShowing) {
				removeCommand(CommandFactory.gDeleteCommand);
				fDeleteCommandShowing = false;
			}
			if (fLaunchCommandShowing) {
				removeCommand(CommandFactory.gLaunchCommand);
				fLaunchCommandShowing = false;
			}
			if (fDetailsCommandShowing) {
				removeCommand(CommandFactory.gDetailsCommand);
				fDetailsCommandShowing = false;
			}
			if (fUpdateCommandShowing) {
				removeCommand(CommandFactory.gUpdateCommand);
				fUpdateCommandShowing = false;
			}
			if (fPermissionCommandShowing) {
				removeCommand(CommandFactory.gPermissionCommand);
				fPermissionCommandShowing = false;
			}
			setTicker(new Ticker(MidpMsg.getString("MidletList.rebuildPage.install_midlet_instructions"))); 			 //$NON-NLS-1$
		}
	}

	/**
	 * Builds the string used to display a midlet entry.
	 *
	 * @param entry the midlet entry to display
	 * @return the string used to represent the given midlet entry
	 */
	String getMidletString(MidletEntry entry) {
		String name = entry.getName();
		String version = entry.getVersion();
		return name + " - " + version; //$NON-NLS-1$
	}

	public void commandAction(Command aCommand, Displayable aDisplayable) {
		if (aCommand == List.SELECT_COMMAND) {
			if (!Device.hasPointerEvents()) {
				/*
				 * For those platforms which do not have pointer events, we can assume
				 * that "selecting" a list item means that we want to launch the item.
				 * This is because all navigation is done with key presses, and key
				 * presses do not fire the selection command unless the user chooses
				 * "fire".
				 */
				 launchMidlet(MidletStorage.get(getSelectedIndex()));
				 return;
			}
		}

		if (aCommand == CommandFactory.gInstallCommand) {
			InstallPage iPage = new InstallPage(fDisplay, this);
			fDisplay.setCurrent(iPage);
			return;
		}

		if (aCommand == CommandFactory.gDeleteCommand) {
			handleDeletion();
			return;
		}

		if (aCommand == CommandFactory.gLaunchCommand) {
			launchMidlet(MidletStorage.get(getSelectedIndex()));
			return;
		}

		if (aCommand == CommandFactory.gDetailsCommand) {
			handleDetails();
			return;
		}

		if (aCommand == CommandFactory.gUpdateCommand) {
			AmsMidlet.startUpdateProcess(MidletStorage.get(getSelectedIndex()), fDisplay);
			return;
		}

		if (aCommand == CommandFactory.gPermissionCommand) {
			PermissionsList list = new PermissionsList(fDisplay, aDisplayable, MidletStorage.get(getSelectedIndex()));
			fDisplay.setCurrent(list);
			return;
		}

		if (aCommand == CommandFactory.gAboutCommand) {
			fAboutPage.show(fDisplay);
			return;
		}

		if (aCommand == CommandFactory.gExitCommand) {
			famsMidlet.notifyDestroyed();
			return;
		}

	}

	void handleDetails() {
		// Find the midlet that is currently selected
		synchronized (MidletStorage.gLock) {
			MidletEntry entry = MidletStorage.get(getSelectedIndex());
			MidletInfoPage page = new MidletInfoPage(entry);
			page.show(fDisplay);
		}
	}

	void handleDeletion() {
		final List list = this;
		(new Thread() {
			public void run() {
				MidletEntry entry = MidletStorage.get(getSelectedIndex());

				//Check for the Multiple MIDlets within the same MIDlet Suite
				String names[] = entry.getMIDletNames();
				int namesLength = names.length;
				StringBuffer midletSuiteInfo = new StringBuffer();
				if (namesLength > 1) {
					midletSuiteInfo.append(MidpMsg.getString("MidletList.multipleMidlets", entry.getName()));
					for (int i = 0; i < namesLength; i++) {
						midletSuiteInfo.append(names[i]+"\n");
					}
				}

				//Check for the RMS record stores assosiated with the MIDlet suite
				MidletLoader.init(entry.getName(), entry.getVendor());
				String stores[] = MidletLoader.listStores(MidletLoader.getMidletSuiteId());
				MidletLoader.init(AmsMidlet.amsMidlet.getAppProperty("MIDlet-Name"), AmsMidlet.amsMidlet.getAppProperty("MIDlet-Vendor"));
				if (stores != null) {
					midletSuiteInfo.append(MidpMsg.getString("MidletList.recordStores", entry.getName()));
					for (int i = 0; i < stores.length; i ++) {
						midletSuiteInfo.append(stores[i]+"\n");
					}
				}

				String message = MidpMsg.getString("MidletList.confirmDelete", entry.getName()); //$NON-NLS-1$
				QuestionPrompt r = new QuestionPrompt(MidpMsg.getString("ActionStatus.promptForAnswer.title"), midletSuiteInfo.append(message).toString()); //$NON-NLS-1$
				r.addResponse(MidpMsg.getString("ActionStatus.promptForAnswer.no"),1, 1); //$NON-NLS-1$
				r.addResponse(MidpMsg.getString("ActionStatus.promptForAnswer.yes"), 0, 0); //$NON-NLS-1$
				int returnValue = r.openPrompt(fDisplay);
				if (returnValue == 0 && deleteMidlet()) {
					rebuildPage();
					fDisplay.setCurrent(list);
				}
			}
		}).start();
	}

	boolean deleteMidlet() {
		// Find the midlet that is currently selected
		synchronized (MidletStorage.gLock) {
			MidletEntry entry = MidletStorage.get(getSelectedIndex());
			try {
				if (!MidletStorage.delete(entry)) {
					Alert deleteFailed = new Alert(MidpMsg.getString("InstallPage.commandAction.empty.history.title"), //$NON-NLS-1$
												   MidpMsg.getString("MidletList.deleteMidlet.error.resource_in_use"), null, AlertType.INFO); //$NON-NLS-1$
					fDisplay.setCurrent(deleteFailed);
					return false;
				}
			} catch (AmsException e) {
				// TODO: do something
				e.printStackTrace();
			}
		}
		return true;
	}

	LaunchConfiguration getLaunchConfiguration() {
		if (fConfiguration == null) fConfiguration = new LaunchConfiguration();

		return fConfiguration;
	}

	public native int getDisplayPeerHandle(Display display);

	void launchMidlet(final MidletEntry entry) {
		final MidletList list = this;
		final int handle = getDisplayPeerHandle(fDisplay);
		Device.syncExec(new Runnable() {
			public void run() {
				OS.EnableWindow(handle, false);
			}
		});

		(new Thread() {
			public void run() {
				synchronized (MidletStorage.gLock) {
					LaunchConfiguration config = getLaunchConfiguration();
					int rc = ProgramLauncher.launchProgram(config.getProgramName(), config.getProgramArguments(entry, ((AmsMidlet) famsMidlet).fVmOptions), entry, true);

					Device.syncExec(new Runnable() {
						public void run() {
							OS.EnableWindow(handle, true);
							OS.SetForegroundWindow(handle);
						}
					});

					if (rc == 0) {
						Alert launchFailureAlert = new Alert(MidpMsg.getString("MidletList.launchMidlet.error.title"), MidpMsg.getString("MidletList.launchMidlet.error.message", entry.getName()), null, AlertType.ERROR); //$NON-NLS-1$ //$NON-NLS-2$
						fDisplay.setCurrent(launchFailureAlert, list);
					}
				}
			}
		}).start();
	}
}
