package com.ibm.ive.midp.ams.ui;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2006  All Rights Reserved
 */

import com.ibm.ive.midp.*;
import com.ibm.ive.midp.ams.*;
import com.ibm.ive.midp.util.*;
import java.io.*;
import java.util.*;

import javax.microedition.io.*;
import javax.microedition.lcdui.*;

import com.ibm.oti.security.midp.PermissionManager;
import com.ibm.oti.vm.*;

public class UEIHandler implements Runnable {

	AmsMidlet fMidlet;
	String[] fCommands;

	protected static final String COMMAND_CLASSPATH = "-classpath";	//$NON-NLS-1$
	protected static final String COMMAND_HELP = "-help";	//$NON-NLS-1$
	protected static final String COMMAND_PROPERTY = "-D";	//$NON-NLS-1$
	protected static final String COMMAND_VERSION = "-version";	//$NON-NLS-1$

	protected static final String COMMAND_DEBUG = "-Xdebug"; 	//$NON-NLS-1$

	protected static final String COMMAND_DESCRIPTOR = "-Xdescriptor";	//$NON-NLS-1$
	protected static final String COMMAND_HEAPSIZE = "-Xheapsize";	//$NON-NLS-1$
	protected static final String COMMAND_JAM = "-Xjam";		//$NON-NLS-1$
	protected static final String COMMAND_QUERY = "-Xquery";	//$NON-NLS-1$
	protected static final String COMMAND_RUNJDWP = "-Xrunjdwp";	//$NON-NLS-1$
	protected static final String COMMAND_VERBOSE = "-Xverbose";	//$NON-NLS-1$
	protected static final String COMMAND_J9ARG = "-Xj9arg";	//$NON-NLS-1$

	// This is a special argument that is not defined by the UEI spec.  This is only
	// to support the J9 2.1 -Xrdbginfo arguments and all arugments that start with this
	// prefix will be passed straight through to the VM
	protected static final String COMMAND_RDBG_INFO	= "-Xrdbginfo";	//$NON-NLS-1$

	// The following constants define the -Xjam:<command>  command constants
	// as specified in the UEI spec
	protected static final String JAM_INSTALL = "install";	//$NON-NLS-1$
	protected static final String JAM_FORCE = "force";		//$NON-NLS-1$
	protected static final String JAM_LIST = "list";	 	//$NON-NLS-1$
	protected static final String JAM_STORAGE_NAMES = "storageNames"; //$NON-NLS-1$
	protected static final String JAM_RUN = "run"; 		//$NON-NLS-1$
	protected static final String JAM_REMOVE = "remove"; 	//$NON-NLS-1$
	protected static final String JAM_TRANSIENT = "transient"; 	//$NON-NLS-1$
	protected static final String JAM_ALL = "all"; 	//$NON-NLS-1$

	// The following constants define the extra -Xjam commands that this
	// implementation supports NOT defined by the UEI spec
	protected static final String JAM_AUTOTEST 	= "autotest"; 	//$NON-NLS-1$
	protected static final String JAM_CLDCTEST 	= "cldctest"; 	//$NON-NLS-1$
	protected static final String JAM_SHOWUI = "showUI"; 	//$NON-NLS-1$
	protected static final String JAM_CONSTANTS_TRUE = "true"; 	//$NON-NLS-1$

	// The following constants define the -Xrunjdwp options
	protected static final String DEBUG_ADDRESS = "address";	//$NON-NLS-1$
	protected static final String DEBUG_TRANSPORT = "transport";	//$NON-NLS-1$
	protected static final String DEBUG_SERVER = "server";		//$NON-NLS-1$
	protected static final String DEBUG_SUSPEND = "suspend";	//$NON-NLS-1$

	final static String EMULATOR_VERSION = MidpMsg.getString("UEIHandler.emulatorVersion", "2.3"); //$NON-NLS-1$//$NON-NLS-2$

	static String gBuildVersion;
	static {
		InputStream stream = VM.getResourceAsStream("/build_version.txt", false);  //$NON-NLS-1$
		if (stream != null) {
			try {
				byte[] bytes = IOUtil.getByteArray(stream);
				gBuildVersion = new String(bytes);
				stream.close();
			} catch (IOException e) {
			}
		}
	}

	boolean fVersionInfoRequested;
	boolean fHelpInfoRequested;
	boolean fQueryInfoRequested;
	boolean fDebugSpecified;
	String fDescriptorArg;
	String fClasspathArg;
	String fHeapSizeArg;
	String fRunJdwpArg;
	String fRdbgInfoArg;
	String fVerboseArg;
	String fApplicationName;
	FastVector fProperties = new FastVector();
	FastVector fJamArgs = new FastVector();
	FastVector fJ9Args = new FastVector();

	public UEIHandler(AmsMidlet midlet, String[] arguments) {
		fMidlet = midlet;
		fCommands = arguments;
	}

	public void run() {
		parseCommands();
		validateArgs();
		executeCommands();

		if (AmsMidlet.fMidletList != null) AmsMidlet.fMidletList.rebuildPage();
		if (Display.getDisplay(fMidlet).getCurrent() == null) exit();

		/*
		 * Save the VM Options so that all the MIDlets launched
		 * from the MidletList could launch with the passed in VM
		 * Options
		 */
		fMidlet.fVmOptions = getVmArguments();
	}

	void exit() {
		fMidlet.notifyDestroyed();
	}

	void parseCommands() {
		int index = 0;
		while (index < fCommands.length) {
			String arg = fCommands[index++];
			if (arg.equals(COMMAND_CLASSPATH)) {
				if (fClasspathArg == null) {
					fClasspathArg = fCommands[index++];
				} else {
					String additionalPaths = fCommands[index++];
					fClasspathArg = fClasspathArg + ';' + additionalPaths;
				}
			} else if (arg.equals(COMMAND_HELP)) {
				fHelpInfoRequested = true;
			} else if (arg.startsWith(COMMAND_PROPERTY)) {
				fProperties.addElement(arg);
			} else if (arg.equals(COMMAND_VERSION)) {
				fVersionInfoRequested = true;
			} else if (arg.equals(COMMAND_DEBUG)) {
				fDebugSpecified = true;
			}  else if (arg.startsWith(COMMAND_DESCRIPTOR + ':')) {
				fDescriptorArg = arg.substring(COMMAND_DESCRIPTOR.length() + 1);
			} else if (arg.equals(COMMAND_HEAPSIZE)) {
				fHeapSizeArg = arg;
			} else if (arg.startsWith(COMMAND_JAM)) {
				if (arg.equals(COMMAND_JAM)) {
					fJamArgs.addElement(JAM_SHOWUI);
				} else {
					if (arg.length() > COMMAND_JAM.length() + 1) {
						String jamArg = arg.substring(COMMAND_JAM.length() + 1);
						fJamArgs.addElement(jamArg);
					} else {
						error(MidpMsg.getString("UEIHandler.handleArgument.unrecognizedArg")+ ": " + arg); //$NON-NLS-1$ //$NON-NLS-2$
					}
				}
			} else if (arg.equals(COMMAND_QUERY)) {
				fQueryInfoRequested = true;
			} else if (arg.startsWith(COMMAND_RUNJDWP)) {
				fRunJdwpArg = arg;
			} else if (arg.startsWith(COMMAND_RDBG_INFO)) {
				fRdbgInfoArg = arg;
			} else if (arg.startsWith(COMMAND_VERBOSE)) {
				fVerboseArg = arg;
			} else if (arg.startsWith(COMMAND_J9ARG + ':')) {
				String j9Arg = arg.substring(COMMAND_J9ARG.length() + 1);
				fJ9Args.addElement(j9Arg);
			} else if (fClasspathArg != null && index == fCommands.length) {
				fApplicationName = arg;
			} else {
				error(MidpMsg.getString("UEIHandler.handleArgument.unrecognizedArg")+ ": " + arg); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
	}

	void validateArgs() {
		if (fJamArgs.size() > 0) {
			if (fClasspathArg != null) {
				warning(MidpMsg.getString("UEIHandler.validateArgs.ignore_other_options")); //$NON-NLS-1$
				fClasspathArg = null;
			}
			if (fApplicationName != null) {
				warning(MidpMsg.getString("UEIHandler.validateArgs.ignore_other_options")); //$NON-NLS-1$
				fApplicationName = null;
			}
			if (fDescriptorArg != null) {
				warning(MidpMsg.getString("UEIHandler.validateArgs.ignore_other_options")); //$NON-NLS-1$
				fDescriptorArg = null;
			}
			if (fQueryInfoRequested) {
				warning(MidpMsg.getString("UEIHandler.validateArgs.ignore_other_options"));//$NON-NLS-1$
				fQueryInfoRequested = false;
			}
		}

//		if (fClasspathArg != null && fApplicationName != null && fDescriptorArg != null) {
//			error(MidpMsg.getString("UEI.ApplicationAndDescriptor")); //$NON-NLS-1$
//			fDescriptorArg = null;
//		}
	}

	void executeCommands() {
		if (fVersionInfoRequested) printVersionInfo();
		if (fHelpInfoRequested) printHelpInfo();
		if (fQueryInfoRequested) printQueryInfo();

		if (fApplicationName != null) {
			launchApplication();
		} else if (fDescriptorArg != null) {
			launchDescriptor(fDescriptorArg);
		} else {
			handleJamArgs();
		}
	}

	void printVersionInfo() {
		System.out.print(EMULATOR_VERSION);
		if (gBuildVersion != null) {
			System.out.println(" build: " + gBuildVersion); //$NON-NLS-1$
		}
		System.out.println();
		System.out.println("Configuration: CLDC-1.1"); //$NON-NLS-1$
		System.out.println("Profile: MIDP-1.0, MIDP-2.0"); //$NON-NLS-1$
	}

	void printHelpInfo() {
		String build = null;
		InputStream stream = this.getClass().getResourceAsStream("/build_version.txt");  //$NON-NLS-1$
		if (stream != null) {
			try {
				byte[] bytes = IOUtil.getByteArray(stream);
				build = new String(bytes);
				stream.close();
			} catch (IOException e) {
			}
		}

		System.out.println("Licensed Materials - Property of IBM\n");  //$NON-NLS-1$
		System.out.print("IBM MIDP 2.0 Emulator; version: 2.2"); //$NON-NLS-1$
		if (build != null) {
			System.out.println(" build: " + build); //$NON-NLS-1$
		}
		System.out.println();
		System.out.println("(c) Copyright IBM Corp. 1991, 2005  All Rights Reserved");  //$NON-NLS-1$

		System.out.println("IBM is a registered trademark of IBM Corp.");  //$NON-NLS-1$
		System.out.println("Java and all Java-based marks and logos are trademarks or registered trademarks of Sun Microsystems, Inc.");  //$NON-NLS-1$

		System.out.println("Usage:  emulator [options] <MidletClassName>");  //$NON-NLS-1$

		System.out.println("[options]");  //$NON-NLS-1$
		System.out.println("-classpath <path>");  //$NON-NLS-1$
		System.out.println("-D<prop>=<val>   set the value of a system property.");  //$NON-NLS-1$
		System.out.println("-help   print this message.");  //$NON-NLS-1$
		System.out.println("-version   print version information.");  //$NON-NLS-1$
		System.out.println("-debug   enable debugging for the launched midlet. This option also requires that the -Xrunjdwp arg must be specified.");  //$NON-NLS-1$

		System.out.println("-Xverbose[:class,gc,stack,sizes]\n\tenable verbose output (default=class).");  //$NON-NLS-1$

		System.out.println("-Xdescriptor=<URL>   run the midlet specified by the given URL. This option cannot be used with -Xjam arguments or with the -classpath <path> <MidletClassName> arguments");  //$NON-NLS-1$
		System.out.println("-Xheapsize<x>   set the heap size of the launched vm. The number is specified in bytes but can be suffixed with \"k\" for kilo or \"M\" for mega.");	//$NON-NLS-1$
		System.out.println("-Xquery   query information about the emulator capabilities.");	//$NON-NLS-1$
		System.out.println("-Xrunjdwp   enable debug, JDWP standard options.");	//$NON-NLS-1$
		System.out.println("-Xj9arg: pass an argument to the j9 vm launched to run a midlet.");	//$NON-NLS-1$

		System.out.println("-Xjam[:<command>[=<args>]]   Java Application Manager args, standard UEI options.");		//$NON-NLS-1$
	}

	void printQueryInfo() {
		System.out.println("uei.arguments: Xjam, Xdebug, Xrunjdwp, Xrdbginfo"); //$NON-NLS-1$

		String securityDomains = Device.getSecurityDomain();
		System.out.println("security.domains: " + securityDomains);

		String deviceName = getDeviceName();
		System.out.println("device.list: " + deviceName); //$NON-NLS-1$
		System.out.println(deviceName + ".screen.width: " + Device.getDisplayWidth()); //$NON-NLS-1$
		System.out.println(deviceName + ".screen.height: " + Device.getDisplayHeight()); //$NON-NLS-1$
		System.out.println(deviceName + ".screen.isColor: " + Device.isColor()); //$NON-NLS-1$
		System.out.println(deviceName + ".screen.isTouch: " + Device.hasPointerEvents()); //$NON-NLS-1$
		System.out.println(deviceName + ".security.domains: " + securityDomains); //$NON-NLS-1$

		String bootClasspath[] = Device.getBootClassPathAndApisInfo();
		if (bootClasspath != null) {
			System.out.println(deviceName + ".bootclasspath: " + bootClasspath[0]); //$NON-NLS-1$
			System.out.println(deviceName + ".apis: " + bootClasspath[1]); //$NON-NLS-1$
		}
	}

	void launchApplication() {
		LaunchConfiguration config = new LaunchConfiguration();
		fApplicationName = "-midlet:" + fApplicationName;
		ProgramLauncher.launchProgram(config.getProgramName(), config.getMidpArguments(fClasspathArg, fApplicationName, getVmArguments()), null, true);
		fMidlet.notifyDestroyed();
	}

	void launchDescriptor(String jadFile) {
		if (jadFile == null || jadFile.length() == 0) {
			error(MidpMsg.getString("UEIHandler.handleDescriptorRequest.missing_descriptor", COMMAND_DESCRIPTOR));  //$NON-NLS-1$
		} else {
			if (fClasspathArg != null) {
				boolean error = false;
				PermissionManager manager = PermissionManager.getManager();
				PermissionManager.setManager(null);
				try {
					Connection c = Connector.open(jadFile);
					c.close();
					error(MidpMsg.getString("UEI.LocalDescriptorRequiredWithClasspath")); //$NON-NLS-1$
					error = true;
				} catch (Exception e) {

				} finally {
					PermissionManager.setManager(manager);
				}

				if (!error) {
					if (jadFile.startsWith("file:")) { //$NON-NLS-1$
						// Remove the "file:" string from the beginning
						jadFile = jadFile.substring(5);
					}
					LaunchConfiguration config = new LaunchConfiguration();
					ProgramLauncher.launchProgram(config.getProgramName(), config.getMidpArguments(fClasspathArg, jadFile, getVmArguments()), null, true);
				}
			} else {
				// Install the MIDlet and make sure that it succeeded.
				MidletEntry entry = install(jadFile, true);
				if (entry != null) {
					launch(entry);
					// Uninstall the midlet whether or not the midlet launched
					// correctly
					try {
						MidletStorage.delete(entry);
					} catch (AmsException e) {
						error(MidpMsg.getString("UEIHandler.handleTransient.error.midlet_failed_to_uninstall")); //$NON-NLS-1$
					}
				}
			}
		}
		fMidlet.notifyDestroyed();
	}

	boolean containsForceArgument() {
		int length = fJamArgs.size();
		for (int i = 0; i < length; i++) {
			String arg = (String) fJamArgs.elementAt(i);
			if (arg.equals(JAM_FORCE)) return true;
		}
		return false;
	}

	void jamPrintStorageNames() {
		int length = MidletStorage.size();
		for (int i = 0; i < length; i++) {
			MidletEntry entry = MidletStorage.get(i);
			System.out.println(entry.getStorageName());
		}
	}

	void jamPrintList() {
		StringBuffer buffer = new StringBuffer();
		int length = MidletStorage.size();

		for (int i = 0; i < length; i++) {
			buffer.append("[");
			buffer.append(i + 1);
			buffer.append("]\n");
			MidletEntry entry = MidletStorage.get(i);
			entry.printToApplicationList(buffer);
		}

		buffer.append("\n");
		System.out.print(buffer.toString());
	}

	void jamRun(String app) {
		MidletEntry entry = MidletStorage.get(app);
		if (entry == null) {
			showMidletList(); /* Spec requires that we show the UI if there is no valid midlet specified. */
		} else {
			launch(entry);
		}
	}

	void jamRemove(String app) {
		MidletEntry entry = MidletStorage.get(app);
		if (entry == null) {
			showMidletList(); /* Spec requires that we show the UI if there is no valid midlet specified. */
		} else {
			try {
				if (app.equals(JAM_ALL)) {
					MidletStorage.deleteAll();
					return;
				}
				MidletStorage.delete(entry);
			} catch (AmsException e) {
				System.err.println(MidpMsg.getString("Autotester.runTests.error.error_removing_midlet")); //$NON-NLS-1$
				e.printStackTrace();
			}
		}
	}

	void jamAutotest(String url) {
		if (url == null) {
			error(MidpMsg.getString("UEIHandler.handleAutotest.error.missing_url")); //$NON-NLS-1$
		} else {
			Autotester tester = new Autotester(new LaunchConfiguration(), getVmArguments());
			tester.runTests(url);
		}
	}

	void jamCldcTest(String url) {
		if (url == null) {
			error(MidpMsg.getString("UEIHandler.handleAutotest.error.missing_url")); //$NON-NLS-1$
		} else {
			new CldcTester(new LaunchConfiguration(), getVmArguments()).runTests(url);
		}
	}

	void handleJamArgs() {
		boolean force = containsForceArgument();
		int length = fJamArgs.size();
		for (int i = 0; i < length; i++) {
			String arg = (String) fJamArgs.elementAt(i);

			if (arg.equals(JAM_STORAGE_NAMES)) {
				jamPrintStorageNames();
				/* According to spec, the emulator must exit immediately after processing this option. */
				exit();
			} else if (arg.equals(JAM_LIST)) {
				jamPrintList();
				/* According to spec, the emulator must exit immediately after processing this option. */
				exit();
			} else if (arg.startsWith(JAM_INSTALL)) {
				install(parseOutApplication(arg), force);
			} else if (arg.startsWith(JAM_RUN)) {
				jamRun(parseOutApplication(arg));
			} else if (arg.startsWith(JAM_REMOVE)) {
				jamRemove(parseOutApplication(arg));
			} else if (arg.startsWith(JAM_AUTOTEST)) {
				jamAutotest(parseOutApplication(arg));
			} else if (arg.startsWith(JAM_CLDCTEST)) {
				jamCldcTest(parseOutApplication(arg));
			} else if (arg.startsWith(JAM_TRANSIENT)) {
				launchDescriptor(parseOutApplication(arg));
			} else if (arg.equals(JAM_SHOWUI)) {
				showMidletList();
			} else if (arg.equals(JAM_FORCE)) {
				// Only -Xjam:force is specified
				if (length == 1) System.err.println(MidpMsg.getString("UEIHandler.processJamRequest.invalid_force_usage")); //$NON-NLS-1$
			} else {
				System.err.println(MidpMsg.getString("UEIHandler.processJamRequest.unknown_command", arg)); //$NON-NLS-1$
			}
		}
	}

	boolean launch(MidletEntry entry) {
		LaunchConfiguration config = new LaunchConfiguration();
		int rc = ProgramLauncher.launchProgram(config.getProgramName(), config.getProgramArguments(entry, getVmArguments()), null, true);
		return rc != 0;
	}

	void error(String error) {
		System.err.println(MidpMsg.getString("UEIHandler.error.error", error)); //$NON-NLS-1$
	}

	void warning(String error) {
		System.err.println(MidpMsg.getString("UEIHandler.warning.warning", error)); //$NON-NLS-1$
	}

	String getDeviceName() {
		if (OS.IsPPCWM2003 || OS.IsPPCWM5) {
			return MidpMsg.getString("UEIHandler.getDeviceName.PocketPC"); //$NON-NLS-1$
		} else if (OS.IsSP) {
			return MidpMsg.getString("UEIHandler.getDeviceName.Smartphone"); //$NON-NLS-1$
		} else {
			return MidpMsg.getString("UEIHandler.getDeviceName.Win32"); //$NON-NLS-1$
		}
	}

	/**
	 * Returns the "application" part of the -Xjam argument.  This basicly means
	 * returning a string of all the chracters after the '='.  This method will return
	 * null if no argument was given.
	 *
	 * @param arg the entire argument, ie. "run=1"
	 * @return only the "application" part of the argument, ie. "1"
	 */
	String parseOutApplication(String arg) {
		int equalsIndex = arg.indexOf('=');
		String argument = equalsIndex == -1 ? null : arg.substring(equalsIndex + 1);
		if (argument == null || argument.equals("")) return null;

		return argument;
	}

	/**
	 * Performs the specified -Xjam:install=url command
	 *
	 * @param url the url to install
	 * @return the freshly installed midlet
	 */
	MidletEntry install(String url, boolean force) {
		if (url == null) {
			/* The spec states that the UI should be shown if the install URL is null. */
			showMidletList();
			return null;
		}

		System.out.println(MidpMsg.getString("UEIHandler.performInstall.installing", url)); //$NON-NLS-1$
		try {
			InstallProcess process = new InstallProcess(url, MidpConstants.INSTALL_MIDLET, new ConsoleBasedInstallListener(force), false);
			process.run();
			return process.getInstalledMidlet();
		} catch (Exception e) {
			System.err.println(MidpMsg.getString("UEIHandler.installFailed.failed", url)); //$NON-NLS-1$
			return null;
		}
	}

	/**
	 * Displays the list of installed MIDlets to the user.
	 * @return the MIDlet list that was shown.
	 */
	MidletList showMidletList() {
		Display d = Display.getDisplay(fMidlet);
		MidletList list = AmsMidlet.getMidletList(d);
		d.setCurrent(list);
		return list;
	}

	Hashtable parseDebugOptions(String arg) {
		Hashtable table = new Hashtable();

		// While there is still more data in the argument
		while (arg.length() > 0) {
			String keyValue;
			String key;
			String value;

			int commaIndex = arg.indexOf(',');
			if (commaIndex == -1) {
				keyValue = arg;
				arg = ""; //$NON-NLS-1$
			} else {
				keyValue = arg.substring(0, commaIndex);
				arg = arg.substring(commaIndex+1);
			}

			int equalsIndex = keyValue.indexOf('=');
			if (equalsIndex == -1) {
				System.err.println(MidpMsg.getString("UEIHandler.parseDebugOptions.error.missing_value", keyValue)); //$NON-NLS-1$
				continue;
			} else {
				// Make sure the key is valid
				key = keyValue.substring(0, equalsIndex);
				if (key.length() == 0) {
					System.err.println(MidpMsg.getString("UEIHandler.parseDebugOptions.error.missing_name", keyValue)); //$NON-NLS-1$
					continue;
				}

				// Make sure that this options has not already been defined
				if (table.get(key) != null) {
					System.err.println(MidpMsg.getString("UEIHandler.parseDebugOptions.error.key_already_defined", new String[] {keyValue, key})); //$NON-NLS-1$
					continue;
				}

				// Make sure the value is valid
				value = keyValue.substring(equalsIndex+1, keyValue.length());
				if (value.length() == 0) {
					System.err.println(MidpMsg.getString("UEIHandler.parseDebugOptions.error.empty_value", keyValue)); //$NON-NLS-1$
					continue;
				}
			}

			table.put(key, value);
		}

		return table;
	}

	String getDebugArguments() {
		if (!fDebugSpecified) return "";	 //$NON-NLS-1$

		Hashtable args = parseDebugOptions(fRunJdwpArg);

		String address = (String)args.get(DEBUG_ADDRESS);
		if (address == null || address.length() == 0) {
			System.err.println(MidpMsg.getString("UEIHandler.getDebugArguments.missing_Xrunjdwp_arg")); //$NON-NLS-1$
			return ""; //$NON-NLS-1$
		}

		// Break the address string into the port and host names
		String host;
		String portString;
		int colonIndex = address.indexOf(':');
		if (colonIndex == -1) {
			// If no colon is in the address string then it must be only the port
			// and the host is assumed to be localhost
			host = "localhost"; //$NON-NLS-1$
			portString = address;
		} else {
			host = address.substring(0, colonIndex);
			portString = address.substring(colonIndex+1);
		}

		// Make sure that the port is a number
		int port;
		try {
			port = Integer.parseInt(portString);
		} catch (NumberFormatException e) {
			System.err.println(MidpMsg.getString("UEIHandler.getDebugArguments.invalid_Xrunjdwp_address_format"));	//$NON-NLS-1$
			return "";	//$NON-NLS-1$
		}

		String returnValue = "-debug:address=" + host + ":" + port; //$NON-NLS-1$ //$NON-NLS-2$
		String server = (String) args.get(DEBUG_SERVER);
		String suspend = (String) args.get(DEBUG_SUSPEND);
		if (server != null && server.equals("y")) returnValue += ",server=y"; //$NON-NLS-1$ //$NON-NLS-2$
		if (suspend != null && suspend.equals("y")) returnValue += ",suspend=y"; //$NON-NLS-1$ //$NON-NLS-2$

		// For all the -Xrdbg args just pass the values straight through to the debugger
		if (fRdbgInfoArg != null) returnValue += ' ' +  fRdbgInfoArg;
		return returnValue;
	}

	String getVmArguments() {
		String debugArgs = getDebugArguments();
		StringBuffer buffer = new StringBuffer(debugArgs);
		if (debugArgs.length() > 0) buffer.append(" ");

		//Append the passed in j9args
		for (int i = 0; i < fJ9Args.size(); i++) {
			buffer.append(LaunchConfiguration.quoteIfNeeded((String)fJ9Args.elementAt(i)));
			buffer.append(" ");
		}
		return buffer.toString();
	}
}
