package com.ibm.microedition.media;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import java.io.*;
import javax.microedition.media.*;
import com.ibm.microedition.media.control.*;

public class MIDIPlayer extends ExtendedMediaPlayer {
	static public native int    midiPlayerCreate(MediaPlayer mediaPlayer, int[] midiData);
	static public native int    midiPlayerDestroy(int handle);
	static public native int    midiPlayerPlay(int handle, int mediaTime, int loopCount);
	static public native int    midiPlayerStop(int handle);
	static public native int    midiPlayerPause(int handle);
	static public native int    midiPlayerRestart(int handle);
	static public native int    midiPlayerGetDuration(int handle);
	static public native int    midiPlayerGetMediaTime(int handle);
	static public native int    midiPlayerSetVolume(int handle, int volume);
	static public native int    midiPlayerGetVolume(int handle);

	protected int[]  fMidiData;
	VolumeCtrl midiVolCtrl = new VolumeCtrl(this);

	public MIDIPlayer() {
		super();
		MediaEventManager.registerPlayer(this);
	}

	public void close(boolean cleanUp) {
		close();
	}

	public Control getControl(String controlType) {
		if (null == controlType) throw new IllegalArgumentException();

		evaluateOperation(MediaPlayer.OPERATION_GETCONTROL);
		if (controlType.equals("VolumeControl") || controlType.equals("javax.microedition.media.control.VolumeControl")) { //$NON-NLS-1$ //$NON-NLS-2$
			return midiVolCtrl;
		}
		return null;
	}

	public Control[] getControls() {
		evaluateOperation(MediaPlayer.OPERATION_GETCONTROL);
		return new Control[] {midiVolCtrl};
	}

	public String getContentType() {
		evaluateOperation(MediaPlayer.OPERATION_GETCONTENTTYPE);
		return MediaConstants.MIME_TYPE_MIDI;
	}

	protected void realizeImpl() throws MediaException {
		try {
			if (readData() == false) {
				throw new MediaException(getTranslatedString("MediaPlayer.msg2") );
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public void prefetchImpl() throws MediaException {
		if (0 != fHandle) return;

		if (parseMidiHeader() == false) {
			throw new MediaException(getTranslatedString("MediaPlayer.msg1") );
		}

		fHandle = midiPlayerCreate(this, fMidiData);
		if (0 == fHandle) {
			throw new MediaException(getTranslatedString("MediaPlayer.native.msg1"));
		}
		// If the user has set the volume, pass it to the device
		int volumeLevel = midiVolCtrl.getLevel();
		if (volumeLevel != -1) midiPlayerSetVolume(fHandle, volumeLevel);
	}

	public void startImpl() throws MediaException {
		if (0 == fHandle) return;

		Thread thread = new Thread() {
			public void run() {
//				Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
				synchronized (fCloseMutex) {
					setState(Player.STARTED);
						if (fExplicitlyStopped) {
							fExplicitlyStopped = false;
							midiPlayerRestart(fHandle);
						} else {
							midiPlayerPlay(fHandle, -1, getLoopCount());
						}
				}
				setState(Player.PREFETCHED);
			}
		};

//		fStartedEvent = new Object();
//
//		synchronized (fStartedEvent) {
//			thread.start();
//			try {
//				fStartedEvent.wait(5000); // timed wait in case play thread doesn't signal
//				fStartedEvent = new Object();
//			}
//			catch (InterruptedException e) {}
//		}

		//TODO: need to find a better way of passing TCK without
		//      having to set thread priority below.

		thread.setPriority(Thread.MAX_PRIORITY);
		thread.start();

	}

	public void stopImpl() throws MediaException {
		if (0 == fHandle) return;

		if (getState() != Player.STARTED) return;

		if (fExplicitlyStopped) return;

		fExplicitlyStopped = true;
		int rc = midiPlayerPause(fHandle);

		if (rc != 0){
			throw new MediaException(getTranslatedString("MediaPlayer.native.msg2"));
		}
	}

	protected void deallocateImpl() {
		if (getState() == Player.PREFETCHED) {
			if (fHandle != 0) {
				try {
					fExplicitlyStopped = true;
					midiPlayerStop(fHandle);
				} catch (IllegalStateException ie) {
					ie.printStackTrace();
				}

				synchronized (fCloseMutex) {
					midiPlayerDestroy(fHandle);
				}

				fHandle = 0;
			}
		}
	}

	public void closeImpl()  {
		if (fHandle != 0) {
			try {
				fExplicitlyStopped = true;
				midiPlayerStop(fHandle);
			} catch (IllegalStateException ie) {
				ie.printStackTrace();
			}

			synchronized (fCloseMutex) {
				midiPlayerDestroy(fHandle);
			}

			fHandle = 0;
		}
		/* If fHandle is not obtained, native calls fail.
		 * This can happen when you do a realize() and
		 * then close the player.
		 */
		MediaEventManager.postEventNull(this, PlayerListener.CLOSED);
	}

	public long getDuration() {
		evaluateOperation(OPERATION_GETDURATION);

		if (fMidiData == null) return Player.TIME_UNKNOWN;

		int duration = midiPlayerGetDuration(fHandle);
		if (-1 == duration) return Player.TIME_UNKNOWN;

		return duration * 1000;
	}

	public long setMediaTime(long mediaTime) throws MediaException {
		evaluateOperation(OPERATION_SETMEDIATIME);
		throw new MediaException(getTranslatedString("AbstractPlayer.msg2"));
	}

	public long getMediaTime() {
		evaluateOperation(OPERATION_GETMEDIATIME);

//		if (0 == fHandle) return Player.TIME_UNKNOWN;
//
//		if (getState() != Player.STARTED) return fMediaTime * 1000;
//
//		int mediaTime = midiPlayerGetMediaTime(fHandle);
////		System.out.println("Media Time:" + mediaTime);
//		if (-1 == mediaTime) return Player.TIME_UNKNOWN;
//
//		return mediaTime * 1000;

		return Player.TIME_UNKNOWN;
	}

	public int setVolumeLevelImpl(int volumeLevel) {
		return 0 == fHandle ? volumeLevel : midiPlayerSetVolume(fHandle, volumeLevel);
	}

	public int getVolumeLevelImpl() {
		return 0 == fHandle ? 0 : midiPlayerGetVolume(fHandle);
	}

	/*  Read header info from fData for format 0 and num. tracks */
	private boolean parseMidiHeader() throws MediaException {
		int length = fData.length;
		byte[] header = new byte[14];
		byte[] data= new byte[length];

		System.arraycopy(fData, 0, header, 0, 14);

		if (header[0] != 77 || header[1] != 84 || header[2] != 104 || header[3] != 100) {
			throw new MediaException(getTranslatedString("MIDIPlayer.msg1"));
		}

		int	ntrks = header[10] << 8 | header[11];
		if (ntrks > 1) ntrks = 1;

		System.arraycopy(fData, 14, data, 0, length - 14);

		/* check for 'Mtrk' */
		if (data[0] != 77 || data[1] != 84 || data[2] != 114 || data[3] != 107) {
			throw new MediaException(getTranslatedString("MIDIPlayer.msg1"));
		}

		fMidiData = new int[length];
		for (int i = 0; i < length; i++) {
			fMidiData[i] = fData[i] & 0x000000ff;
		}
		return true;
	}
}
