package com.ibm.microedition.media;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2003, 2005  All Rights Reserved
 */

import java.io.IOException;
import javax.microedition.media.*;
import com.ibm.microedition.media.control.*;

/**
 * WAVE Player to play .wav files
 *
 */
public class WAVPlayer extends ExtendedMediaPlayer  {
	static public native int    wavePlayerCreate       (MediaPlayer mediaPlayer, byte[] midiData);
	static public native int    wavePlayerDestroy      (int handle);
	static public native int    wavePlayerPlay  (int handle, int mediaTime, int loopCount);
	static public native int    wavePlayerStop         (int handle);
	static public native int    wavePlayerPause        (int handle);
	static public native int    wavePlayerRestart      (int handle);
	static public native int    wavePlayerGetDuration  (int handle);
	static public native int    wavePlayerGetMediaTime (int handle);
	static public native int    wavePlayerSetVolume    (int handle, int volume);
	static public native int    wavePlayerGetVolume    (int handle);

	protected int[]  fMidiData;
	VolumeCtrl wavVolCtrl = new VolumeCtrl(this);

	public WAVPlayer() {
		super();
		MediaEventManager.registerPlayer(this);
	}

	public void close(boolean cleanUp) {
		close();
	}

	public Control getControl(String controlType) {
		if (null == controlType) throw new IllegalArgumentException();

		evaluateOperation(MediaPlayer.OPERATION_GETCONTROL);
		if (controlType.equals("VolumeControl") || controlType.equals("javax.microedition.media.control.VolumeControl")) { //$NON-NLS-1$ //$NON-NLS-2$
			return wavVolCtrl;
		}
		return null;
	}

	public Control[] getControls() {
		evaluateOperation(MediaPlayer.OPERATION_GETCONTROL);
		return new Control[] {wavVolCtrl};
	}

	public String getContentType() {
		evaluateOperation(MediaPlayer.OPERATION_GETCONTENTTYPE);
		return MediaConstants.MIME_TYPE_WAVE;
	}

	protected void realizeImpl() throws MediaException {
		try {
			if (readData() == false) {
				throw new MediaException(getTranslatedString("MediaPlayer.msg2") );
			}
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	public void prefetchImpl() throws MediaException {
		if (0 != fHandle) return;

		fHandle = wavePlayerCreate(this, fData);
		if (0 == fHandle) {
			throw new MediaException(getTranslatedString("MediaPlayer.msg3")); //$NON-NLS-1$
		}
		// If the user has set the volume, pass it to the device
		int volumeLevel = wavVolCtrl.getLevel();
		if (volumeLevel != -1) wavePlayerSetVolume(fHandle, volumeLevel);
	}

	public void startImpl() throws MediaException {
		if (0 == fHandle) return;

		Thread thread = new Thread() {
			public void run() {
//				Thread.currentThread().setPriority(Thread.MAX_PRIORITY);
				int loopCount = getLoopCount();
				if (fMediaTime != 0 && loopCount > 1) loopCount--;

				synchronized (fCloseMutex) {
					setState(Player.STARTED);
					if (fExplicitlyStopped ) {
						fExplicitlyStopped = false;
						wavePlayerRestart(fHandle);
					} else {
						wavePlayerPlay(fHandle, fMediaTime, getLoopCount());
					}
				}

				setState(Player.PREFETCHED);
			}
		};

//		fStartedEvent = new Object();
//
//		synchronized (fStartedEvent) {
//			thread.start();
//			try {
//				fStartedEvent.wait(5000); // timed wait in case play thread doesn't signal
//				fStartedEvent = new Object();
//			}
//			catch (InterruptedException e) {}
//		}

		//TODO: need to find a better way of passing TCK without
		//      having to set thread priority below.

		thread.setPriority(Thread.MAX_PRIORITY);
		thread.start();

	}

	public void stopImpl() throws MediaException {
		if (0 == fHandle) return;

		if (getState() != Player.STARTED) return;

		if (fExplicitlyStopped) return;

		fExplicitlyStopped = true;
		int rc = wavePlayerPause(fHandle);

		if (rc != 0){
			throw new MediaException(getTranslatedString("MediaPlayer.native.msg2"));
		}
	}

	public void deallocateImpl()  {
		if (getState() == Player.PREFETCHED) {
			if (fHandle != 0) {
				try {
					fExplicitlyStopped = true;
					wavePlayerStop(fHandle);
				} catch (IllegalStateException ie) {
					ie.printStackTrace();
				}

				synchronized (fCloseMutex) {
					wavePlayerDestroy(fHandle);
				}

				fHandle = 0;
			}
		}
	}

	public void closeImpl()  {
		if (fHandle != 0) {
			try {
				fExplicitlyStopped = true;
				wavePlayerStop(fHandle);
			} catch (IllegalStateException ie) {
				ie.printStackTrace();
			}

			synchronized (fCloseMutex) {
				wavePlayerDestroy(fHandle);
			}

			fHandle = 0;
		}
		/* If fHandle is not obtained, native calls fail.
		 * This can happen when you do a realize() and
		 * then close the player.
		 */
		MediaEventManager.postEventNull(this, PlayerListener.CLOSED);
	}

	public long getDuration() {
		evaluateOperation(OPERATION_GETDURATION);
		if (0 == fHandle) return Player.TIME_UNKNOWN;

		int duration = wavePlayerGetDuration(fHandle);
		if (-1 == duration) return Player.TIME_UNKNOWN;

		return duration * 1000;
	}

	public long setMediaTime(long mediaTime) throws MediaException {
		evaluateOperation(OPERATION_SETMEDIATIME);
		if (mediaTime <= 0 ) {
			mediaTime = 0;
		} else {
			long duration = getDuration();
			if (duration < mediaTime) mediaTime = duration;
		}

		if (getState() != Player.STARTED) {
			fMediaTime = (int) mediaTime / 1000;
			return mediaTime;
		}
		stop();
		fMediaTime = (int) mediaTime / 1000;
		fExplicitlyStopped = false; // Player should play with new media time and not restart!
		start();

		return mediaTime;
	}

	public long getMediaTime() {
		evaluateOperation(OPERATION_GETMEDIATIME);

		if (0 == fHandle) return Player.TIME_UNKNOWN;

		if (getState() != Player.STARTED) return fMediaTime * 1000;

		int mediaTime = wavePlayerGetMediaTime(fHandle);
		if (-1 == mediaTime) return Player.TIME_UNKNOWN;

		return mediaTime * 1000;
	}

	public int setVolumeLevelImpl(int volumeLevel) {
		return 0 == fHandle ? volumeLevel : wavePlayerSetVolume(fHandle, volumeLevel);
	}

	public int getVolumeLevelImpl() {
		return 0 == fHandle ? 0 : wavePlayerGetVolume(fHandle);
	}
}
