package com.ibm.oti.connection;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 1998, 2004  All Rights Reserved
 */

import java.io.*;

/**
 * BufferedInputStream is a class which takes an input stream and <em>buffers</em> the
 * input.  In this way, costly interaction with the original input stream can be minimized
 * by reading buffered amounts of data infrequently.  The drawback is that extra space is
 * required to hold the buffer and that copying takes place when reading that buffer.
 *
 * @author		OTI
 * @version		initial
 *
 * @see 		BufferedOutputStream
 */
public class BufferedInputStream extends InputStream {
	/**
	 * The target InputStream which is being filtered.
	 */
	protected InputStream in;
	/**
	 * The buffer containing the current bytes read from the target InputStream.
	 */
	protected byte[] buf;
	/**
	 * The total number of bytes inside the byte array <code>buf</code>.
	 */
	protected int count;
	/**
	 * The current position within the byte array <code>buf</code>.
	 */
	protected int pos;

/**
 * Constructs a new BufferedInputStream on the InputStream <code>in</code>.  The default
 * buffer size (512 bytes) is allocated and all reads can now be filtered through this stream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		in		the InputStream to buffer reads on.
 *
 */
public BufferedInputStream(InputStream in) {
	this.in = in;
	buf = new byte[512];
}

/**
 * Constructs a new BufferedInputStream on the InputStream <code>in</code>.  The
 * buffer size is specified by the parameter <code>size</code> and all reads can
 * now be filtered through this BufferedInputStream.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		in		the InputStream to buffer reads on.
 * @param		size	the size of buffer to allocate.
 */
public BufferedInputStream(InputStream in, int size) {
	this.in = in;
	if (size > 0) buf = new byte[size];
	// K0058 = size must be > 0
	else throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0058"));
}

/**
 * Answers a int representing then number of bytes that are available
 * before this BufferedInputStream will block.  This method returns the
 * number of bytes available in the buffer plus those available
 * in the target stream.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		the number of bytes available before blocking.
 *
 * @exception 	java.io.IOException	If an error occurs in this stream.
 */
public synchronized int available() throws IOException {
	return count - pos +  in.available();
}

/**
 * Close this BufferedInputStream. This implementation closes the target stream and
 * releases any resources associated with it.
 *
 * @author		OTI
 * @version		initial
 *
 * @exception 	java.io.IOException	If an error occurs attempting to close this stream.
 */
public void close() throws IOException {
	in.close();
	buf = null;
}

private int fillbuf() throws IOException {
	int result = in.read(buf);
	pos = 0;
	count = result == -1 ? 0:result;
	return result;
}

/**
 * Reads a single byte from this BufferedInputStream and returns the result as
 * an int.  The low-order byte is returned or -1 of the end of stream was
 * encountered.  If the underlying buffer does not contain any available bytes
 * then it is filled and the first byte is returned.
 *
 * @author		OTI
 * @version		initial
 *
 * @return 		the byte read or -1 if end of stream.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public synchronized int read() throws IOException {
	if (buf !=null) {
		/* Are there buffered bytes available? */
		if (pos >= count && fillbuf() == -1) return -1; /* no, fill buffer */

		/* Did filling the buffer fail with -1 (EOF)? */
		if (count - pos > 0) return buf[pos++] & 0xFF;
		return -1;
	}
	// K0059 = Stream is closed
	throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
}

/**
 * Reads at most <code>length</code> bytes from this BufferedInputStream and stores them in byte
 * array <code>buffer</code> starting at offset <code>offset</code>. Answer the number of bytes
 * actually read or -1 if no bytes were read and end of stream was encountered.  If all the
 * buffered bytes have been used, a mark has not been set, and the requested number
 * of bytes is larger than the receiver's buffer size, this implementation bypasses
 * the buffer and simply places the results directly into <code>buffer</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		buffer	the byte array in which to store the read bytes.
 * @param		offset	the offset in <code>buffer</code> to store the read bytes.
 * @param		length	the maximum number of bytes to store in <code>buffer</code>.
 * @return 		the number of bytes actually read or -1 if end of stream.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public synchronized int read(byte[] buffer, int offset, int length) throws IOException {
	if (buf != null && buffer != null) {
		// avoid int overflow
		if (0 <= offset && offset <= buffer.length && 0 <= length && length <= buffer.length - offset) {
			if (length == 0) return 0;

			int required;
			if (pos < count) {
				/* There are bytes available in the buffer. */
				int copylength = count - pos >= length ? length : count - pos;
				System.arraycopy(buf, pos, buffer, offset, copylength);
				pos += copylength;
				if (copylength == length || in.available() == 0)
					return copylength;
				offset += copylength;
				required = length - copylength;
			} else required = length;

			while (true) {
				int read;
				/* If we're not marked and the required size is greater than
				 * the buffer, simply read the bytes directly bypassing the
				 * buffer. */
				if (required >= buf.length) {
					read = in.read(buffer, offset, required);
					if (read == -1)
						return required == length ? -1 : length - required;
				} else {
					if (fillbuf() == -1)
						return required == length ? -1 : length - required;
					read = count - pos >= required ? required : count - pos;
					System.arraycopy(buf, pos, buffer, offset, read);
					pos += read;
				}
				required -= read;
				if (required == 0) return length;
				if (in.available() == 0) return length - required;
				offset += read;
			}
		} else throw new ArrayIndexOutOfBoundsException();
	} else {
		// K0059 = Stream is closed
		if (buf == null) throw new IOException(com.ibm.oti.util.Msg.getString("K0059"));
		// K0047 = buffer is null
		throw new NullPointerException(com.ibm.oti.util.Msg.getString("K0047"));
	}
}

/**
 * Skips <code>amount</code> number of bytes in this BufferedInputStream.  Subsequent
 * <code>read()</code>'s will not return these bytes unless <code>reset()</code>
 * is used.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		amount		the number of bytes to skip.
 * @return		the number of bytes actually skipped.
 *
 * @exception 	java.io.IOException If the stream is already closed or another IOException occurs.
 */
public synchronized long skip(long amount) throws IOException {
	if (amount < 1) return 0;

	if (count - pos >= amount) {
		pos += amount;
		return amount;
	}
	long read = count - pos;
	pos = count;
	return read + in.skip(amount - read);
}
}
