package com.ibm.oti.connection.datagram;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2005  All Rights Reserved
 */

import java.io.*;
import javax.microedition.io.*;
import com.ibm.oti.connection.socket.Socket;

/**
 * Implements the javax.microedition.io.Datagram interface.
 *
 * @see		javax.microedition.io.Datagram
 */

public class DatagramPacket implements Datagram {
	String host;
	int port;
	private byte[] data;
	private int length, offset;
	private int pos;
	private boolean changeLength;

String getHost() {
	return host;
}

public String getAddress() {
	if (host == null) return null;
	return "datagram://" + host + ":" + port;
}

public byte[] getData() {
	return data;
}

public int getLength() {
	return length;
}

public int getOffset() {
	return offset;
}

public void reset() {
	offset = 0;
	length = 0;
	pos = 0;
	changeLength = true;
}

public void setAddress(Datagram address) {
	if (address == null)
		throw new IllegalArgumentException();
	String addressString = address.getAddress();
	if (addressString == null)
		throw new IllegalArgumentException();
	try {
		setAddress(addressString);
	} catch (IOException e) {
		throw new IllegalArgumentException(e.getMessage());
	}
}

public void setAddress(String rawAddress) throws IOException {
	if (rawAddress == null || !rawAddress.startsWith("datagram:"))
		// K00a0 = invalid protocol, must be 'datagram:'
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00a0"));
	String addr = rawAddress.substring(9);
	if (!addr.startsWith("//"))
		// K00a1 = Missing //
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00a1"));
	int[] result = new int[1];
	host = Socket.parseURL(addr, result, true, false);
	port = result[0];
}

public void setData(byte[] data, int start, int length) {
	if (data == null || start < 0 || length < 0 || data.length - start < length) // start + length could overflow
		throw new IllegalArgumentException();
	this.data = data;
	offset = start;
	this.length = length;
	pos = start;
	changeLength = false;
}

public void setLength(int length) {
	if (length < 0 || length > (getData().length - getOffset()))
		throw new IllegalArgumentException();
	this.length = length;
	changeLength = false;
}

/**
 * Writes <code>count</code> bytes from the byte array
 * <code>buffer</code> starting at offset <code>index</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		buffer		the buffer to be written
 * @param		offset		offset in buffer to get bytes
 * @param		count		number of bytes in buffer to write
 *
 * @exception 	java.io.IOException If an error occurs attempting to write.
 *
 * @see 		DataInput#readFully(byte[])
 * @see 		DataInput#readFully(byte[], int, int)
 */
public void write(byte buffer[], int offset, int count) throws IOException {
	if (offset < 0 || count < 0 || offset > buffer.length || buffer.length - offset < count) // offset + count could overflow
		throw new IndexOutOfBoundsException();
	if (changeLength) {
		if (count > getData().length - pos) throw new EOFException();
	} else if (count > getLength() - (pos - offset)) throw new EOFException();
	System.arraycopy(buffer, offset, data, pos, count);
	pos += count;
	if (changeLength)
		length = pos - this.offset;
}

/**
 * Writes the specified <code>byte</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		oneByte		the byte to be written
 *
 * @exception 	java.io.IOException If an error occurs attempting to write.
 *
 * @see 		DataInput#readByte()
 */
public void write(int oneByte) throws IOException {
	if (changeLength) {
		if (pos == getData().length) throw new EOFException();
	} else if (pos - getOffset() == getLength()) throw new EOFException();
	data[pos++] = (byte)oneByte;
	if (changeLength)
		length = pos - offset;
}

/**
 * Writes a boolean.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		val	the boolean value to write
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 *
 * @see 		DataInput#readBoolean()
 */
public void writeBoolean(boolean val) throws IOException {
	write(val ? 1:0);
}

/**
 * Writes a 8-bit byte.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		val	the byte value to write
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 *
 * @see 		DataInput#readByte()
 * @see 		DataInput#readUnsignedByte()
 */
public void writeByte(int val) throws IOException {
	write(val);
}

/**
 * Writes the specified 16-bit character. The higher of the 2 bytes is
 * written first. This represents the Unicode value of val.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val		the character to be written
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 *
 * @see 		DataInput#readChar()
 */
public void writeChar(int val) throws IOException {
	write(val >> 8);
	write(val);
}

/**
 * Writes the specified 16-bit characters contained in str to the OutputStream.
 * The higher of the 2 bytes for each character is written first. This represents
 * the Unicode value of each character in str.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		str		the String whose characters are to be written.
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 *
 * @see 		DataInput#readChar()
 */
public void writeChars(String str) throws IOException {
	int destIndex = 0;
	byte newBytes[] = new byte[str.length() * 2];
	for (int index = 0; index < str.length(); index++) {
		newBytes[destIndex++] = (byte)(str.charAt(index) >> 8);
		newBytes[destIndex++] = (byte)str.charAt(index);
	}
	write(newBytes);
}

/**
 * Writes a 32-bit int.  The resulting output
 * is the 4 bytes, highest order first, of val.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val		the int to be written.
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 *
 * @see 		DataInput#readInt()
 */
public void writeInt(int val) throws IOException {
	write(val >> 24);
	write(val >> 16);
	write(val >> 8);
	write(val);
}

/**
 * Writes a 64-bit long.  The resulting output
 * is the 8 bytes, highest order first, of val.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val		the long to be written.
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 *
 * @see 		DataInput#readLong()
 */
public void writeLong(long val) throws IOException {
	writeInt((int)(val >> 32));
	writeInt((int)val);
}

/**
 * Writes the specified 16-bit short. The higher of the 2 bytes is written first.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val		the short to be written
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 *
 * @see 		DataInput#readShort()
 * @see 		DataInput#readUnsignedShort()
 */
public void writeShort(int val) throws IOException {
	writeChar(val);
}

/**
 * Writes the specified String out in UTF format.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		str		the String to be written in UTF format.
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 *
 * @see 		DataInput#readUTF()
 */
public final void writeUTF(String str) throws IOException {
	int utfCount = 0, length = str.length();
	for (int i = 0; i < length; i++) {
		int charValue = (int) str.charAt(i);
		if (charValue > 0 && charValue <= 127)
			utfCount++;
		else
			if (charValue <= 2047)
				utfCount += 2;
			else
				utfCount += 3;
	}
	if (utfCount > 65535)
		// K0068 = String is too long
		throw new UTFDataFormatException(com.ibm.oti.util.Msg.getString("K0068"));
	byte utfBytes[] = new byte[utfCount];
	int utfIndex = 0;
	for (int i = 0; i < length; i++) {
		int charValue = (int) str.charAt(i);
		if (charValue > 0 && charValue <= 127)
			utfBytes[utfIndex++] = (byte) charValue;
		else
			if (charValue <= 2047) {
				utfBytes[utfIndex++] = (byte) (0xc0 | (0x1f & (charValue >> 6)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & charValue));
			} else {
				utfBytes[utfIndex++] = (byte) (0xe0 | (0x0f & (charValue >> 12)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & (charValue >> 6)));
				utfBytes[utfIndex++] = (byte) (0x80 | (0x3f & charValue));
			}
	}
	writeShort(utfCount);
	if (utfCount > 0)
		write(utfBytes);
}

/**
 * Writes the entire contents of the byte array <code>buffer</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		buffer		the buffer to be written
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write.
 */

public void write(byte buffer[]) throws IOException {
	write(buffer, 0, buffer.length);
}

/**
 * Reads a boolean.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next boolean value.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#writeBoolean(boolean)
 */
public boolean readBoolean() throws IOException {
	return readUnsignedByte() != 0;
}

/**
 * Reads an 8-bit byte value.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next byte value.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#writeByte(int)
 */
public byte readByte() throws IOException {
	if (pos - getOffset() == getLength()) throw new EOFException();
	return (byte)data[pos++];
}

/**
 * Reads a 16-bit character value.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next <code>char</code> value.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#writeChar(int)
 */
public char readChar() throws IOException {
	int b1 = readUnsignedByte();
	int b2 = readUnsignedByte();
	return (char) ((b1 << 8) + b2);
}

/**
 * Reads bytes into the byte array <code>buffer</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		buffer	the buffer to read bytes into
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#write(byte[])
 * @see 		DataOutput#write(byte[], int, int)
 */
public void readFully(byte[] buffer) throws IOException {
	readFully(buffer, 0, buffer.length);
}

/**
 * Read bytes and stores them in byte array <code>buffer</code> starting
 * at offset <code>offset</code>.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		buffer	the byte array in which to store the read bytes.
 * @param		offset	the offset in <code>buffer</code> to store the read bytes.
 * @param		count	the maximum number of bytes to store in <code>buffer</code>.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#write(byte[])
 * @see 		DataOutput#write(byte[], int, int)
 */
public void readFully(byte[] buffer, int offset, int count) throws IOException {
	// avoid int overflow
	if (offset < 0 || count < 0 || offset > buffer.length || buffer.length - offset < count)
		throw new IndexOutOfBoundsException();
	if (count > getLength() - (pos - getOffset())) throw new EOFException();
	System.arraycopy(data, pos, buffer, offset, count);
	pos += count;
}

/**
 * Reads a 32-bit integer value.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next <code>int</code> value from the source stream.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#writeInt(int)
 */
public int readInt() throws IOException {
	int b1 = readUnsignedByte();
	int b2 = readUnsignedByte();
	int b3 = readUnsignedByte();
	int b4 = readUnsignedByte();
	return ((b1 << 24) + (b2 << 16) + (b3 << 8) + b4);
}

/**
 * Reads a 64-bit <code>long</code> value.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next <code>long</code> value from the source stream.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#writeLong(long)
 */
public long readLong() throws IOException {
	int i1 = readInt();
	int i2 = readInt();
	return ((((long) i1) << 32) + (i2 & 0xFFFFFFFFL));
}

/**
 * Reads a 16-bit <code>short</code> value.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next <code>short</code> value from the source stream.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#writeShort(int)
 */
public short readShort() throws IOException {
	return (short)readUnsignedShort();
}

/**
 * Reads an unsigned 8-bit <code>byte</code> value and returns it
 * as an int.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next unsigned byte value from the source stream.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#writeByte(int)
 */
public final int readUnsignedByte() throws IOException {
	if (pos - getOffset() == getLength()) throw new EOFException();
	return data[pos++] & 0xff;
}

/**
 * Reads a 16-bit unsigned <code>short</code> value and returns
 * it as an int.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next unsigned <code>short</code> value from the source stream.
 *
 * @exception 	java.io.IOException	if a problem occurs reading.
 *
 * @see 		DataOutput#writeShort(int)
 */
public final int readUnsignedShort() throws IOException {
	int b1 = readUnsignedByte();
	int b2 = readUnsignedByte();
	return ((b1 << 8) + b2);
}

/**
 * Reads a UTF format String.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next UTF String.
 *
 * @exception 	java.io.IOException if a problem occurs reading.
 *
 * @see 		DataOutput#writeUTF(java.lang.String)
 */
public final String readUTF() throws IOException {
	int utfSize = readUnsignedShort();
	byte[] buf = new byte[utfSize];
	readFully(buf, 0, utfSize);
	return com.ibm.oti.util.Util.convertFromUTF8(buf, 0, utfSize);
}

/**
 * Skips <code>count</code> number of bytes.  Subsequent
 * <code>read()</code>'s will not return these bytes unless <code>reset()</code>
 * is used.
 *
 * @author		OTI
 * @version		initial
 *
 * @param 		count		the number of bytes to skip.
 * @return		the number of bytes actually skipped.
 */
public int skipBytes(int count) throws IOException {
	int max = getLength() - (pos - getOffset());
	if (count > max) count = max;
	pos += count;
	return count;
}

/**
 * Reads a 32-bit <code>float</code> value from this stream.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next <code>float</code> value from the source stream.
 *
 * @exception 	java.io.IOException	if a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeFloat(float)
 */
public final float readFloat() throws IOException {
	return Float.intBitsToFloat(readInt());
}

/**
 * Reads a 64-bit <code>double</code> value from this stream.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		the next <code>double</code> value from the source stream.
 *
 * @exception 	java.io.IOException	if a problem occurs reading from this DataInputStream.
 *
 * @see 		DataOutput#writeDouble(double)
 */
public final double readDouble() throws IOException {
	return Double.longBitsToDouble(readLong());
}

/**
 * Writes a 32-bit float to this output stream.  The resulting output
 * is the 4 bytes resulting from calling Float.floatToIntBits().
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val		the float to be written.
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readFloat()
 */
public final void writeFloat(float val) throws IOException {
	writeInt(Float.floatToIntBits(val));
}

/**
 * Writes a 64-bit double to this output stream.  The resulting output
 * is the 8 bytes resulting from calling Double.doubleToLongBits().
 *
 * @author		OTI
 * @version		initial
 *
 * @param		val		the double to be written.
 *
 * @exception 	java.io.IOException	If an error occurs attempting to write to this DataOutputStream.
 *
 * @see 		DataInput#readDouble()
 */
public final void writeDouble(double val) throws IOException {
	writeLong(Double.doubleToLongBits(val));
}

}
