
package com.ibm.oti.connection.https;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2002, 2006  All Rights Reserved
 */

import java.io.ByteArrayInputStream;
import java.io.IOException;

import javax.microedition.io.StreamConnection;
import javax.microedition.io.Connector;

import com.ibm.oti.security.provider.X500Principal;
import com.ibm.oti.util.ASN1Decoder;
import com.ibm.oti.util.ASN1Exception;
import com.ibm.j9.ssl.Util;
import com.ibm.oti.security.provider.X509Certificate;

import javax.microedition.io.HttpsConnection;
import javax.microedition.io.SecurityInfo;
import javax.microedition.pki.Certificate;

public class Connection extends com.ibm.oti.connection.http.Connection implements HttpsConnection {

	private com.ibm.oti.connection.ssl.Connection sslConnection = null;

public Connection() {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
}

/**
 * Answers the security information associated with this connection.
 *
 * @return the security information for this connection
 * @throws IOException if a connection failure occurs
 */
public SecurityInfo getSecurityInfo() throws IOException {
	if (isClosed()) throw new IOException();
	if (sslConnection == null) {
		super.connect();
	}
	return sslConnection.getSecurityInfo();
}

/**
 * Answers the protocol component of this connection.
 *
 * @author		OTI
 * @version		initial
 *
 * @return		String
 *					the receiver's protocol.
 */
public String getProtocol() {
	return "https";
}

/**
 * @see com.ibm.oti.connection.http.Connection#getDefaultPort()
 */
protected int getDefaultPort() {
	return 443;
}

/**
 * @see com.ibm.oti.connection.http.Connection#openSocket(boolean, String)
 */
protected StreamConnection openSocket(boolean timeout, String socketOptions) throws IOException {
	if (sslConnection != null) return sslConnection;

	//Open socket over which SSL will be applied
	String spec = "//" + getHostName() + ":" + getHostPort() + socketOptions;
	com.ibm.oti.connection.socket.Connection connection;
	connection = new com.ibm.oti.connection.socket.Connection(0);
	connection.setParameters(spec, Connector.READ_WRITE, timeout);

	try {
        //Get an SSL connection
        sslConnection = new com.ibm.oti.connection.ssl.Connection(connection, getHostName());
        verifyHostname();
    } catch (IOException e) {
        connection.close();
        throw e;
    }

	return (StreamConnection) sslConnection;

}

protected void verifyHostname() throws IOException {

	String boundHostname = getSubjectCN( sslConnection.getServerCertSubject() );
	if (boundHostname==null) {
		// K0201 = server certificate subject ({0}) does not match connection hostname ({1})
		throw new IOException(com.ibm.oti.util.Msg.getString("K0201", null, null ));
	}

	String host = getHost();
	int idxHostEnd = host.length();
	int idxBoundEnd = boundHostname.length();
	while(true) {
		int idxHostStart = host.lastIndexOf('.', idxHostEnd-1);
		if (idxHostStart==-1)
			idxHostStart = 0;

		int idxBoundStart = boundHostname.lastIndexOf('.', idxBoundEnd-1);
		if (idxBoundStart==-1)
			idxBoundStart = 0;

		String hostSegment = host.substring(idxHostStart, idxHostEnd).toLowerCase();
		String boundSegment = boundHostname.substring(idxBoundStart, idxBoundEnd).toLowerCase();

		if (!Util.matchesPattern(hostSegment, boundSegment)) {
			// the segments do not match
			// K0201 = server certificate subject ({0}) does not match connection hostname ({1})
			throw new IOException(com.ibm.oti.util.Msg.getString("K0201", boundHostname, host));
		}

		if (idxHostStart==0 || idxBoundStart==0) {
			if (idxHostStart != idxBoundStart) {
				// must have same number of segments
				// K0201 = server certificate subject ({0}) does not match connection hostname ({1})
				throw new IOException(com.ibm.oti.util.Msg.getString("K0201", boundHostname, host));
			} else {
				break;
			}
		}

		idxHostEnd = idxHostStart;
		idxBoundEnd = idxBoundStart;
	}
}

protected String getSubjectAltName(X509Certificate cert) {
	try {
		// OID for subjectAltName extension
		byte[] subjectAltNameBytes = cert.getExtensionValue("2.5.29.17");

		if (subjectAltNameBytes==null)
			return null;

		ASN1Decoder decoder = new ASN1Decoder(new ByteArrayInputStream(subjectAltNameBytes));
		ASN1Decoder.Node subjectAltName = decoder.readContents();
		if (subjectAltName.type!=ASN1Decoder.OCTET_STRING)
			return null;

		decoder = new ASN1Decoder(new ByteArrayInputStream((byte[])subjectAltName.data));
		ASN1Decoder.Node generalNamesSequence = decoder.readContents();
		if (generalNamesSequence.type!=ASN1Decoder.SEQUENCE)
			return null;

		ASN1Decoder.Node[] generalNames = (ASN1Decoder.Node[])generalNamesSequence.data;
		for(int i=0; i<generalNames.length; i++) {
			// a tag type of 2 indicates a dNSName
			if (generalNames[i].type==2) {
				// decode again from the raw data to get the type correct
				ByteArrayInputStream bis = new ByteArrayInputStream(
					(byte[])subjectAltName.data, generalNames[i].startPosition, generalNames[i].endPosition);
				decoder = new ASN1Decoder(bis);
				decoder.configureTypeRedirection(0, new ASN1Decoder.TypeMapper() {
					public int map(
						int originalType,
						int nesting,
						int sequenceItem) {
							if (originalType==2) return ASN1Decoder.IA5_STRING;
							return originalType;
					}
				});
				ASN1Decoder.Node dNSNameNode = decoder.readContents();
				if (dNSNameNode.type!=ASN1Decoder.IA5_STRING)
					return null;
				return (String)dNSNameNode.data;
			}
		}
		return null;
	} catch (ASN1Exception e) {
		return null;
	}
}

protected String getSubjectCN(String hostName) {
	X500Principal subject = new X500Principal(hostName);
	return subject.getValueForKey("CN");
}

}
