package com.ibm.oti.connection.serversocket;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2006  All Rights Reserved
 */

import java.io.*;
import javax.microedition.io.*;
import com.ibm.oti.connection.*;
import com.ibm.oti.connection.socket.Socket;
import com.ibm.oti.security.midp.PermissionManager;

/**
 * Implements a server socket connection. Accepts the following parameters:
 * <UL>
 * <LI>so_timeout - set the accept timeout in milliseconds, 0 waits forever,
 * the default is zero. If the connection is opened with the timeout argument set
 * to true, and the so_timeout is not set, the default is 3000 milliseconds.</LI>
 * <LI>backlog - the number of pending connection to accept, the default is 50.</LI>
 * </UL>
 *
 * @author		OTI
 * @version		initial
 *
 * @see		javax.microedition.io.StreamConnectionNotifier
 */
public class Connection extends Socket implements CreateConnection, ServerSocketConnection {
	static final int DEFAULT_TIMEOUT = 8000;
	private boolean closed = false;
	private int timeout = 0, backlog = 50;

public Connection() {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	if (PermissionManager.getManager() != null)
		PermissionManager.getManager().checkPermission("javax.microedition.io.Connector.serversocket");
}

/**
 * Passes the parameters from the Connector.open() method to this
 * object. Protocol used by MIDP 2.0
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		timeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
public javax.microedition.io.Connection setParameters2(String spec, int access, boolean timeout) throws IOException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();

	setParameters(spec, access, timeout);
	return this;
}

/**
 * Passes the parameters from the Connector.open() method to this
 * object. Protocol used by MIDP 1.0
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		timeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
public void setParameters(String spec, int access, boolean timeout) throws IOException {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();

	String[][] equates = ConnectionUtil.NO_PARAMETERS;
	int index = spec.indexOf(';');
	if (index != -1) {
		equates = ConnectionUtil.getParameters(spec.substring(index + 1));
		spec = spec.substring(0, index);
	}
	setParameters(spec, equates, access, timeout);
}

/**
 * Passes the parameters from the Connector.open() method to this
 * object.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		spec String
 *					The address passed to Connector.open()
 * @param		equates String[][]
 *					An 2-dimensional array of strings that
 *					contain the parameters and their keys
 *					passed to Connector.open()
 * @param		access int
 *					The type of access this Connection is
 *					granted (READ, WRITE, READ_WRITE)
 * @param		throwTimeout boolean
 *					A boolean indicating whether or not the
 *					caller to Connector.open() wants timeout
 *					exceptions
 * @exception	IOException
 *					If an error occured opening and configuring
 *					serial port.
 *
 * @see javax.microedition.io.Connector
 */
private void setParameters(String spec, String[][] equates, int access, boolean throwTimeout) throws IOException {
	int[] result = new int[1];
	for (int i=0; i<equates.length; i++) {
		String key = equates[i][0];
		equates[i][0] = equates[i][0].toLowerCase();
		if (ConnectionUtil.intParam("so_timeout", equates[i], ConnectionUtil.NEGATIVE, result))
			timeout = result[0];
		else if (ConnectionUtil.intParam("backlog", equates[i], ConnectionUtil.NEGATIVE, result))
			backlog = result[0];
		// K00a5 = Invalid parameter - {0}
		else throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00a5", key));
	}
	if (throwTimeout && timeout == 0)
		timeout = DEFAULT_TIMEOUT;

	result[0] = 0;
	Socket.parseURL(spec, result, false, true);

	try {
		listenStreamImpl(result[0], backlog);
	} catch (IOException e) {
		socketClose();
		throw e;
	}
	socketOpen();
}

public void close() throws IOException {
	closed = true;
	socketClose();
}

public StreamConnection acceptAndOpen() throws IOException {
	// K00ac = Connection is closed
	if (closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));

	com.ibm.oti.connection.socket.Connection socket = new com.ibm.oti.connection.socket.Connection(Connector.READ_WRITE);
	acceptStreamSocketImpl2(socket, timeout);
	return socket;
}

/**
 * Answer the result of attempting to listen on a stream socket in
 * the IP stack.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		localPort	the local machine port to use
 * @param		backlog	the number of connection requests that may be queued
 *						before requests are rejected
 * @exception	IOException	if an error occurs while listening
 */
native void listenStreamImpl(int localPort, int backlog) throws IOException;

/**
 * Answer the result of attempting to accept a connection request
 * to this stream socket in the IP stack.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		socket	the host socket a connection will be accepted on
 * @param		timeout		the timeout that the server should listen for
 * @exception	IOException	if an error occurs while accepting connections
 */
native void acceptStreamSocketImpl2(Socket socket, int timeout) throws IOException;

/**
 * @see javax.microedition.io.ServerSocketConnection#getLocalAddress()
 */
public String getLocalAddress() throws IOException {
	// K00ac = Connection is closed
	if(closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));

	boolean addressZero = true;
	for( int i=0; i< address.length; i++ ) {
		if (address[i] != 0) {
			addressZero = false;
			break;
		}
	}
	if( addressZero ) {
		String hostName = System.getProperty("microedition.hostname");
		if (hostName != null) {
			if (hostName.length() > 0 && hostName.charAt(0) >= '0' &&
				hostName.charAt(0) <= '9') return hostName;
			try {
				address = Socket.getHostByName(hostName);
			} catch (IOException e) {}
		}
		addressZero = true;
		for( int i=0; i< address.length; i++ ) {
			if (address[i] != 0) {
				addressZero = false;
				break;
			}
		}
		if (addressZero) {
			address = new byte[4];
			address[0]=127; address[1]=0; address[2]=0; address[3]=1;
		}
	}
	return addressToString(address);
}

/**
 * @see javax.microedition.io.ServerSocketConnection#getLocalPort()
 */
public int getLocalPort() throws IOException {
	// K00ac = Connection is closed
	if(closed) throw new IOException(com.ibm.oti.util.Msg.getString("K00ac"));
	return localport;
}

}
