package com.ibm.oti.connection.socket;

/*
 * Licensed Materials - Property of IBM,
 * (c) Copyright IBM Corp. 2000, 2005  All Rights Reserved
 */

import java.io.IOException;
import java.util.Enumeration;
import java.util.Hashtable;

import com.ibm.oti.util.NetworkActivityHook;
import com.ibm.oti.util.NetworkActivityListener;

public abstract class Socket {
	private long descriptor = -1;
	protected int localport;
	protected byte[] address = {0,0,0,0};
	private static Hashtable openSockets = new Hashtable(15);

	public static final int SO_LINGER = 	128;
	public static final int SO_TIMEOUT =	4102;
	public static final int TCP_NODELAY = 	1;
	public static final int SO_REUSEADDR =	4;
	public static final int SO_SNDBUF =		4097;
	public static final int SO_RCVBUF =		4098;
	public static final int SO_KEEPALIVE =	8;

	public static final int FLAG_BROKEN_TCP_NODELAY = 4;
	// signals that when SO_LINGER is enabled and shutdown() is called, a subsequent call to closesocket() will unnecessarily hang
	public static final int FLAG_BROKEN_SO_LINGER_SHUTDOWN = 8;

	public static final int LOOPBACK = 0x7f000001;

	static {
		com.ibm.oti.vm.VM.closeSockets();
	}

/**
 * Close the socket in the IP stack.
 *
 * @author		OTI
 * @version		initial
 */
private native void socketCloseImpl();

/**
 * Set the nominated socket option in the IP stack.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		option			the option selector
 * @param		value				the nominated option value
 * @exception	IOException			thrown if the option is invalid or cannot be set
 */
public native void setSocketOptionImpl(int option, int value) throws IOException;
public native int getSocketOptionImpl(int option) throws IOException;

public static native int getSocketFlags();

/**
 * Query the IP stack for the host address.
 * The host is in string name form.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		name		the host name to lookup
 * @return		int	the host address
 * @exception	IOException	if an error occurs during lookup
 */
public static byte[] getHostByName(String name) throws IOException {
	return getHostByNameImpl(name);
}

private static native byte[] getHostByNameImpl(String name) throws IOException;

public static native int getIPImpl(String host);

public static byte[] resolveHost(String host) throws IOException {
	char ch = 0;
	int index = host.lastIndexOf('.');
	boolean isIPv6 = false;
	if (index != -1 && (index + 1) < host.length()) {
		ch = host.charAt(index+1);
	}
	if (ch >= '0' && ch <= '9') {
		try {
			byte[] addr = new byte[4];
			intToBytes( getIPImpl(host), addr, 0 );
			return addr;
		} catch (IllegalArgumentException e) {
			// K01cd = Invalid host: {0}
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01cd", host));
		}
	} else {
		if (isIPv6){
			// K01cd = Invalid host: {0}
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01cd", host));
		} else {
		if (!isValidHost(host))
			// K01cd = Invalid host: {0}
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K01cd", host));

		try {
 			NetworkActivityHook.getListener().networkActivityStarted(NetworkActivityListener.LOOKUP);
			return getHostByNameImpl(host);
		} catch (IOException e) {
			// Some platforms (i.e. Palm) don't resolve localhost
			if (host.equals("localhost")) {
				byte[] addr = new byte[4];
				intToBytes( LOOPBACK, addr, 0 );
				return addr;
			}

			else throw e;
		}
 			finally {
 				NetworkActivityHook.getListener().networkActivityComplete(NetworkActivityListener.LOOKUP);
 			}
 	}
	}
}

private static boolean isValidHost(String host) {

	// verify that the hostname conforms to RFC# 2396
	char current;
	char lastChar = 0;
	boolean hasName = false;
	boolean beginLabel = true;
	boolean notAlpha = false;
	for(int i = 0; i < host.length(); i++) {
		current = host.charAt(i);
		if (current == '.') {
			if (beginLabel) {
				return false;
			} else {
				// a label cannot end with '-'
				if (lastChar == '-') return false;
				beginLabel = true;
			}
		} else if (current >= '0' && current <= '9') {
			// can't occur at beginning of last label
			if (beginLabel) notAlpha = true;
			beginLabel = false;
		} else if (current == '-') {
			// can't occur at beginning of a label
			if (beginLabel) return false;
			beginLabel = false;
		} else if ( (current >= 'A' && current <= 'Z') || (current >= 'a' && current <= 'z') ) {
			// they can occur anywhere
			hasName = true;
			if (beginLabel) notAlpha = false;
			beginLabel = false;
		} else {
			// no other characters allowed
			return false;
		}
		lastChar = current;
	}
	// a label cannot end with '-'
	// there must be an alpha, i.e. "." is not valid
	// last label must start with an alpha
	if (lastChar == '-' || !hasName || notAlpha) {
		return false;
	}

	return true;
}

/**
 * Query the IP stack for the host address.
 * The host is in address form.
 *
 * @author		OTI
 * @version		initial
 *
 * @param		address		the host address to lookup
 */
public static String getHostByAddr(byte[] address) {
	String host = getHostByAddrImpl(address);
	return host;
}

private static native String getHostByAddrImpl(byte[] address);

public static String parseURL(String spec, int[] returnPort, boolean portRequired, boolean allowNoPort) {
	if (spec.startsWith("//")) spec = spec.substring(2);
	int index = spec.lastIndexOf(':');
	int endOfIPv6Addr = spec.lastIndexOf(']');
	/* if there is no colon, or it is part of the IPv6 address contained in [] */
	if ((index == -1) || (index < endOfIPv6Addr)) {
		if (portRequired)
			// K00a6 = Must specify port
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00a6"));
		return spec;
	}
	if ( (-1 != spec.indexOf('/')) || (-1 != spec.indexOf('@')) || (index != spec.indexOf(':')) ||
			(-1 != spec.indexOf('?')) || (-1 != spec.indexOf(';'))) {
		// K039d = host cannot contain '/', '@', '?', ';', ':'
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K039d"));
	}
	int port;
	String portString = spec.substring(index + 1, spec.length());
	if (allowNoPort && portString.equals("")) port = 0;
	else {
		try {
			port = Integer.parseInt(portString);
		} catch (NumberFormatException e) {
			// K00a7 = Invalid port: {0}
			throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K00a7", portString));
		}
	}
	if (port < 0 || port > 65535)
		// K0325 = Port out of range: {0}
		throw new IllegalArgumentException(com.ibm.oti.util.Msg.getString("K0325", port));
	returnPort[0] = port;
	return spec.substring(0, index);
}

public void socketOpen() {
	openSockets.put(this, this);
}

public void socketClose() {
	NetworkActivityHook.getListener().networkActivityStarted(NetworkActivityListener.CLOSE);
	try {
		socketCloseImpl();
		openSockets.remove(this);
	} finally {
		NetworkActivityHook.getListener().networkActivityComplete(NetworkActivityListener.CLOSE);
	}

}

public static void closeOpenSockets() {
	if (!com.ibm.oti.vm.VM.callerIsBootstrap())
		throw new SecurityException();
	Enumeration keysEnum = openSockets.keys();
	while (keysEnum.hasMoreElements()) {
		Socket socket = (Socket)keysEnum.nextElement();
		socket.socketCloseImpl();
	}
}

public static String addressToString( byte [] value ) {
	return (value[0] & 0xff) + "." + (value[1] & 0xff) + "." + (value[2] & 0xff) + "." + (value[3] & 0xff);
}

public static int socketCount() {
	return openSockets.size();
}

/** Takes the integer and chops it into 4 bytes, putting it
 * into the byte array starting with the high order byte at the
 * index start.  This method makes no checks on the validity of
 * the paramaters.
 */
static void intToBytes( int value, byte bytes[], int start ) {

	// Shift the int so the current byte is right-most
	// Use a byte mask of 255 to single out the last byte.
	bytes[start] = (byte) ((value >> 24) & 255);
	bytes[start + 1] = (byte) ((value >> 16) & 255);
	bytes[start + 2] = (byte) ((value >> 8) & 255);
	bytes[start + 3] = (byte) (value & 255);
}

}
